/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: accueil
 * Version: 1.02.03
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.dao.impl;

import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.univ.objetspartages.bean.FicheAccueilBean;
import com.univ.objetspartages.dao.AbstractFicheDAO;
import com.univ.objetspartages.om.FicheAccueil;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;
import org.apache.commons.lang3.StringUtils;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

public class FicheAccueilDAO extends AbstractFicheDAO<FicheAccueilBean> {

    public FicheAccueilDAO() {
        this.tableName = "FICHEACCUEIL";
    }

    @Override
    public FicheAccueilBean add(final FicheAccueilBean fiche) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into FICHEACCUEIL (TITRE, CODE, META_KEYWORDS, META_DESCRIPTION, CODE_RUBRIQUE, CODE_RATTACHEMENT, TITRE_ENCADRE, CONTENU_ENCADRE, DATE_ALERTE, MESSAGE_ALERTE, DATE_CREATION, DATE_PROPOSITION, DATE_VALIDATION, DATE_MODIFICATION, CODE_REDACTEUR, CODE_VALIDATION, DATE_ARCHIVAGE, LANGUE, ETAT_OBJET, NB_HITS, DIFFUSION_PUBLIC_VISE, DIFFUSION_MODE_RESTRICTION, DIFFUSION_PUBLIC_VISE_RESTRICTION) values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)", PreparedStatement.RETURN_GENERATED_KEYS)) {
            stmt.setObject(1, fiche.getTitle(), Types.LONGVARCHAR);
            stmt.setObject(2, fiche.getCode(), Types.VARCHAR);
            stmt.setObject(3, fiche.getMetaKeywords(), Types.LONGVARCHAR);
            stmt.setObject(4, fiche.getMetaDescription(), Types.LONGVARCHAR);
            stmt.setObject(5, fiche.getCodeRubrique(), Types.VARCHAR);
            stmt.setObject(6, fiche.getCodeRattachement(), Types.VARCHAR);
            stmt.setObject(7, fiche.getTitreEncadre(), Types.VARCHAR);
            stmt.setObject(8, fiche.getContenuEncadre(), Types.VARCHAR);
            stmt.setObject(9, fiche.getDateAlerte(), Types.DATE);
            stmt.setObject(10, fiche.getMessageAlerte(), Types.LONGVARCHAR);
            stmt.setObject(11, fiche.getDateCreation(), Types.TIMESTAMP);
            stmt.setObject(12, fiche.getDateProposition(), Types.TIMESTAMP);
            stmt.setObject(13, fiche.getDateValidation(), Types.TIMESTAMP);
            stmt.setObject(14, fiche.getDateModification(), Types.TIMESTAMP);
            stmt.setObject(15, fiche.getCodeRedacteur(), Types.VARCHAR);
            stmt.setObject(16, fiche.getCodeValidation(), Types.VARCHAR);
            stmt.setObject(17, fiche.getDateArchivage(), Types.DATE);
            stmt.setObject(18, fiche.getLangue(), Types.VARCHAR);
            stmt.setObject(19, fiche.getEtatObjet(), Types.VARCHAR);
            stmt.setObject(20, fiche.getNbHits(), Types.BIGINT);
            stmt.setObject(21, fiche.getDiffusionPublicVise(), Types.VARCHAR);
            stmt.setObject(22, fiche.getDiffusionModeRestriction(), Types.VARCHAR);
            stmt.setObject(23, fiche.getDiffusionPublicViseRestriction(), Types.VARCHAR);
            int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", fiche.toString(), tableName, rowsAffected));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            fiche.setId(rs.getLong(1));
        } catch (SQLException e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", fiche.toString(), tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occured closing Resultset during add() on table %s", tableName), e);
                }
            }
        }
        return fiche;
    }

    @Override
    public FicheAccueilBean update(final FicheAccueilBean fiche) throws UpdateToDataSourceException {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update FICHEACCUEIL set TITRE = ?, CODE = ?, META_KEYWORDS = ?, META_DESCRIPTION = ?, CODE_RUBRIQUE = ?, CODE_RATTACHEMENT = ?, TITRE_ENCADRE = ?, CONTENU_ENCADRE = ?, DATE_ALERTE = ?, MESSAGE_ALERTE = ?, DATE_CREATION = ?, DATE_PROPOSITION = ?, DATE_VALIDATION = ?, DATE_MODIFICATION = ?, CODE_REDACTEUR = ?, CODE_VALIDATION = ?, DATE_ARCHIVAGE = ?, LANGUE = ?, ETAT_OBJET = ?, NB_HITS = ?, DIFFUSION_PUBLIC_VISE = ?, DIFFUSION_MODE_RESTRICTION = ?, DIFFUSION_PUBLIC_VISE_RESTRICTION = ? WHERE ID_FICHEACCUEIL = ?")) {
            stmt.setObject(1, fiche.getTitle(), Types.LONGVARCHAR);
            stmt.setObject(2, fiche.getCode(), Types.VARCHAR);
            stmt.setObject(3, fiche.getMetaKeywords(), Types.LONGVARCHAR);
            stmt.setObject(4, fiche.getMetaDescription(), Types.LONGVARCHAR);
            stmt.setObject(5, fiche.getCodeRubrique(), Types.VARCHAR);
            stmt.setObject(6, fiche.getCodeRattachement(), Types.VARCHAR);
            stmt.setObject(7, fiche.getTitreEncadre(), Types.VARCHAR);
            stmt.setObject(8, fiche.getContenuEncadre(), Types.VARCHAR);
            stmt.setObject(9, fiche.getDateAlerte(), Types.DATE);
            stmt.setObject(10, fiche.getMessageAlerte(), Types.LONGVARCHAR);
            stmt.setObject(11, fiche.getDateCreation(), Types.TIMESTAMP);
            stmt.setObject(12, fiche.getDateProposition(), Types.TIMESTAMP);
            stmt.setObject(13, fiche.getDateValidation(), Types.TIMESTAMP);
            stmt.setObject(14, fiche.getDateModification(), Types.TIMESTAMP);
            stmt.setObject(15, fiche.getCodeRedacteur(), Types.VARCHAR);
            stmt.setObject(16, fiche.getCodeValidation(), Types.VARCHAR);
            stmt.setObject(17, fiche.getDateArchivage(), Types.DATE);
            stmt.setObject(18, fiche.getLangue(), Types.VARCHAR);
            stmt.setObject(19, fiche.getEtatObjet(), Types.VARCHAR);
            stmt.setObject(20, fiche.getNbHits(), Types.BIGINT);
            stmt.setObject(21, fiche.getDiffusionPublicVise(), Types.VARCHAR);
            stmt.setObject(22, fiche.getDiffusionModeRestriction(), Types.VARCHAR);
            stmt.setObject(23, fiche.getDiffusionPublicViseRestriction(), Types.VARCHAR);
            stmt.setObject(24, fiche.getId(), Types.BIGINT);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", fiche.toString(), tableName), e);
        }
        return fiche;
    }

    protected FicheAccueilBean fill(ResultSet rs) throws DataSourceException {
        FicheAccueilBean ficheAccueil = new FicheAccueilBean();
        try {
            ficheAccueil.setId(rs.getLong("ID_FICHEACCUEIL"));
            ficheAccueil.setTitle(rs.getString("TITRE"));
            ficheAccueil.setCode(rs.getString("CODE"));
            ficheAccueil.setCodeRubrique(rs.getString("CODE_RUBRIQUE"));
            ficheAccueil.setCodeRattachement(rs.getString("CODE_RATTACHEMENT"));
            ficheAccueil.setTitreEncadre(rs.getString("TITRE_ENCADRE"));
            ficheAccueil.setContenuEncadre(rs.getString("CONTENU_ENCADRE"));
            ficheAccueil.setMetaKeywords(rs.getString("META_KEYWORDS"));
            ficheAccueil.setMetaDescription(rs.getString("META_DESCRIPTION"));
            ficheAccueil.setTitreEncadre(rs.getString("TITRE_ENCADRE"));
            ficheAccueil.setContenuEncadre(rs.getString("CONTENU_ENCADRE"));
            ficheAccueil.setEncadreRecherche(rs.getString("ENCADRE_RECHERCHE"));
            ficheAccueil.setEncadreRechercheBis(rs.getString("ENCADRE_RECHERCHE_BIS"));
            ficheAccueil.setDateAlerte(rs.getDate("DATE_ALERTE"));
            ficheAccueil.setMessageAlerte(rs.getString("MESSAGE_ALERTE"));
            ficheAccueil.setDateCreation(new Date(rs.getTimestamp("DATE_CREATION").getTime()));
            ficheAccueil.setDateProposition(new Date(rs.getTimestamp("DATE_PROPOSITION").getTime()));
            ficheAccueil.setDateValidation(new Date(rs.getTimestamp("DATE_VALIDATION").getTime()));
            ficheAccueil.setDateModification(new Date(rs.getTimestamp("DATE_MODIFICATION").getTime()));
            ficheAccueil.setCodeRedacteur(rs.getString("CODE_REDACTEUR"));
            ficheAccueil.setCodeValidation(rs.getString("CODE_VALIDATION"));
            ficheAccueil.setLangue(rs.getString("LANGUE"));
            ficheAccueil.setEtatObjet(rs.getString("ETAT_OBJET"));
            ficheAccueil.setNbHits(rs.getLong("NB_HITS"));
            ficheAccueil.setDiffusionModeRestriction(rs.getString("DIFFUSION_MODE_RESTRICTION"));
            ficheAccueil.setDiffusionPublicVise(rs.getString("DIFFUSION_PUBLIC_VISE"));
            ficheAccueil.setDiffusionPublicViseRestriction(rs.getString("DIFFUSION_PUBLIC_VISE_RESTRICTION"));
        } catch (SQLException e) {
            throw new DataSourceException("An error occured trying to map resultset to object", e);
        }
        return ficheAccueil;
    }

    public List<FicheAccueilBean> select(FicheAccueil ficheAccueil, String code, String titre, String codeRattachement, String codeRubrique, String langue, String codeRedacteur,
                                         String etatObjet, String nombre, String ids) throws Exception {
        ClauseWhere where = new ClauseWhere();
        if (StringUtils.isNotEmpty(titre)) {
            where.and(ConditionHelper.like("TITRE", titre, "%", "%"));
        }
        if (StringUtils.isNotEmpty(codeRattachement)) {
            where.and(ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", codeRattachement));
        }
        if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
            where.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
        }
        if (StringUtils.isNotEmpty(etatObjet) && !"0000".equals(etatObjet)) {
            where.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", etatObjet));
        }
        if (StringUtils.isNotEmpty(codeRedacteur)) {
            where.and(ConditionHelper.egalVarchar("CODE_REDACTEUR", codeRedacteur));
        }
        if (StringUtils.isNotEmpty(ids) && !"0000".equals(ids)) {
            where.and(ConditionHelper.in("ID_FICHEACCUEIL", Chaine.getVecteurPointsVirgules(ids)));
        }
        if (StringUtils.isNotEmpty(code)) {
            where.and(ConditionHelper.egalVarchar("T1.CODE", code));
        }
        try (final ContexteDao ctx = new ContexteDao()) {
            final RequeteSQL requeteSelect = RequeteSQLHelper.getRequeteGenerique(where, ctx, ficheAccueil, codeRubrique);
            ClauseOrderBy orderBy = new ClauseOrderBy();
            orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
            requeteSelect.orderBy(orderBy);
            ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
            requeteSelect.limit(limite);
            return select(requeteSelect.formaterRequete());
        }
    }
}
