/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: actualite
 * Version: 1.02.07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.Date;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.cms.objetspartages.annotation.FicheAnnotation;
import com.univ.mediatheque.UsingLibelleMedia;
import com.univ.objetspartages.sgbd.ActualiteDB;
import com.univ.objetspartages.util.ActualiteHelper;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;
import com.univ.utils.DateUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RequeteUtil;
import com.univ.utils.UnivWebFmt;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;
import com.univ.xml.NodeUtil;

/**
 * Classe representant un objet actualite.
 */
@FicheAnnotation()
public class Actualite extends ActualiteDB implements FicheUniv, DiffusionSelective, FicheExportableXML, UsingLibelleMedia, FicheArchivable {

	/**
	 *
	 */
	private static final long serialVersionUID = -270335972303607245L;

	/**
	 * Renvoie le libelle a afficher (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param codes
	 *            the codes
	 * @param langue
	 *            the langue
	 *
	 * @return the libelle affichable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleAffichable(final OMContext ctx, final String codes, String langue) throws Exception {
		String res = "";
		if (codes == null || codes.length() == 0) {
			return res;
		}
		final Actualite actualite = new Actualite();
		actualite.init();
		actualite.setCtx(ctx);
		if (langue == null || langue.length() == 0) {
			langue = "0";
		}
		final StringTokenizer st = new StringTokenizer(codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			// On cherche d'abord la version en ligne puis les autres versions
			int count = actualite.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) {
				count = actualite.selectCodeLangueEtat(code, langue, "");
			}
			if (count > 0) {
				if (res.length() > 1) {
					res += ";";
				}
				if (actualite.nextItem()) {
					res += actualite.getLibelleAffichable();
				}
			} else {
				res += "-";
			}
		}
		return res;
	}

	/**
	 * Renvoie la fiche demandee (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 *
	 * @return the fiche actualite
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Actualite getFicheActualite(final OMContext ctx, final String code, String langue) throws Exception {
		Actualite fiche = null;
		if (code != null && code.length() > 0) {
			if (langue == null || langue.length() == 0) {
				langue = "0"; // francais par defaut
			}
			fiche = new Actualite();
			fiche.init();
			fiche.setCtx(ctx);
			int count = fiche.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) // si pas de version en ligne
			{
				count = fiche.selectCodeLangueEtat(code, langue, "");
			}
			if (count == 0) // si pas de version dans la langue demandee
			{
				count = fiche.selectCodeLangueEtat(code, "", "0003");
			}
			if (count > 0) {
				fiche.nextItem();
			}
		}
		return fiche;
	}

	/**
	 * Initialise l'objet metier.
	 */
	@Override
	public void init() {
		FicheUnivMgr.init(this);
		setIdActualite((long) 0);
		setCode("");
		setLangue("0");
		setTitre("");
		setResume("");
		setDateEvenement(new Date(0));
		setLieu("");
		setIdPlanAcces((long) 0);
		setTypeEvenement("");
		setThematique("");
		setDescription("");
		setResponsableEvenement("");
		setEmailResponsableEvenement("");
		setRattachementBandeau("0");
		setDateSuppression(new Date(0));
		setComplementDate("");
		setSiteEvenement("");
		setDateFinEvenement(new Date(0));
		setDateArchivage(new Date(0));
		setIdVignette((long) 0);
		setDateDeclenchement(new Date(0));
		setPartenaires("");
		setInsertionAgenda("");
	}

	/**
	 * Renvoie le libelle a afficher.
	 *
	 * @return the libelle affichable
	 */
	@Override
	public String getLibelleAffichable() {
		return getTitre();
	}

	/**
	 * Récupération du libellé à afficher (spécifique au back-office).
	 *
	 * @return the libelle back office
	 *
	 */
	public String getLibelleBackOffice() {
		return getLibelleAffichable() + " [" + Rubrique.getIntitule(getCodeRubrique()) + "]";
	}

	/**
	 * Renvoie l'identifiant de la fiche.
	 *
	 * @return the id fiche
	 */
	@Override
	public Long getIdFiche() {
		return getIdActualite();
	}

	/**
	 * Valorise l'identifiant de la fiche.
	 *
	 * @param idFiche
	 *            the id fiche
	 */
	@Override
	public void setIdFiche(final Long idFiche) {
		setIdActualite(idFiche);
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut lieu.
	 *
	 * @return the formated lieu
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedLieu() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getLieu());
	}

	/**
	 * Renvoie le libelle pour l'attribut typeEvenement.
	 *
	 * @return the libelle type evenement
	 *
	 */
	public String getLibelleTypeEvenement() {
		return Libelle.getLibelle("03", getTypeEvenement(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le libelle pour l'attribut thematique.
	 *
	 * @return the libelle thematique
	 *
	 */
	public String getLibelleThematique() {
		return Libelle.getLibelle("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie les thematiques sous forme de liste pour qu'elles soient affichees sous forme de liste en front.
	 */
	public List<String> getListLibelleThematique() {
		return Libelle.getLibelleSousFormeDeListe("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut description.
	 *
	 * @return the formated description
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedDescription() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getDescription());
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut complementDate.
	 *
	 * @return the formated complement date
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedComplementDate() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getComplementDate());
	}

	/**
	 * Renvoie le libelle pour l'attribut siteEvenement.
	 *
	 * @return the libelle site evenement
	 *
	 */
	public String getLibelleSiteEvenement() {
		return Libelle.getLibelle("08", getSiteEvenement(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie la liste des fichiers joints de la fiche.
	 *
	 * @return the liste fichiers
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<Ressource> getListeFichiers() throws Exception {
		return Ressource.getListeFichier(this);
	}

	/**
	 * Renvoie la liste des fichiers indicés joints de la fiche.
	 *
	 * @return the liste fichiers indices
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<Ressource> getListeFichiersIndices() throws Exception {
		return Ressource.getListeFichierIndice(this);
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut partenaires.
	 *
	 * @return the formated partenaires
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedPartenaires() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getPartenaires());
	}

	/**
	 * Renvoie la rubrique de la fiche.
	 *
	 * @return the infos rubrique
	 *
	 */
	public InfosRubriques getInfosRubrique() {
		return Rubrique.renvoyerItemRubrique(getCodeRubrique());
	}

	/**
	 * Renvoie la structure de la fiche.
	 *
	 * @return the infos structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public InfosStructure getInfosStructure() throws Exception {
		return Structure.renvoyerItemStructure(getCodeRattachement(), getLangue());
	}

	/**
	 * Renvoie les metadonnees de la fiche.
	 *
	 * @return the meta
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Metatag getMeta() throws Exception {
		return FicheUnivMgr.lireMeta(this);
	}

	/**
	 * Renvoie une chaine correspondant a la concatenation des champs texte de l'objet (utilise pour l'indexation Lucene).
	 *
	 * @return the full text string
	 *
	 */
	@Override
	public String getFullTextString() {
		return getResume() + " " + getLieu() + " " + StringUtils.replace(getLibelleTypeEvenement(), ";", " ") + " " + StringUtils.replace(
			getLibelleThematique(), ";", " ") + " " + getDescription() + " " + getResponsableEvenement() + " " + getEmailResponsableEvenement() + " " + getComplementDate() + " " + StringUtils.replace(
				getLibelleSiteEvenement(), ";", " ") + " " + getPartenaires();
	}

	/**
	 * Duplication dans une nouvelle langue, le nouvel objet devient l'objet courant.
	 *
	 * @param nouvelleLangue
	 *            the nouvelle langue
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public void dupliquer(final String nouvelleLangue) throws Exception {
		// donnees a reinitialiser dans le nouvel objet
		FicheUnivMgr.dupliquer(this);
		// creation objet dans la langue demandee
		setIdFiche(new Long(0));
		setLangue(nouvelleLangue);
		add();
	}

	/**
	 * Traitement d'une requete sur l'objet.
	 *
	 * @param requete
	 *            the requete
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int traiterRequete(final String requete) throws Exception {
		// Recuperation des parametres de la requete
		final String code = RequeteUtil.renvoyerParametre(requete, "CODE");
		final String titre = RequeteUtil.renvoyerParametre(requete, "TITRE");
		final String typeEvenement = RequeteUtil.renvoyerParametre(requete, "TYPE_EVENEMENT");
		final String thematique = RequeteUtil.renvoyerParametre(requete, "THEMATIQUE");
		final String siteEvenement = RequeteUtil.renvoyerParametre(requete, "SITE");
		final String mois = RequeteUtil.renvoyerParametre(requete, "MOIS");
		final String sDateDebut = RequeteUtil.renvoyerParametre(requete, "DATE_EVENEMENT_DEBUT");
		final String sDateFin = RequeteUtil.renvoyerParametre(requete, "DATE_EVENEMENT_FIN");
		final String nbJours = RequeteUtil.renvoyerParametre(requete, "JOUR");
		final String selection = RequeteUtil.renvoyerParametre(requete, "SELECTION");
		final String tri = RequeteUtil.renvoyerParametre(requete, "TRI_DATE");
		final String triDesc = RequeteUtil.renvoyerParametre(requete, "TRI_DATE_DESC");
		final String ids = RequeteUtil.renvoyerParametre(requete, "IDS");
		final String nombre = RequeteUtil.renvoyerParametre(requete, "NOMBRE");
		final String codeRubrique = RequeteUtil.renvoyerParametre(requete, "CODE_RUBRIQUE");
		final String codeRattachement = RequeteUtil.renvoyerParametre(requete, "CODE_RATTACHEMENT");
		final String langue = RequeteUtil.renvoyerParametre(requete, "LANGUE");
		final String etatObjet = StringUtils.defaultIfBlank(RequeteUtil.renvoyerParametre(requete, "ETAT_OBJET"), EtatFiche.EN_LIGNE.toString());
		final String codeRedacteur = RequeteUtil.renvoyerParametre(requete, "CODE_REDACTEUR");
		Date dateDebut = null;
		Date dateFin = null;
		String order = "T1.DATE_EVENEMENT";
		String orderTri = "";
		if (selection.length() > 0) {
			final Date today = new Date(System.currentTimeMillis());
			// Agenda (évènements à venir)
			if (selection.equals("0001")) {
				dateDebut = today; // aujourd'hui
			} else if (selection.equals("0003")) {
				dateFin = today; // aujourd'hui
				orderTri = "DESC";
			}
			// Actualités du jour
			else if (selection.equals("0004")) {
				// date début dans la journée ou date de fin dans la journée
				dateDebut = today; // aujourd'hui
				dateFin = today; // aujourd'hui
			}
			// Actualités de la semaine
			else if (selection.equals("0005")) {
				// date début dans la semaine ou date de fin dans la semaine
				dateDebut = DateUtil.getFirstDayOfWeek(today);
				dateFin = DateUtil.getLastDayOfWeek(today);
			}
			// Actualités du mois
			else if (selection.equals("0006")) {
				// date début dans le mois ou date de fin dans le mois
				dateDebut = DateUtil.getFirstDayOfMonth(today);
				dateFin = DateUtil.getLastDayOfMonth(today);
			}
			// Dernières actualités publiées
			else if (selection.equals("0007")) {
				order = "META.META_DATE_MISE_EN_LIGNE";
				orderTri = "DESC";
			}
		} else if (mois.length() > 0 && !mois.equals("0000")) {
			final GregorianCalendar cal = new GregorianCalendar();
			cal.setTimeInMillis(System.currentTimeMillis());
			cal.set(Calendar.MONTH, Integer.parseInt(mois) - 1);
			dateDebut = DateUtil.getFirstDayOfMonth(new Date(cal.getTimeInMillis()));
			dateFin = DateUtil.getLastDayOfMonth(new Date(cal.getTimeInMillis()));
		} else if (nbJours.length() > 0) {
			dateDebut = DateUtil.addDays(new Date(System.currentTimeMillis()), 1);
			dateFin = DateUtil.addDays(new Date(System.currentTimeMillis()), Integer.parseInt(nbJours)); // aujourd'hui + nbJours
		} else {
			dateDebut = DateUtil.parseDate(sDateDebut);
			dateFin = DateUtil.parseDate(sDateFin);
		}
		if ("0".equals(tri) || "DATE_DESC".equals(tri) || "1".equals(triDesc)) {
			orderTri = "DESC";
		}
		if ("1".equals(tri) || "DATE_ASC".equals(tri)) {
			orderTri = "ASC";
		}
		return select(code, titre, typeEvenement, thematique, dateDebut, dateFin, codeRattachement, codeRubrique, langue, codeRedacteur, etatObjet, siteEvenement, nombre,
			order + " " + orderTri, ids);
	}

	/**
	 * Selection d'une fiche a partir de son code, de sa langue et de son etat.
	 *
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 * @param etatObjet
	 *            the etat objet
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int selectCodeLangueEtat(final String code, final String langue, final String etatObjet) throws Exception {
		final RequeteSQL codeLangueEtatOrdreSurTitre = new RequeteSQL();
		final ClauseWhere whereCodeLangueEtat = ConditionHelper.whereCodeLangueEtat(code, langue, etatObjet);
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		codeLangueEtatOrdreSurTitre.where(whereCodeLangueEtat).orderBy(orderBy);
		return select(codeLangueEtatOrdreSurTitre.formaterRequete());
	}

	/**
	 * Renvoie les actualites pour la rubrique demandee.
	 *
	 * @param codeRubrique
	 *            the code rubrique
	 * @param langue
	 *            the langue
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
    @Deprecated
	public int selectParCodeRubrique(final String codeRubrique, final String langue) throws Exception {
		final RequeteSQL codeRubriqueLangueEnLigneOrdreTitre = new RequeteSQL();
		final ClauseWhere whereCodeRubriqueLangueEtat = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeRubrique)) {
			whereCodeRubriqueLangueEtat.setPremiereCondition(ConditionHelper.egalVarchar("T1.CODE_RUBRIQUE", codeRubrique));
		}
		if (StringUtils.isNotEmpty(langue)) {
			whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", "0003"));
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		codeRubriqueLangueEnLigneOrdreTitre.where(whereCodeRubriqueLangueEtat).orderBy(orderBy);
		return select(codeRubriqueLangueEnLigneOrdreTitre.formaterRequete());
	}

	/**
	 * Renvoie la liste des references a cette fiche.
	 *
	 * @return the references
	 */
	@Override
	public String getReferences() {
		String res = FicheUnivMgr.getReferenceParTexte(getLieu()) + FicheUnivMgr.getReferenceParPhoto(getIdPlanAcces()) + FicheUnivMgr.getReferenceParTexte(getDescription()) + FicheUnivMgr.getReferenceParTexte(getComplementDate()) + FicheUnivMgr.getReferenceParPhoto(getIdVignette()) + FicheUnivMgr.getReferenceParTexte(getPartenaires()) + FicheUnivMgr.getReferenceParJointure(
			"structure", getCodeRattachement());
		try {
			res += FicheUnivMgr.getReferenceParListePhoto(getListeFichiers()) + FicheUnivMgr.getReferenceParListePhoto(getListeFichiersIndices());
		} catch (final Exception ignored) {}
		return res;
	}

	/**
	 * Selection d'une instance de l'objet Actualite a partir de l'ensemble des criteres.
	 *
	 * @param code
	 *            the code
	 * @param titre
	 *            the titre
	 * @param typeEvenement
	 *            the type evenement
	 * @param thematique
	 *            the thematique
	 * @param dateEvenementDebut
	 *            the date evenement debut
	 * @param dateEvenementFin
	 *            the date evenement fin
	 * @param codeRattachement
	 *            the code rattachement
	 * @param codeRubrique
	 *            the code rubrique
	 * @param langue
	 *            the langue
	 * @param codeRedacteur
	 *            the code redacteur
	 * @param etatObjet
	 *            the etat objet
	 * @param siteEvenement
	 *            the site evenement
	 * @param nombre
	 *            the nombre
	 * @param order
	 *            the order
	 * @param ids
	 *            the ids
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String code, final String titre, final String typeEvenement, final String thematique, final Date dateEvenementDebut, final Date dateEvenementFin,
		final String codeRattachement, final String codeRubrique, final String langue, final String codeRedacteur, final String etatObjet, final String siteEvenement,
		final String nombre, final String order, final String ids) throws Exception {
		RequeteSQL requeteSelect;
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(titre)) {
			where.setPremiereCondition(ConditionHelper.rechercheMots("TITRE", titre));
		}
		if (StringUtils.isNotEmpty(typeEvenement) && !"0000".equals(typeEvenement)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.TYPE_EVENEMENT", typeEvenement));
		}
		if (StringUtils.isNotEmpty(thematique) && !"0000".equals(thematique)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.THEMATIQUE", thematique));
		}
		if (Formateur.estSaisie(dateEvenementDebut)) {
			where.and(ConditionHelper.critereDateDebut("DATE_FIN_EVENEMENT", dateEvenementDebut));
		}
		if (Formateur.estSaisie(dateEvenementFin)) {
			where.and(ConditionHelper.critereDateFin("DATE_EVENEMENT", dateEvenementFin));
		}
		if (StringUtils.isNotEmpty(siteEvenement) && !"0000".equals(siteEvenement)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.SITE_EVENEMENT", siteEvenement));
		}
		if (StringUtils.isNotEmpty(ids) && !"0000".equals(ids)) {
			where.and(ConditionHelper.in("ID_ACTUALITE", Chaine.getVecteurPointsVirgules(ids)));
		}
		if (StringUtils.isNotEmpty(codeRattachement)) {
			where.and(ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", codeRattachement));
		}
		if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(etatObjet) && !"0000".equals(etatObjet)) {
			where.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", etatObjet));
		}
		if (StringUtils.isNotEmpty(codeRedacteur)) {
			where.and(ConditionHelper.egalVarchar("CODE_REDACTEUR", codeRedacteur));
		}
		if (StringUtils.isNotEmpty(code)) {
			where.and(ConditionHelper.egalVarchar("T1.CODE", code));
		}
		requeteSelect = RequeteSQLHelper.getRequeteGenerique(where, ctx, this, codeRubrique);
		ClauseOrderBy orderBy = new ClauseOrderBy();
		if (StringUtils.isNotBlank(order)) {
			orderBy = OrderByHelper.reconstruireClauseOrderBy(order);
		} else {
			orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		}
		requeteSelect.orderBy(orderBy);
		final ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
		requeteSelect.limit(limite);
		return select(requeteSelect.formaterRequete());
	}

	/**
	 * Renvoie la description de la fiche (utilise dans SaisieCommentaire).
	 *
	 * @return the contenu
	 *
	 * @see Actualite#getDescription() 
	 */
	public String getContenu() {
		return getDescription();
	}

	/**
	 * Renvoie la liste des types d'evenements sous forme de vecteur.
	 *
	 * @return the vecteur types evenement
	 */
	public Vector<String> getVecteurTypesEvenement() {
		final Vector<String> v = new Vector<>();
		final StringTokenizer st = new StringTokenizer(getTypeEvenement(), ";");
		while (st.hasMoreTokens()) {
			v.add(st.nextToken());
		}
		return v;
	}

	/**
	 * Valorise la liste des types d'evenement sous forme de vecteur.
	 *
	 * @param v
	 *            the v
	 */
	public void setVecteurTypesEvenements(final Vector<String> v) {
		String liste = "";
		final Enumeration<String> en = v.elements();
		while (en.hasMoreElements()) {
			liste += en.nextElement() + ";";
		}
		setTypeEvenement(liste);
	}

	/**
	 * Exporte cette actualite.
	 *
	 * @param liste
	 *            the liste
	 * @param formatExport
	 *            the format export
	 *
	 * @return the node
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public Node exportNodeFiche(final Node liste, final String formatExport) throws Exception {
		final Document document = liste.getOwnerDocument();
		final Node fiche = document.createElement("ACTUALITE");
		NodeUtil.addNode(fiche, "CODE", getCode());
		NodeUtil.addNode(fiche, "LANGUE", getLangue());
		NodeUtil.addNode(fiche, "TITRE", getTitre());
		NodeUtil.addNode(fiche, "RESUME", getResume());
		NodeUtil.addNode(fiche, "DATE_EVENEMENT", Formateur.formaterDate(getDateEvenement(), ""));
		NodeUtil.addNode(fiche, "DATE_EVENEMENT_FIN", Formateur.formaterDate(getDateFinEvenement(), ""));
		NodeUtil.addNode(fiche, "RATTACHEMENT", Structure.getLibelleRattachement(ctx, getCodeRattachement(), getLangue(), true));
		NodeUtil.addNode(fiche, "CODE_RUBRIQUE", getCodeRubrique());
		NodeUtil.addNode(fiche, "RUBRIQUE", Rubrique.renvoyerItemRubrique(getCodeRubrique()).getIntitule());
		NodeUtil.addNode(fiche, "LIEU", getLieu());
		NodeUtil.addNode(fiche, "PARTENAIRES", getPartenaires());
		NodeUtil.addNode(fiche, "RESPONSABLE", getResponsableEvenement());
		NodeUtil.addNode(fiche, "EMAIL_RESPONSABLE", getEmailResponsableEvenement());
		NodeUtil.addNode(fiche, "DESCRIPTION", getDescription());
		NodeUtil.addNode(fiche, "COMPLEMENT_DATE", getComplementDate());
		NodeUtil.addNode(fiche, "SITE_EVENEMENT", Libelle.getLibelle("08", getSiteEvenement(), ctx.getLocale()));
		NodeUtil.addNode(fiche, "TYPE_EVENEMENT", Libelle.getLibelle("03", getTypeEvenement(), ctx.getLocale()));
		NodeUtil.addNode(fiche, "THEMATIQUE", Libelle.getLibelle("04", getThematique(), ctx.getLocale()));
		NodeUtil.addNode(fiche, "URL_VIGNETTE", Media.getMedia(getIdVignette()).getUrlAbsolue());
		return fiche;
	}

	@Override
	public List<String> getCodesLibelleMedia() {
		// 0003, le plan d'acces est un type photo de libelle utilise dans la mediatheque
		return Arrays.asList("0003");
	}

	@Override
	public void checkCodesLibelleMedia(final OMContext ctx, final long idMedia) throws Exception {
		// controle plan d'acces actualite
		final Actualite actualite = new Actualite();
		actualite.setCtx(ctx);
		actualite.init();
		actualite.select("WHERE ID_PLAN_ACCES=" + idMedia);
		if (actualite.nextItem()) {
			throw new ErreurApplicative(String.format(ActualiteHelper.getMessage("ACTUALITE.ERREUR.RESSOURCE_NON_SUPPRIMABLE"), actualite.getLibelleAffichable()));
		}
	}
}
