/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: actualite
 * Version: 1.02.07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.sql.Date;
import java.util.Vector;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.om.Actualite;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.om.Media;

/**
 * Processus de gestion des actualites.
 */
public class SaisieActualite extends SaisieFiche {

	/** The actualite. */
	private Actualite actualite = null;

	/**
	 * Constructeur.
	 *
	 * @param infoBean
	 *            the info bean
	 */
	public SaisieActualite(final InfoBean infoBean) {
		super(infoBean);
	}

	/**
	 * Point d'entree du processus.
	 *
	 * @return true, if traiter action
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		infoBean.set("CODE_OBJET", "0008");
		actualite = new Actualite();
		actualite.setCtx(this);
		actualite.init();
		final AutorisationBean autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
		if (autorisations != null) {
			this.getDatas().put("AUTORISATIONS", autorisations);
		}
		traiterActionParDefaut(actualite);
		infoBean.set("NOM_ONGLET", "actualite");
		// on continue si on n'est pas a la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Affichage de l'ecran des criteres de recherche d'une actualite.
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void preparerRECHERCHE() throws Exception {
		ecranLogique = ECRAN_RECHERCHE;
		infoBean.set("LISTE_LANGUES", LangueUtil.getListeLangues(getLocale()));
		infoBean.set("LISTE_TYPE_EVENEMENT", Libelle.getListe("03", getLocale()));
		infoBean.set("LISTE_THEMATIQUES", Libelle.getListe("04", getLocale()));
		infoBean.set("LISTE_SITES", Libelle.getListe("08", getLocale()));
	}

	/**
	 * Affichage de l'ecran de saisie d'une actualite.
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void preparerPRINCIPAL() throws Exception {
		ecranLogique = ECRAN_PRINCIPAL;
		infoBean.set("ID_ACTUALITE", actualite.getIdActualite().toString());
		infoBean.set("TITRE", actualite.getTitre());
		infoBean.set("DATE_EVENEMENT", actualite.getDateEvenement());
		infoBean.set("DATE_FIN_EVENEMENT", actualite.getDateFinEvenement());
		infoBean.set("CORPS", actualite.getDescription());
		infoBean.set("THEMATIQUE", actualite.getThematique());
		infoBean.set("LIBELLE_THEMATIQUE", actualite.getLibelleThematique());
		infoBean.set("LISTE_THEMATIQUES", Libelle.getListe("04", LangueUtil.getLocale(actualite.getLangue())));
		infoBean.set("SITE_EVENEMENT", actualite.getSiteEvenement());
		infoBean.set("LIBELLE_SITE_EVENEMENT", actualite.getLibelleSiteEvenement());
		infoBean.set("LISTE_SITES_EVENEMENT", Libelle.getListe("08", LangueUtil.getLocale(actualite.getLangue())));
		infoBean.set("COMPLEMENT_DATE", actualite.getComplementDate());
		infoBean.set("RESUME", actualite.getResume());
		infoBean.set("PARTENAIRES", actualite.getPartenaires());
		infoBean.set("RATTACHEMENT_BANDEAU", actualite.getRattachementBandeau());
		infoBean.set("LIEU", actualite.getLieu());
		infoBean.set("ID_PLAN_ACCES", actualite.getIdPlanAcces());
		infoBean.set("LISTE_PLANS_ACCES", Media.getListePhotoParType(this, "0003"));
		infoBean.set("TYPE_EVENEMENT", actualite.getTypeEvenement());
		infoBean.set("LIBELLE_TYPE_EVENEMENT", actualite.getLibelleTypeEvenement());
		infoBean.set("LISTE_TYPE_EVENEMENT", Libelle.getListe("03", LangueUtil.getLocale(actualite.getLangue())));
		infoBean.set("RESPONSABLE_EVENEMENT", actualite.getResponsableEvenement());
		infoBean.set("EMAIL_RESPONSABLE_EVENEMENT", actualite.getEmailResponsableEvenement());
		infoBean.set("CODE_RUBRIQUE", actualite.getCodeRubrique());
		infoBean.set("LIBELLE_CODE_RUBRIQUE", actualite.getInfosRubrique().getLibelleAffichable());
		infoBean.set("CODE_RATTACHEMENT", actualite.getCodeRattachement());
		infoBean.set("LIBELLE_CODE_RATTACHEMENT", actualite.getInfosStructure().getLibelleAffichable());
		infoBean.set("LIBELLE_AFFICHABLE", actualite.getLibelleAffichable());
		infoBean.set("SOUS_ONGLET", "PRINCIPAL"); // onglet par defaut
		if (infoBean.get("SAISIE_FRONT") != null) {
			// dans le front, les types d'evenements sont selectionnes via des cases a cocher
			int iType = 0;
			// tri des types par libellés
			final Libelle libelle = new Libelle();
			libelle.setCtx(this);
			if (libelle.select("WHERE TYPE='03' AND LANGUE='0' ORDER BY LIBELLE") > 0) {
				while (libelle.nextItem()) {
					// Pour chaque type, on regarde s'il est sélectionné
					String valeurType = "0";
					if (actualite.getVecteurTypesEvenement().contains(libelle.getCode())) {
						valeurType = "1";
					}
					infoBean.set("CODE_TYPE#" + iType, libelle.getCode());
					infoBean.set("LIBELLE_TYPE#" + iType, libelle.getLibelle());
					infoBean.set("VALEUR_TYPE#" + iType, valeurType);
					iType++;
				}
			}
			infoBean.set("TYPES_NB_ITEMS", String.valueOf(iType));
		}
		ControleurUniv.preparerPRINCIPAL(infoBean, actualite, this);
	}

	/**
	 * Traite l'ecran de saisie d'une fiche.
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void traiterPRINCIPAL() throws Exception {
		// chargement de l'objet
		if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
			actualite.init();
		} else {
			actualite.setIdActualite(new Long(infoBean.getString("ID_ACTUALITE")));
			actualite.retrieve();
		}
		// changement d'onglet
		if (action.equals(InfoBean.ACTION_ONGLET)) {
			infoBean.set("SOUS_ONGLET", infoBean.getString("SOUS_ONGLET_DEMANDE"));
		}
		// validation de l'ecran
		else if (action.equals(InfoBean.ACTION_ENREGISTRER)) {
			alimenteDonneesCreation(actualite, false);
			actualite.setTitre(infoBean.getString("TITRE"));
			actualite.setDateEvenement(infoBean.getDate("DATE_EVENEMENT"));
			actualite.setCodeRattachement(infoBean.getString("CODE_RATTACHEMENT"));
			if (!"0000".equals(infoBean.getString("CODE_RUBRIQUE"))) {
				actualite.setCodeRubrique(infoBean.getString("CODE_RUBRIQUE"));
			} else {
				actualite.setCodeRubrique("");
			}
			final Date dateFinSaisie = infoBean.getDate("DATE_FIN_EVENEMENT");
			if (Formateur.estSaisie(dateFinSaisie)) {
				if (dateFinSaisie.before(actualite.getDateEvenement())) {
					throw new ErreurApplicative(MessageHelper.getMessage(infoBean.getNomExtension(), "BO_ACTUALITE_ERREUR_DATE_FIN"));
				}
				actualite.setDateFinEvenement(infoBean.getDate("DATE_FIN_EVENEMENT"));
			} else {
				if (Formateur.estSaisie(infoBean.getDate("DATE_EVENEMENT"))) {
					actualite.setDateFinEvenement(infoBean.getDate("DATE_EVENEMENT"));
				} else {
					actualite.setDateFinEvenement(new Date(0));
				}
			}
			actualite.setThematique(infoBean.getString("THEMATIQUE"));
			actualite.setResume(infoBean.getString("RESUME"));
			actualite.setRattachementBandeau(infoBean.getString("RATTACHEMENT_BANDEAU"));
			actualite.setLieu(infoBean.getString("LIEU"));
			if (infoBean.get("ID_PLAN_ACCES") instanceof String) {
				//si l'onglet en cours ne contient pas le plan d'accès
				actualite.setIdPlanAcces(new Long(infoBean.getString("ID_PLAN_ACCES")));
			} else {
				//si l'onglet en cours contient le plan d'accès
				actualite.setIdPlanAcces((Long) infoBean.get("ID_PLAN_ACCES"));
			}
			actualite.setEmailResponsableEvenement(infoBean.getString("EMAIL_RESPONSABLE_EVENEMENT"));
			actualite.setResponsableEvenement(infoBean.getString("RESPONSABLE_EVENEMENT"));
			if (infoBean.get("SAISIE_FRONT") != null) {
				//dans le front, les types d'evenements sont des cases a cocher
				if (infoBean.get("TYPES_NB_ITEMS") != null) {
					final Vector<String> v = new Vector<>();
					for (int j = 0; j < Integer.parseInt(infoBean.getString("TYPES_NB_ITEMS")); j++) {
						if (infoBean.getString("VALEUR_TYPE#" + j) != null && infoBean.getString("VALEUR_TYPE#" + j).equals("1")) {
							v.add(infoBean.getString("CODE_TYPE#" + j));
						}
					}
					actualite.setVecteurTypesEvenements(v);
				}
				actualite.setDescription(infoBean.getString("CORPS"));
                actualite.setSiteEvenement(infoBean.getString("SITE_EVENEMENT"));
                actualite.setPartenaires(infoBean.getString("PARTENAIRES"));
                actualite.setComplementDate(infoBean.getString("COMPLEMENT_DATE"));
			} else {
				actualite.setDescription(infoBean.getString("CORPS"));
				actualite.setTypeEvenement(infoBean.getString("TYPE_EVENEMENT"));
				actualite.setSiteEvenement(infoBean.getString("SITE_EVENEMENT"));
				actualite.setPartenaires(infoBean.getString("PARTENAIRES"));
				actualite.setComplementDate(infoBean.getString("COMPLEMENT_DATE"));
			}
		}
		// appel au traitement general
		ecranLogique = ControleurUniv.traiterPRINCIPAL(infoBean, actualite, this);
		if (ecranLogique.length() == 0) {
			etat = FIN;
		}
	}
}
