/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: actualite
 * Version: 1.02.07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.rss;

import java.lang.reflect.Method;

import org.apache.commons.lang.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.config.PropertyHelper;
import com.univ.objetspartages.om.Actualite;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.util.ActualiteHelper;
import com.univ.utils.ContexteUniv;
import com.univ.utils.RequeteUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.UnivWebFmt;

/**
 * Gestionnaire d'export RSS sur des actualités.
 *
 * @author fred
 */
public class ExportRssActualite extends AbstractExportRss {

	/**
	 * The Constructor.
	 *
	 * @param query
	 *            the query
	 * @param objectName
	 *            the object name
	 * @param feedType
	 *            the feed type
	 */
	public ExportRssActualite(final String query, final String objectName, final String feedType) {
		super(query, objectName, feedType);
	}

	/**
	 * Renvoie la description du channel correspondant à cet export rss.
	 *
	 * @return La description du channel
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected String getDescription(final OMContext ctx) throws Exception {
		String description = super.getDescription(ctx);
		if (description.length() == 0) {
			final String typeEvenement = RequeteUtil.renvoyerParametre(getQuery(), "TYPE_EVENEMENT");
			if (typeEvenement.length() > 0) {
				description += Libelle.getLibelle("03", typeEvenement, ctx.getLocale()) + " ";
			}
			final String thematique = RequeteUtil.renvoyerParametre(getQuery(), "THEMATIQUE");
			if (thematique.length() > 0) {
				description += "\"" + Libelle.getLibelle("04", thematique, ctx.getLocale()) + "\" ";
			}
			final String codeRattachement = RequeteUtil.renvoyerParametre(getQuery(), "CODE_RATTACHEMENT");
			if (codeRattachement.length() > 0) {
				description += "[" + Structure.getLibelleRattachement(ctx, codeRattachement, "0", true).replaceAll("<br />", " / ") + "]";
			}
			if (description.length() == 0) {
				description = "Les 10 dernières actualités";
			} else {
				description = "Sélection " + description;
			}
		}
		return description;
	}

	/**
	 * Renvoie le titre de l'actu.
	 *
	 * @param fiche
	 *            L'actu à exporter
	 *
	 * @return Le titre
	 *
	 */
	@Override
	protected String getItemTitle(final FicheUniv fiche) {
		return ((Actualite) fiche).getTitre();
	}

	/**
	 * Renvoie la description de l'actu.
	 *
	 * @param fiche
	 *            L'actu à exporter
	 *
	 * @return La description
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected String getItemDescription(final ContexteUniv ctx, final FicheUniv fiche) throws Exception {
		String description = StringUtils.EMPTY;
		final Ressource ressource = Ressource.getFichier(fiche);
		if (ressource != null) {
			final String urlVignette = URLResolver.getAbsoluteUrl(ressource.getUrlVignette(), ctx);
			description = "<img src=\"" + urlVignette + "\" width=\"150px\" /><br />";
		}
		final String optionExport = PropertyHelper.getProperty("actualite", "export_rss.actualite.description");
		if (optionExport == null || optionExport.length() == 0) {
			description += ((Actualite) fiche).getDescription();
		} else {
			try {
				// Récupération de la liste de valeurs
				final String nomAccesseurGET = "get" + optionExport.substring(0, 1).toUpperCase() + optionExport.substring(1);
				// ex méthode getParcourss() sur l'objet Formation
				final Method accesseurGET = fiche.getClass().getMethod(nomAccesseurGET);
				final Object o[] = new Object[0];
				description += (String) accesseurGET.invoke(fiche, o);
			} catch (final Exception e) {
				throw new ErreurApplicative(String.format(ActualiteHelper.getMessage("ACTUALITE_ERREUR_RSS"), optionExport));
			}
		}
		//	on a besoin de cette info pour le formatage HTML
		ctx.setFicheCourante(fiche);
		return UnivWebFmt.formaterEnHTML(ctx, description);
	}
}
