/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: cataloguelien
 * Version: 1.02.04
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.Enumeration;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import com.kportal.cms.objetspartages.om.ContentRedirect;
import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.cms.objetspartages.annotation.FicheAnnotation;
import com.univ.objetspartages.sgbd.LienDB;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RequeteUtil;
import com.univ.utils.UnivWebFmt;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;

/**
 * Classe representant un objet lien.
 */
@FicheAnnotation()
public class Lien extends LienDB implements FicheUniv, DiffusionSelective, Cloneable, ContentRedirect {

	/**
	 *
	 */
	private static final long serialVersionUID = 4900701412833993966L;

	/**
	 * Renvoie le libelle a afficher (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param codes
	 *            the codes
	 * @param langue
	 *            the langue
	 *
	 * @return the libelle affichable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleAffichable(final OMContext ctx, final String codes, String langue) throws Exception {
		String res = "";
		if (codes == null || codes.length() == 0) {
			return res;
		}
		final Lien lien = new Lien();
		lien.init();
		lien.setCtx(ctx);
		if (langue == null || langue.length() == 0) {
			langue = "0";
		}
		final StringTokenizer st = new StringTokenizer(codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			// On cherche d'abord la version en ligne puis les autres versions
			int count = lien.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) {
				count = lien.selectCodeLangueEtat(code, langue, "");
			}
			if (count > 0) {
				if (res.length() > 1) {
					res += ";";
				}
				if (lien.nextItem()) {
					res += lien.getLibelleAffichable();
				}
			} else {
				res += "-";
			}
		}
		return res;
	}

	/**
	 * Renvoie la fiche demandee (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 *
	 * @return the fiche lien
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Lien getFicheLien(final OMContext ctx, final String code, String langue) throws Exception {
		Lien fiche = null;
		if (code != null && code.length() > 0) {
			if (langue == null || langue.length() == 0) {
				langue = "0"; // francais par defaut
			}
			fiche = new Lien();
			fiche.init();
			fiche.setCtx(ctx);
			int count = fiche.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) // si pas de version en ligne
			{
				count = fiche.selectCodeLangueEtat(code, langue, "");
			}
			if (count == 0) // si pas de version dans la langue demandee
			{
				count = fiche.selectCodeLangueEtat(code, "", "0003");
			}
			if (count > 0) {
				fiche.nextItem();
			}
		}
		return fiche;
	}

	/**
	 * Initialise l'objet metier.
	 */
	@Override
	public void init() {
		FicheUnivMgr.init(this);
		setIdLien(new Long(0));
		setCode("");
		setLangue("0");
		setTitre("");
		setUrl("");
		setThematique("");
		setType("");
		setCategorie("");
		setResume("");
		setDescription("");
		setAuteurediteur("");
		setNomresponsable("");
		setEmailresponsable("");
	}

	/**
	 * Renvoie le libelle a afficher.
	 *
	 * @return the libelle affichable
	 */
	@Override
	public String getLibelleAffichable() {
		return getTitre();
	}

	/**
	 * Renvoie l'identifiant de la fiche.
	 *
	 * @return the id fiche
	 */
	@Override
	public Long getIdFiche() {
		return getIdLien();
	}

	/**
	 * Valorise l'identifiant de la fiche.
	 *
	 * @param idFiche
	 *            the id fiche
	 */
	@Override
	public void setIdFiche(final Long idFiche) {
		setIdLien(idFiche);
	}

	/**
	 * Renvoie le libelle pour l'attribut thematique.
	 *
	 * @return the libelle thematique
	 *
	 */
	public String getLibelleThematique() {
		return Libelle.getLibelle("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le libelle pour l'attribut type.
	 *
	 * @return the libelle type
	 *
	 */
	public String getLibelleType() {
		return Libelle.getLibelle("70", getType(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le libelle pour l'attribut categorie.
	 *
	 * @return the libelle categorie
	 *
	 */
	public String getLibelleCategorie() {
		return Libelle.getLibelle("71", getCategorie(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Retourne une liste de libelle representant les catgorie
	 *
	 * @return
	 */
	public List<String> getListLibelleCategorie() {
		return Libelle.getLibelleSousFormeDeListe("71", getCategorie(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Récupération du libellé à afficher (spécifique au back-office).
	 *
	 * @return the libelle back office
	 *
	 */
	public String getLibelleBackOffice() {
		return getLibelleAffichable() + " [" + Rubrique.getIntitule(getCodeRubrique()) + "]";
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut description.
	 *
	 * @return the formated description
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedDescription() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getDescription());
	}

	/**
	 * Renvoie le fichier joint correspondant a l'attribut vignette.
	 *
	 * @return the vignette
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Ressource getVignette() throws Exception {
		return Ressource.getFichier(this);
	}

	/**
	 * Renvoie la rubrique de la fiche.
	 *
	 * @return the infos rubrique
	 *
	 */
	public InfosRubriques getInfosRubrique() {
		return Rubrique.renvoyerItemRubrique(getCodeRubrique());
	}

	/**
	 * Renvoie la structure de la fiche.
	 *
	 * @return the infos structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public InfosStructure getInfosStructure() throws Exception {
		return Structure.renvoyerItemStructure(getCodeRattachement(), getLangue());
	}

	/**
	 * Renvoie les metadonnees de la fiche.
	 *
	 * @return the meta
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Metatag getMeta() throws Exception {
		return FicheUnivMgr.lireMeta(this);
	}

	/**
	 * Renvoie une chaine correspondant a la concatenation des champs texte de l'objet (utilise pour l'indexation Lucene).
	 *
	 * @return the full text string
	 *
	 */
	@Override
	public String getFullTextString() {
		final String s = getUrl() + " " + StringUtils.replace(getLibelleThematique(), ";", " ") + " " + StringUtils.replace(getLibelleType(), ";",
			" ") + " " + StringUtils.replace(getLibelleCategorie(), ";", " ") + " " + getResume() + " " + getDescription() + " " + getAuteurediteur() + " " + getNomresponsable() + " " + getEmailresponsable();
		return s;
	}

	/**
	 * Duplication dans une nouvelle langue, le nouvel objet devient l'objet courant.
	 *
	 * @param nouvelleLangue
	 *            the nouvelle langue
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public void dupliquer(final String nouvelleLangue) throws Exception {
		// donnees a reinitialiser dans le nouvel objet
		FicheUnivMgr.dupliquer(this);
		// creation objet dans la langue demandee
		setIdFiche(new Long(0));
		setLangue(nouvelleLangue);
		add();
	}

	/**
	 * Traitement d'une requete sur l'objet.
	 *
	 * @param requete
	 *            the requete
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int traiterRequete(final String requete) throws Exception {
		// Recuperation des parametres de la requete
		final String titre = RequeteUtil.renvoyerParametre(requete, "TITRE");
		final String url = RequeteUtil.renvoyerParametre(requete, "URL");
		final String auteurediteur = RequeteUtil.renvoyerParametre(requete, "AUTEUREDITEUR");
		final String thematique = RequeteUtil.renvoyerParametre(requete, "THEMATIQUE");
		final String type = RequeteUtil.renvoyerParametre(requete, "TYPE");
		final String categorie = RequeteUtil.renvoyerParametre(requete, "CATEGORIE");
		final String description = RequeteUtil.renvoyerParametre(requete, "DESCRIPTION");
		final String ids = RequeteUtil.renvoyerParametre(requete, "IDS");
		final String nombre = RequeteUtil.renvoyerParametre(requete, "NOMBRE");
		final String codeRubrique = RequeteUtil.renvoyerParametre(requete, "CODE_RUBRIQUE");
		final String codeRattachement = RequeteUtil.renvoyerParametre(requete, "CODE_RATTACHEMENT");
		final String langue = RequeteUtil.renvoyerParametre(requete, "LANGUE");
		final String nomResponsable = RequeteUtil.renvoyerParametre(requete, "NOMRESPONSABLE");
		final String selection = RequeteUtil.renvoyerParametre(requete, "SELECTION");
		final String etatObjet = StringUtils.defaultIfBlank(RequeteUtil.renvoyerParametre(requete, "ETAT_OBJET"), EtatFiche.EN_LIGNE.toString());
		final String codeRedacteur = RequeteUtil.renvoyerParametre(requete, "CODE_REDACTEUR");
		final String codeFiche = RequeteUtil.renvoyerParametre(requete, "CODE");
		String order = "T1.TITRE";
		String orderTri = "ASC";
		if (selection.length() > 0) {
			// Derniers liens publiés
			if (selection.equals("0007")) {
				order = "META.META_DATE_MISE_EN_LIGNE";
				orderTri = "DESC";
			}
		}
		return select(titre, url, auteurediteur, nomResponsable, thematique, type, categorie, description, ids, nombre, codeRubrique, codeRattachement, langue, etatObjet,
			codeRedacteur, order + " " + orderTri, codeFiche);
	}

	/**
	 * Selection d'une fiche a partir de son code, de sa langue et de son etat.
	 *
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 * @param etatObjet
	 *            the etat objet
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int selectCodeLangueEtat(final String code, final String langue, final String etatObjet) throws Exception {
		final ClauseWhere whereCodeLangueEtat = ConditionHelper.whereCodeLangueEtat(code, langue, etatObjet);
		return select(whereCodeLangueEtat.formaterSQL());
	}

	/**
	 * Renvoie les liens pour la rubrique demandee.
	 *
	 * @param codeRubrique
	 *            the code rubrique
	 * @param langue
	 *            the langue
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
    @Deprecated
	public int selectParCodeRubrique(final String codeRubrique, final String langue) throws Exception {
		final ClauseWhere whereCodeRubriqueLangueEtat = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeRubrique)) {
			whereCodeRubriqueLangueEtat.setPremiereCondition(ConditionHelper.egalVarchar("T1.CODE_RUBRIQUE", codeRubrique));
		}
		if (StringUtils.isNotEmpty(langue)) {
			whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(langue)) {
			whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", "0003"));
		}
		return select(whereCodeRubriqueLangueEtat.formaterSQL());
	}

	/**
	 * Renvoie la liste des references a cette fiche.
	 *
	 * @return the references
	 */
	@Override
	public String getReferences() {
		return FicheUnivMgr.getReferenceParTexte(getDescription()) + FicheUnivMgr.getReferenceParJointure("structure", getCodeRattachement());
	}

	/**
	 * Selection d'une instance de l'objet Lien a partir de l'ensemble des criteres.
	 *
	 * @param titre
	 *            the titre
	 * @param url
	 *            the url
	 * @param auteurediteur
	 *            the auteurediteur
	 * @param nomResponsable
	 *            the nom responsable
	 * @param thematique
	 *            the thematique
	 * @param type
	 *            the type
	 * @param categorie
	 *            the categorie
	 * @param description
	 *            the description
	 * @param ids
	 *            the ids
	 * @param nombre
	 *            the nombre
	 * @param codeRubrique
	 *            the code rubrique
	 * @param codeRattachement
	 *            the code rattachement
	 * @param langue
	 *            the langue
	 * @param etatObjet
	 *            the etat objet
	 * @param codeRedacteur
	 *            the code redacteur
	 * @param dateDebut
	 *            the date debut
	 * @param dateFin
	 *            the date fin
	 * @param ordre
	 *            the ordre
	 * @param code
	 *            the code
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String titre, final String url, final String auteurediteur, final String nomResponsable, final String thematique, final String type,
		final String categorie, final String description, final String ids, final String nombre, final String codeRubrique, final String codeRattachement, final String langue,
		final String etatObjet, final String codeRedacteur, final String ordre, final String code) throws Exception {
		RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(titre)) {
			where.setPremiereCondition(ConditionHelper.rechercheMots("T1.TITRE", titre));
		}
		if (StringUtils.isNotEmpty(url)) {
			where.and(ConditionHelper.rechercheMots("T1.URL", url));
		}
		if (StringUtils.isNotEmpty(auteurediteur)) {
			where.and(ConditionHelper.rechercheMots("T1.AUTEUREDITEUR", auteurediteur));
		}
		if (StringUtils.isNotEmpty(nomResponsable)) {
			where.and(ConditionHelper.rechercheMots("T1.NOMRESPONSABLE", nomResponsable));
		}
		if (StringUtils.isNotEmpty(thematique) && !"0000".equals(thematique)) {
			where.and(ConditionHelper.likePourValeursMultiple("THEMATIQUE", thematique));
		}
		if (StringUtils.isNotEmpty(type) && !"0000".equals(type)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.TYPE", type));
		}
		if (StringUtils.isNotEmpty(categorie) && !"0000".equals(categorie)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.CATEGORIE", categorie));
		}
		if (StringUtils.isNotEmpty(description)) {
			where.and(ConditionHelper.rechercheMots("T1.DESCRIPTION", description));
		}
		if (StringUtils.isNotEmpty(ids) && !"0000".equals(ids)) {
			where.and(ConditionHelper.in("ID_LIEN", Chaine.getVecteurPointsVirgules(ids)));
		}
		if (StringUtils.isNotEmpty(codeRattachement)) {
			where.and(ConditionHelper.getConditionStructure("T1.CODE_RATTACHEMENT", codeRattachement));
		}
		if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(etatObjet) && !"0000".equals(etatObjet)) {
			where.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", etatObjet));
		}
		if (StringUtils.isNotEmpty(codeRedacteur)) {
			where.and(ConditionHelper.egalVarchar("T1.CODE_REDACTEUR", codeRedacteur));
		}
		if (StringUtils.isNotEmpty(code)) {
			where.and(ConditionHelper.egalVarchar("T1.CODE", code));
		}
		requeteSelect = RequeteSQLHelper.getRequeteGenerique(where, ctx, this, codeRubrique);
		ClauseOrderBy orderBy = new ClauseOrderBy();
		if (StringUtils.isNotEmpty(ordre)) {
			orderBy = OrderByHelper.reconstruireClauseOrderBy(ordre);
		} else {
			orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		}
		requeteSelect.orderBy(orderBy);
		final ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
		requeteSelect.limit(limite);
		return select(requeteSelect.formaterRequete());
	}

	/**
	 * Renvoie la description de la fiche (utilise dans SaisieCommentaire).
	 *
	 * @return the contenu
	 *
	 * @see com.univ.objetspartages.om.DescriptionCommentaire#getContenu()
	 */
	public String getContenu() {
		return getDescription();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Object clone() {
		final Lien clone = new Lien();
		clone.setAuteurediteur(getAuteurediteur());
		clone.setCategorie(getCategorie());
		clone.setCode(getCode());
		clone.setCodeRattachement(getCodeRattachement());
		clone.setCodeRedacteur(getCodeRedacteur());
		clone.setCodeRubrique(getCodeRubrique());
		clone.setCodeValidation(getCodeValidation());
		clone.setContenuEncadre(getContenuEncadre());
		clone.setCtx(ctx);
		clone.setDateAlerte(getDateAlerte());
		clone.setDateCreation(getDateCreation());
		clone.setDateModification(getDateModification());
		clone.setDateProposition(getDateProposition());
		clone.setDateValidation(getDateValidation());
		clone.setDescription(getDescription());
		clone.setDiffusionModeRestriction(getDiffusionModeRestriction());
		clone.setDiffusionPublicVise(getDiffusionPublicVise());
		clone.setDiffusionPublicViseRestriction(getDiffusionPublicViseRestriction());
		clone.setEmailresponsable(getEmailresponsable());
		clone.setEncadreRecherche(getEncadreRecherche());
		clone.setEncadreRechercheBis(getEncadreRechercheBis());
		clone.setEtatObjet(getEtatObjet());
		clone.setIdFiche(getIdFiche());
		clone.setIdLien(getIdLien());
		clone.setLangue(getLangue());
		clone.setMessageAlerte(getMessageAlerte());
		clone.setMetaDescription(getMetaDescription());
		clone.setMetaKeywords(getMetaKeywords());
		clone.setNbHits(getNbHits());
		clone.setNomresponsable(getNomresponsable());
		clone.setQualifier(qualifier);
		clone.setResume(getResume());
		clone.setThematique(getThematique());
		clone.setTitre(getTitre());
		clone.setTitreEncadre(getTitreEncadre());
		clone.setType(getType());
		clone.setUrl(getUrl());
		return clone;
	}

	/**
	 * Renvoie la liste des types sous forme de vecteur.
	 *
	 * @return the vecteur types
	 */
	public Vector<String> getVecteurTypes() {
		final Vector<String> v = new Vector<>();
		final StringTokenizer st = new StringTokenizer(getType(), ";");
		while (st.hasMoreTokens()) {
			v.add(st.nextToken());
		}
		return v;
	}

	/**
	 * Valorise la liste des types sous forme de vecteur.
	 *
	 * @param v
	 *            the v
	 */
	public void setVecteurTypes(final Vector<String> v) {
		String liste = "";
		final Enumeration<String> en = v.elements();
		while (en.hasMoreElements()) {
			liste += en.nextElement() + ";";
		}
		setType(liste);
	}

	/**
	 * Renvoie la liste des thematiques sous forme de vecteur.
	 *
	 * @return the vecteur thematiques
	 */
	public Vector<String> getVecteurThematiques() {
		final Vector<String> v = new Vector<>();
		final StringTokenizer st = new StringTokenizer(getThematique(), ";");
		while (st.hasMoreTokens()) {
			v.add(st.nextToken());
		}
		return v;
	}

	/**
	 * Valorise la liste des thematiques sous forme de vecteur.
	 *
	 * @param v
	 *            the v
	 */
	public void setVecteurThematiques(final Vector<String> v) {
		String liste = "";
		final Enumeration<String> en = v.elements();
		while (en.hasMoreElements()) {
			liste += en.nextElement() + ";";
		}
		setThematique(liste);
	}

	/**
	 * Renvoie la liste des categories sous forme de vecteur.
	 *
	 * @return the vecteur categories
	 */
	public Vector<String> getVecteurCategories() {
		final Vector<String> v = new Vector<>();
		final StringTokenizer st = new StringTokenizer(getCategorie(), ";");
		while (st.hasMoreTokens()) {
			v.add(st.nextToken());
		}
		return v;
	}

	/**
	 * Valorise la liste des categories sous forme de vecteur.
	 *
	 * @param v
	 *            the v
	 */
	public void setVecteurCategories(final Vector<String> v) {
		String liste = "";
		final Enumeration<String> en = v.elements();
		while (en.hasMoreElements()) {
			liste += en.nextElement() + ";";
		}
		setCategorie(liste);
	}

	@Override
	public String getUrlRedirect() {
		return getUrl();
	}
}
