/**
 * Copyright (C) 2015 Kosmos contact@kosmos.fr
 *
 * Projet: ged
 * Version: 1.02.04
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.Date;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.cms.objetspartages.annotation.FicheAnnotation;
import com.kportal.cms.objetspartages.om.ContentRedirect;
import com.kportal.core.config.PropertyHelper;
import com.univ.objetspartages.sgbd.DocumentDB;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;
import com.univ.utils.DateUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RequeteUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.UnivWebFmt;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;
import com.univ.utils.sql.operande.TypeOperande;
import com.univ.xml.NodeUtil;

// TODO: Auto-generated Javadoc
/**
 * Classe representant un objet document.
 */
@FicheAnnotation()
public class Document extends DocumentDB implements FicheUniv, DiffusionSelective, FicheExportableXML, ContentRedirect {

	/**
	 *
	 */
	private static final long serialVersionUID = -427556824853088356L;

	/** attribut permettant de stocker l'url de redirection cf ContentRedirect#getUrlRedirect */
	private String urlRedirect;

	/**
	 * Renvoie le libelle a afficher (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param codes
	 *            the codes
	 * @param langue
	 *            the langue
	 *
	 * @return the libelle affichable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleAffichable(final OMContext ctx, final String codes, String langue) throws Exception {
		String res = "";
		if (codes == null || codes.length() == 0) {
			return res;
		}
		final Document document = new Document();
		document.init();
		document.setCtx(ctx);
		if (langue == null || langue.length() == 0) {
			langue = "0";
		}
		final StringTokenizer st = new StringTokenizer(codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			// On cherche d'abord la version en ligne puis les autres versions
			int count = document.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) {
				count = document.selectCodeLangueEtat(code, langue, "");
			}
			if (count > 0) {
				if (res.length() > 1) {
					res += ";";
				}
				if (document.nextItem()) {
					res += document.getLibelleAffichable();
				}
			} else {
				res += "-";
			}
		}
		return res;
	}

	/**
	 * Renvoie la fiche demandee (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param ctx
	 *            the ctx
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 *
	 * @return the fiche document
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Document getFicheDocument(final OMContext ctx, final String code, String langue) throws Exception {
		Document fiche = null;
		if (code != null && code.length() > 0) {
			if (langue == null || langue.length() == 0) {
				langue = "0"; // francais par defaut
			}
			fiche = new Document();
			fiche.init();
			fiche.setCtx(ctx);
			int count = fiche.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) // si pas de version en ligne
			{
				count = fiche.selectCodeLangueEtat(code, langue, "");
			}
			if (count == 0) // si pas de version dans la langue demandee
			{
				count = fiche.selectCodeLangueEtat(code, "", "0003");
			}
			if (count > 0) {
				fiche.nextItem();
			}
		}
		return fiche;
	}

	/**
	 * Initialise l'objet metier.
	 */
	@Override
	public void init() {
		FicheUnivMgr.init(this);
		setIdDocument(new Long(0));
		setCode("");
		setLangue("0");
		setTitre("");
		setSousTitre("");
		setResume("");
		setDateCreationDocument(new Date(System.currentTimeMillis()));
		setTypeDocument("");
		setSourceSupport("");
		setAuteurs("");
		setEditeur("");
		setDescription("");
		setReferencesIsdn("");
		setDateParutionDocument(new Date(0));
		setDateDerniereMajDocument(new Date(0));
		setProchaineMajPeriodicite("");
		setThematique("");
		setComplements("");
		// champs plus utilises, initialises a vide
		setDateArchivageDocument(new Date(0));
		setTheme("");
		setComiteSelection("");
	}

	/**
	 * Récupération du libellé à afficher (spécifique au back-office).
	 *
	 * @return the libelle back office
	 *
	 */
	public String getLibelleBackOffice() {
		return getLibelleAffichable() + " [" + Rubrique.getIntitule(getCodeRubrique()) + "]";
	}

	/**
	 * Renvoie le libelle a afficher.
	 *
	 * @return the libelle affichable
	 */
	@Override
	public String getLibelleAffichable() {
		return getTitre();
	}

	/**
	 * Renvoie l'identifiant de la fiche.
	 *
	 * @return the id fiche
	 */
	@Override
	public Long getIdFiche() {
		return getIdDocument();
	}

	/**
	 * Valorise l'identifiant de la fiche.
	 *
	 * @param idFiche
	 *            the id fiche
	 */
	@Override
	public void setIdFiche(final Long idFiche) {
		setIdDocument(idFiche);
	}

	/**
	 * Renvoie la liste des fichiers joints de la fiche.
	 *
	 * @return the liste fichiers
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<Ressource> getListeFichiers() throws Exception {
		return Ressource.getListeFichier(this);
	}

	/**
	 * Renvoie le libelle pour l'attribut typeDocument.
	 *
	 * @return the libelle type document
	 *
	 */
	public String getLibelleTypeDocument() {
		return Libelle.getLibelle("40", getTypeDocument(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le libelle pour l'attribut thematique.
	 *
	 * @return the libelle thematique
	 *
	 */
	public String getLibelleThematique() {
		return Libelle.getLibelle("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie les thematiques sous forme de liste pour qu'elles soient affichees sous forme de liste en front.
	 */
	public List<String> getListLibelleThematique() {
		return Libelle.getLibelleSousFormeDeListe("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie la rubrique de la fiche.
	 *
	 * @return the infos rubrique
	 *
	 */
	public InfosRubriques getInfosRubrique() {
		return Rubrique.renvoyerItemRubrique(getCodeRubrique());
	}

	/**
	 * Renvoie la structure de la fiche.
	 *
	 * @return the infos structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public InfosStructure getInfosStructure() throws Exception {
		return Structure.renvoyerItemStructure(getCodeRattachement(), getLangue());
	}

	/**
	 * Renvoie les metadonnees de la fiche.
	 *
	 * @return the meta
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Metatag getMeta() throws Exception {
		return FicheUnivMgr.lireMeta(this);
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut auteurs.
	 *
	 * @return the formated auteurs
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedAuteurs() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getAuteurs());
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut complements.
	 *
	 * @return the formated complements
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedComplements() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getComplements());
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut description.
	 *
	 * @return the formated description
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedDescription() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getDescription());
	}

	/**
	 * Renvoie une chaine correspondant a la concatenation des champs texte de l'objet (utilise pour l'indexation Lucene).
	 *
	 * @return the full text string
	 *
	 */
	@Override
	public String getFullTextString() {
		final String s = getSousTitre() + " " + getResume() + " " + getLibelleTypeDocument() + " " + getSourceSupport() + " " + getAuteurs() + " " + getEditeur() + " " + getDescription() + " " + getReferencesIsdn() + " " + getProchaineMajPeriodicite() + " " + StringUtils.replace(
			getLibelleThematique(), ";", " ") + " " + getComplements();
		return s;
	}

	/**
	 * Duplication dans une nouvelle langue, le nouvel objet devient l'objet courant.
	 *
	 * @param nouvelleLangue
	 *            the nouvelle langue
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public void dupliquer(final String nouvelleLangue) throws Exception {
		// donnees a reinitialiser dans le nouvel objet
		FicheUnivMgr.dupliquer(this);
		// creation objet dans la langue demandee
		setIdFiche(new Long(0));
		setLangue(nouvelleLangue);
		add();
	}

	/**
	 * Traitement d'une requete sur l'objet.
	 *
	 * @param requete
	 *            the requete
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int traiterRequete(final String requete) throws Exception {
		// Recuperation des parametres de la requete
		final String titre = RequeteUtil.renvoyerParametre(requete, "TITRE");
		final String auteur = RequeteUtil.renvoyerParametre(requete, "AUTEUR");
		final String typeDocument = RequeteUtil.renvoyerParametre(requete, "TYPE_DOCUMENT");
		final String thematique = RequeteUtil.renvoyerParametre(requete, "THEMATIQUE");
		final String codeRubrique = RequeteUtil.renvoyerParametre(requete, "CODE_RUBRIQUE");
		final String codeRattachement = RequeteUtil.renvoyerParametre(requete, "CODE_RATTACHEMENT");
		final String langue = RequeteUtil.renvoyerParametre(requete, "LANGUE");
		final String ids = RequeteUtil.renvoyerParametre(requete, "IDS");
		final String nbMax = RequeteUtil.renvoyerParametre(requete, "NOMBRE");
		final String period = RequeteUtil.renvoyerParametre(requete, "SELECTION");
		final String tri = RequeteUtil.renvoyerParametre(requete, "TRI_DATE");
		final String nbDays = RequeteUtil.renvoyerParametre(requete, "NB_DAYS");
		final String sDateDebut = RequeteUtil.renvoyerParametre(requete, "DATE_DEBUT");
		final String sDateFin = RequeteUtil.renvoyerParametre(requete, "DATE_FIN");
		final String code = RequeteUtil.renvoyerParametre(requete, "CODE");
		final String etatObjet = StringUtils.defaultIfBlank(RequeteUtil.renvoyerParametre(requete, "ETAT_OBJET"), EtatFiche.EN_LIGNE.toString());
		final String codeRedacteur = RequeteUtil.renvoyerParametre(requete, "CODE_REDACTEUR");
		Date dateDebut = null;
		Date dateFin = null;
		String orderDate = "";
		String orderTri = "";
		if (period.length() > 0) {
			orderDate = "T1.DATE_CREATION_DOCUMENT";
			final Date today = new Date(System.currentTimeMillis());
			if (period.endsWith("01")) // derniers documents
			{
				dateFin = today; // aujourd'hui
				orderTri = "DESC";
			} else if (period.endsWith("02")) // documents du jour
			{
				dateDebut = today; // aujourd'hui
				dateFin = today; // aujourd'hui
			} else if (period.endsWith("03")) // documents de la semaine
			{
				dateDebut = DateUtil.getFirstDayOfWeek(today);
				dateFin = DateUtil.getLastDayOfWeek(today);
			} else if (period.endsWith("04")) // documents du mois
			{
				dateDebut = DateUtil.getFirstDayOfMonth(today);
				dateFin = DateUtil.getLastDayOfMonth(today);
			}
			// Derniers documents publiés
			else if (period.endsWith("07")) {
				orderDate = "META.META_DATE_MISE_EN_LIGNE";
				orderTri = "DESC";
			}
		} else if (nbDays.length() > 0) // documents mis en ligne depuis n jours
		{
			//construction de la date : aujourd'hui - nb de jours saisis
			final GregorianCalendar gCal = new GregorianCalendar();
			gCal.setTimeInMillis(System.currentTimeMillis());
			gCal.add(Calendar.DAY_OF_MONTH, -Integer.parseInt(nbDays));
			dateDebut = new Date(gCal.getTimeInMillis());
			dateFin = new Date(System.currentTimeMillis()); // aujourd'hui
			orderDate = "T1.DATE_CREATION_DOCUMENT";
		} else if (sDateDebut.length() > 0 || sDateFin.length() > 0) {
			dateDebut = DateUtil.parseDate(sDateDebut);
			dateFin = DateUtil.parseDate(sDateFin);
			orderDate = "T1.DATE_CREATION_DOCUMENT";
		}
		if ("DATE_DESC".equals(tri)) {
			orderTri = "DESC";
		}
		if ("DATE_ASC".equals(tri)) {
			orderTri = "ASC";
		}
		String order = "T1.TITRE";
		if (orderDate.length() > 0) {
			order = orderDate + " " + orderTri + ", " + order;
		}
		return select(code, titre, typeDocument, auteur, thematique, codeRattachement, codeRubrique, langue, codeRedacteur, etatObjet, dateDebut, dateFin, nbMax, order, ids);
	}

	/**
	 * Selection d'une fiche a partir de son code, de sa langue et de son etat.
	 *
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 * @param etatObjet
	 *            the etat objet
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int selectCodeLangueEtat(final String code, final String langue, final String etatObjet) throws Exception {
		final RequeteSQL codeLangueEtatOrdreSurTitre = new RequeteSQL();
		final ClauseWhere whereCodeLangueEtat = ConditionHelper.whereCodeLangueEtat(code, langue, etatObjet);
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		codeLangueEtatOrdreSurTitre.where(whereCodeLangueEtat).orderBy(orderBy);
		return select(codeLangueEtatOrdreSurTitre.formaterRequete());
	}

	/**
	 * Renvoie les documents pour la rubrique demandee.
	 *
	 * @param codeRubrique
	 *            the code rubrique
	 * @param langue
	 *            the langue
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int selectParCodeRubrique(final String codeRubrique, final String langue) throws Exception {
		final RequeteSQL codeRubriqueLangueEnLigneOrdreTitre = new RequeteSQL();
		final ClauseWhere whereCodeRubriqueLangueEtat = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeRubrique)) {
			whereCodeRubriqueLangueEtat.setPremiereCondition(ConditionHelper.egalVarchar("T1.CODE_RUBRIQUE", codeRubrique));
		}
		if (StringUtils.isNotEmpty(langue)) {
			whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		whereCodeRubriqueLangueEtat.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", "0003"));
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		codeRubriqueLangueEnLigneOrdreTitre.where(whereCodeRubriqueLangueEtat).orderBy(orderBy);
		return select(codeRubriqueLangueEnLigneOrdreTitre.formaterRequete());
	}

	/**
	 * Renvoie la liste des references a cette fiche.
	 *
	 * @return the references
	 */
	@Override
	public String getReferences() {
		return FicheUnivMgr.getReferenceParTexte(getSourceSupport()) + FicheUnivMgr.getReferenceParTexte(getAuteurs()) + FicheUnivMgr.getReferenceParTexte(getEditeur()) + FicheUnivMgr.getReferenceParTexte(getDescription()) + FicheUnivMgr.getReferenceParTexte(getEditeur()) + FicheUnivMgr.getReferenceParTexte(getReferencesIsdn()) + FicheUnivMgr.getReferenceParJointure(
			"structure", getCodeRattachement()) + FicheUnivMgr.getReferenceParTexte(getComplements());
	}

	/**
	 * Selection d'une instance de l'objet Document a partir de l'ensemble des criteres.
	 *
	 * @param code
	 *            the code
	 * @param titre
	 *            the titre
	 * @param typeDocument
	 *            the type document
	 * @param auteurs
	 *            the auteurs
	 * @param thematique
	 *            the thematique
	 * @param codeRattachement
	 *            the code rattachement
	 * @param codeRubrique
	 *            the code rubrique
	 * @param langue
	 *            the langue
	 * @param codeRedacteur
	 *            the code redacteur
	 * @param etatObjet
	 *            the etat objet
	 * @param dtstart
	 *            the dtstart
	 * @param dtend
	 *            the dtend
	 * @param nbMax
	 *            the nb max
	 * @param order
	 *            the order
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String code, final String titre, final String typeDocument, final String auteurs, final String thematique, final String codeRattachement,
		final String codeRubrique, final String langue, final String codeRedacteur, final String etatObjet, final Date dtstart, final Date dtend, final String nbMax,
		final String order, final String ids) throws Exception {
		RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(titre)) {
			where.setPremiereCondition(ConditionHelper.rechercheMots("TITRE", titre));
		}
		if (StringUtils.isNotEmpty(auteurs)) {
			where.and(ConditionHelper.rechercheMots("AUTEURS", auteurs));
		}
		if (StringUtils.isNotEmpty(typeDocument) && !"0000".equals(typeDocument)) {
			where.and(ConditionHelper.egalVarchar("T1.TYPE_DOCUMENT", typeDocument));
		}
		if (StringUtils.isNotEmpty(thematique) && !"0000".equals(thematique)) {
			where.and(ConditionHelper.likePourValeursMultiple("THEMATIQUE", thematique));
		}
		if (dtstart != null && Formateur.estSaisie(dtstart)) {
			where.and(ConditionHelper.critereDateDebut("DATE_CREATION_DOCUMENT", dtstart));
		}
		// FIXME : le comportement de dtend n'est pas le même que sur les autres objets (calcul d'interval)
		if (dtend != null && Formateur.estSaisie(dtend)) {
			where.and(ConditionHelper.lessEquals("DATE_CREATION_DOCUMENT", dtend, TypeOperande.DATE));
		}
		if (StringUtils.isNotEmpty(ids) && !"0000".equals(ids)) {
			where.and(ConditionHelper.in("ID_DOCUMENT", Chaine.getVecteurPointsVirgules(ids)));
		}
		if (StringUtils.isNotEmpty(codeRattachement)) {
			where.and(ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", codeRattachement));
		}
		if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(etatObjet) && !"0000".equals(etatObjet)) {
			where.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", etatObjet));
		}
		if (StringUtils.isNotEmpty(codeRedacteur)) {
			where.and(ConditionHelper.egalVarchar("CODE_REDACTEUR", codeRedacteur));
		}
		if (StringUtils.isNotEmpty(code)) {
			where.and(ConditionHelper.egalVarchar("T1.CODE", code));
		}
		requeteSelect = RequeteSQLHelper.getRequeteGenerique(where, ctx, this, codeRubrique);
		ClauseOrderBy orderBy = new ClauseOrderBy();
		if (StringUtils.isNotEmpty(order)) {
			orderBy = OrderByHelper.reconstruireClauseOrderBy(order);
		} else {
			orderBy.orderBy("T1.TITRE", SensDeTri.ASC);
		}
		requeteSelect.orderBy(orderBy);
		final ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nbMax);
		requeteSelect.limit(limite);
		return select(requeteSelect.formaterRequete());
	}

	/* (non-Javadoc)
	 * @see com.univ.objetspartages.om.FicheExportableXML#exportNodeFiche(org.w3c.dom.Node, java.lang.String)
	 */
	@Override
	public Node exportNodeFiche(final Node liste, final String formatExport) throws Exception {
		final org.w3c.dom.Document document = liste.getOwnerDocument();
		final Node fiche = document.createElement("DOCUMENT");
		NodeUtil.addNode(fiche, "CODE", getCode());
		NodeUtil.addNode(fiche, "LANGUE", getLangue());
		NodeUtil.addNode(fiche, "LIBELLE", getLibelleAffichable());
		NodeUtil.addNode(fiche, "TITRE", getTitre());
		NodeUtil.addNode(fiche, "AUTEURS", getAuteurs());
		NodeUtil.addNode(fiche, "RATTACHEMENT", Structure.getLibelleRattachement(ctx, getCodeRattachement(), getLangue(), true));
		NodeUtil.addNode(fiche, "CODE_RUBRIQUE", getCodeRubrique());
		NodeUtil.addNode(fiche, "RUBRIQUE", Rubrique.renvoyerItemRubrique(getCodeRubrique()).getIntitule());
		NodeUtil.addNode(fiche, "COMPLEMENTS", getComplements());
		NodeUtil.addNode(fiche, "DATE_CREATION_DOCUMENT", Formateur.formaterDate(getDateCreationDocument(), ""));
		NodeUtil.addNode(fiche, "DATE_DERNIERE_MAJ_DOCUMENT", Formateur.formaterDate(getDateDerniereMajDocument(), ""));
		NodeUtil.addNode(fiche, "DESCRIPTION", getDescription());
		NodeUtil.addNode(fiche, "EDITEUR", getEditeur());
		NodeUtil.addNode(fiche, "PROCHAINE_MAJ_PERIODICITE", getProchaineMajPeriodicite());
		NodeUtil.addNode(fiche, "REFERENCES_ISDN", getReferencesIsdn());
		NodeUtil.addNode(fiche, "RESUME", getResume());
		NodeUtil.addNode(fiche, "SOURCE_SUPPORT", getSourceSupport());
		NodeUtil.addNode(fiche, "SOUS_TITRE", getSousTitre());
		NodeUtil.addNode(fiche, "THEMATIQUE", getThematique());
		NodeUtil.addNode(fiche, "TYPE_DOCUMENT", Libelle.getLibelle("40", getTypeDocument(), ctx.getLocale()));
		return fiche;
	}

	@Override
	public String getUrlRedirect() {
		if (urlRedirect == null) {
			urlRedirect = "";
			try {
				final Metatag meta = FicheUnivMgr.lireMeta(this);
				if ("1".equals(meta.getMetaDocumentFichiergw())) {
					final List<Ressource> fichiersJoint = getListeFichiers();
					if (fichiersJoint.size() == 1) {
						urlRedirect = URLResolver.getAbsoluteUrl(fichiersJoint.get(0).getUrl(), ctx);
						final String typeDisposition = StringUtils.defaultIfEmpty(PropertyHelper.getCoreProperty("telechargement.disposition"), "attachment");
						if (typeDisposition.equalsIgnoreCase("attachment")) {
							urlRedirect += (urlRedirect.indexOf("?") != -1 ? "&" : "?") + "INLINE=FALSE";
						}
					}
				}
			} catch (final Exception ignore) {}
		}
		return urlRedirect;
	}
}
