/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.core;

import java.sql.Date;
import java.sql.Time;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Classe de formatage (classe interne jtf).
 */
public class Formateur {

	private static final Logger LOG = LoggerFactory.getLogger(Formateur.class);

	/**
	 * Formater.
	 * 
	 * @param _o
	 *            the _o
	 * 
	 * @return the string
	 */
	public static String formater(final Object _o) {
		return formater(_o, "");
	}

	/**
	 * Formater.
	 * 
	 * @param _o
	 *            the _o
	 * @param format
	 *            the format
	 * 
	 * @return the string
	 */
	public static String formater(final Object _o, final String format) {
		String res = "";
		if (_o instanceof Date) {
			res = formaterDate(_o, format);
		}
		if (_o instanceof Time) {
			res = formaterTime(_o, format);
		}
		if (_o instanceof Integer) {
			res = formaterNombre(_o, format);
		}
		if (_o instanceof Long) {
			res = formaterNombre(_o, format);
		}
		if (_o instanceof Double) {
			res = formaterDecimal(_o, format);
		}
		if (_o instanceof String) {
			res = _o.toString();
		}
		return res;
	}

	/**
	 * Formater date.
	 * 
	 * @param _date
	 *            the _date
	 * @param format
	 *            the format
	 * 
	 * @return the string
	 */
	public static String formaterDate(final Object _date, final String format) {
		String value = "";
		try {
			final Date date = (Date) _date;
			final GregorianCalendar cal = new GregorianCalendar();
			cal.setTime(date);
			if (estSaisie(date)) {
				final Object[] arguments = {cal.get(Calendar.DAY_OF_MONTH), cal.get(Calendar.MONTH) + 1, cal.get(Calendar.YEAR)};
				value = MessageFormat.format("{0,number,00}/{1,number,00}/{2,number,0000}", arguments);
			} else {
				value = "";
			}
		} catch (final Exception e) {
			LOG.error("Erreur lors du formatage d'une Date", e);
		}
		return value;
	}

	/**
	 * Formater time.
	 * 
	 * @param _time
	 *            the time
	 * @param format
	 *            the format
	 * 
	 * @return the string
	 */
	public static String formaterTime(final Object _time, final String format) {
		String value = "";
		try {
			if (_time != null && StringUtils.isNotBlank(_time.toString())) {
				final Time time = (Time) _time;
				if (time.getTime() != -3599000) {
					final Date date = new Date(time.getTime());
					final SimpleDateFormat sdf = new SimpleDateFormat("HH:mm");
					value = sdf.format(date);
				}
			}
		} catch (final Exception e) {
			LOG.error("Erreur lors du formatage d'un Time", e);
		}
		return value;
	}

	/**
	 * Formater libelle.
	 * 
	 * @param _code
	 *            the _code
	 * @param nomTable
	 *            the nom table
	 * 
	 * @return the string
	 */
	public static String formaterLibelle(final String idExt, final Object _code, final String nomTable) {
		String value = "";
		try {
			value = CodeLibelle.lireLibelle(idExt, nomTable, (String) _code);
		} catch (final Exception e) {
			LOG.error("erreur lors du formatage d'un libelle", e);
		}
		return value;
	}

	/**
	 * Formater nombre.
	 * 
	 * @param _nombre
	 *            the _nombre
	 * @param format
	 *            the format
	 * 
	 * @return the string
	 */
	public static String formaterNombre(final Object _nombre, final String format) {
		String value = "";
		try {
			value = _nombre.toString();
		} catch (final Exception e) {
			LOG.error("erreur lors du formatage d'un nombre", e);
		}
		return value;
	}

	/**
	 * Renvoie true si une date est valorisée.
	 * 
	 * @param _date
	 *            the _date
	 * 
	 * @return true, if est saisie
	 */
	public static boolean estSaisie(final Date _date) {
		boolean res = false;
		if (_date != null) {
			try {
				final GregorianCalendar cal = new GregorianCalendar();
				cal.setTime(_date);
				res = cal.get(Calendar.YEAR) != 1970 || cal.get(Calendar.MONTH) != 0 || (cal.get(Calendar.DAY_OF_MONTH) != 1);
			} catch (final Exception e) {
				LOG.error("erreur lors de la vérification d'une date", e);
			}
		}
		return res;
	}

	/**
	 * Renvoie true si un champ type "site web" est valorisé.
	 * 
	 * @param siteWeb
	 *            the _site web
	 * 
	 * @return true, if est saisi
	 */
	public static boolean estSaisi(final String siteWeb) {
		return StringUtils.isNotEmpty(siteWeb) && !"http://".equals(siteWeb);
	}

	/**
	 * Formater decimal.
	 * 
	 * @param _nombre
	 *            the _nombre
	 * @param format
	 *            the format
	 * 
	 * @return the string
	 */
	public static String formaterDecimal(final Object _nombre, final String format) {
		String value = StringUtils.EMPTY;
		if (_nombre != null) {
			value = _nombre.toString();
		}
		return value;
	}

	/**
	 * Parser date.
	 * 
	 * @param valeur
	 *            the valeur
	 * 
	 * @return the date
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static Date parserDate(final String valeur) throws Exception {
		if (valeur.length() > 0) {
			final SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy");
			final GregorianCalendar cal = new GregorianCalendar();
			//Force le parser à générer une erreur en cas de date invalide
			dateFormat.setLenient(false);
			cal.setTime(dateFormat.parse(valeur));
			cal.set(Calendar.HOUR_OF_DAY, 0);
			cal.set(Calendar.MINUTE, 0);
			cal.set(Calendar.SECOND, 0);
			//transformation en date SQL
			return new Date(cal.getTimeInMillis());
		} else {
			return new Date(0);
		}
	}

	/**
	 * Renvoyer date jour.
	 * 
	 * @param valeur
	 *            the valeur
	 * @param plus
	 *            the plus
	 * 
	 * @return the date
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static Date renvoyerDateJour(final String valeur, final boolean plus) throws Exception {
		Date res = new Date(0);
		int jour = 0;
		if (valeur.length() > 0) {
			try {
				jour = Integer.parseInt(valeur);
			} catch (final Exception e) {
				return res;
			}
			final Date dateAujourdhui = new Date(System.currentTimeMillis());
			final GregorianCalendar cal = new GregorianCalendar();
			cal.setTime(dateAujourdhui);
			if (plus) {
				cal.set(Calendar.DAY_OF_YEAR, cal.get(Calendar.DAY_OF_YEAR) + jour);
			} else {
				cal.set(Calendar.DAY_OF_YEAR, cal.get(Calendar.DAY_OF_YEAR) - jour);
			}
			res = new Date(cal.getTimeInMillis());
		}
		return res;
	}
}
