/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.core;

import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Map;
import java.util.UUID;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.AsyncAppender;
import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.rolling.DefaultTimeBasedFileNamingAndTriggeringPolicy;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.rolling.TimeBasedRollingPolicy;

import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.PropertyConfigurer;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.logging.LogHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;

public class LogAppFilter implements Filter {

	public static final String FILE_NAME_REQUETES = "requetes";

	private static final String LOGAPP_APPENDER_NAME = "LOGAPP";

	private static final Logger LOGGER = (Logger) LoggerFactory.getLogger(LogAppFilter.class);

	private static final String PROP_LOGAPP_FILTER = "logapp.filter";

	private AsyncAppender asyncAppender = null;

	@Override
	public void destroy() {}

	@Override
	public void doFilter(final ServletRequest req, final ServletResponse res, final FilterChain filter) throws IOException, ServletException {
		if (isActivated()) {
			final long timerStart = System.currentTimeMillis();
			final String uuid = UUID.randomUUID().toString();
			final HttpServletRequest request = (HttpServletRequest) req;
			// le code su site
			String codeSite = "";
			// le user-agent du client
			String userAgent = "";
			// le host referer
			String hostReferer = "";
			// l'adresse IP du client
			String remoteAddr = "";
			// la session utilisateur
			SessionUtilisateur sessionUtilisateur = null;
			// l'url
			final String url = request.getRequestURI();
			// la requete
			String requete = request.getQueryString();
			try {
				final InfosSite infosSite = Site.renvoyerItemSiteParHost(request.getServerName());
				codeSite = infosSite.getAlias();
				// le referer
				if (request.getHeader("referer") != null && request.getHeader("referer").length() > 0) {
					hostReferer = request.getHeader("referer");
					if (hostReferer != null) {
						if (requete == null) {
							if (hostReferer.indexOf("?") != -1) {
								requete = hostReferer.substring(hostReferer.indexOf("?") + 1) + "(referer)";
							}
						}
						if (hostReferer.indexOf("//") != -1) {
							hostReferer = hostReferer.substring(hostReferer.indexOf("//") + 2);
							if (hostReferer.indexOf("/") != -1) {
								hostReferer = hostReferer.substring(0, hostReferer.indexOf("/"));
							}
							if (hostReferer.indexOf(":") != -1) {
								hostReferer = hostReferer.substring(0, hostReferer.indexOf(":"));
							}
						}
					}
				}
				// le user-agent
				if (request.getHeader("user-agent") != null && request.getHeader("user-agent").length() > 0) {
					userAgent = request.getHeader("user-agent");
				}
				// l'adresse ip
				remoteAddr = request.getRemoteAddr();
				// /!\ avec un load balancer, la bonne IP client est dans le header
				if (request.getHeader("REMOTE_ADDR") != null && request.getHeader("REMOTE_ADDR").length() > 0) {
					remoteAddr = request.getHeader("REMOTE_ADDR");
				}
				final HttpSession _session = request.getSession(false);
				if (_session != null) {
					sessionUtilisateur = (SessionUtilisateur) _session.getAttribute(SessionUtilisateur.CLE_SESSION_UTILISATEUR_DANS_SESSION_HTTP);
				}
				logTraceActiviteWeb(sessionUtilisateur, uuid, 0, hostReferer, remoteAddr, userAgent, codeSite, url, requete, null);
			} finally {
				final Map<String, String> postParams = null;
				/*if(request.getAttribute("POST_PARAMS") != null) {
					postParams = (Map<String, String>)request.getAttribute("POST_PARAMS");
				}*/
				logTraceActiviteWeb(sessionUtilisateur, uuid, timerStart, hostReferer, remoteAddr, userAgent, codeSite, url, requete, postParams);
			}
		}
		filter.doFilter(req, res);
	}

	/**
	 * Methode qui trace l'activite web a chaque appel
	 * 
	 * @param ctx
	 * @param uuid
	 * @param hostReferer
	 * @param ip
	 * @param userAgent
	 */
	private void logTraceActiviteWeb(final SessionUtilisateur sessionUtilisateur, final String uuid, final long timerStart, final String hostReferer, final String ip,
		final String userAgent, final String codeSite, final String url, final String requete, final Map<String, String> paramsPost) {
		try {
			final long timerNow = System.currentTimeMillis();
			Hashtable<String, Object> infosSession = null;
			if (sessionUtilisateur != null) {
				infosSession = sessionUtilisateur.getInfos();
			}
			/*String formattedParamsPost = "";
			if(paramsPost != null) {
				for (String keyParamPost : paramsPost.keySet()) {
					if(!keyParamPost.startsWith("#FORMAT_")) {
						formattedParamsPost += keyParamPost + "="+paramsPost.get(keyParamPost) + " | ";
					}
				}
			}*/
			String logLine = uuid + ";" + timerNow + ";" + (timerStart > 0 ? (timerNow - timerStart) : "") + ";" + ip + ";";
			logLine += (infosSession != null ? (String) infosSession.get(SessionUtilisateur.KSESSION) : "") + ";" + (infosSession != null && infosSession.get(SessionUtilisateur.CODE) != null ? (String) infosSession.get(SessionUtilisateur.CODE) : "") + ";" + codeSite + ";" + ";" //service
				+ ";" //rubrique
				+ url + ";" + (requete != null ? requete : "") + ";";
			logLine += hostReferer + ";" + "\"" + userAgent + "\";"; // guillemets car on a des ";" dans les user-agents
			//logLine += "\"" + formattedParamsPost+"\";"; // guillemets car on peut avoir des ";" dans les parametres POST
			LOGGER.trace(logLine);
		} catch (final Exception e) {
			LOGGER.error("Impossible de tracer l'activite web : ", e);
		}
	}

	public boolean isActivated() {
		if (!"1".equals(PropertyHelper.getCoreProperty(PROP_LOGAPP_FILTER))) {
			// on supprime l'appender précédent
			if (asyncAppender != null) {
				LOGGER.detachAppender(asyncAppender);
				asyncAppender.stop();
				asyncAppender = null;
			}
			return Boolean.FALSE;
		}
		// on initialise l'appender
		if (asyncAppender == null) {
			asyncAppender = new AsyncAppender();
			asyncAppender.setDiscardingThreshold(0);
			asyncAppender.setContext(LOGGER.getLoggerContext());
			asyncAppender.setName(LOGAPP_APPENDER_NAME);
			final RollingFileAppender<ILoggingEvent> fileAppender = new RollingFileAppender<ILoggingEvent>();
			fileAppender.setContext(LOGGER.getLoggerContext());
			final File folderApp = new File(WebAppUtil.getLogsPath() + File.separator + "app");
			if (!folderApp.exists()) {
				folderApp.mkdir();
			}
			// Pattern de l'encoder
			final PatternLayoutEncoder encoder = new PatternLayoutEncoder();
			encoder.setPattern("%d{HH:mm:ss,SSS};%m %n");
			encoder.setContext(LOGGER.getLoggerContext());
			// Système de trigger
			final DefaultTimeBasedFileNamingAndTriggeringPolicy<ILoggingEvent> timeTrigger = new DefaultTimeBasedFileNamingAndTriggeringPolicy<ILoggingEvent>();
			timeTrigger.setContext(LOGGER.getLoggerContext());
			// Système de roulement des fichiers (en fonction du jour)
			final TimeBasedRollingPolicy<ILoggingEvent> policy = new TimeBasedRollingPolicy<ILoggingEvent>();
			policy.setContext(LOGGER.getLoggerContext());
			// nom du fichier archivé
			policy.setFileNamePattern(folderApp.getAbsolutePath() + File.separator + FILE_NAME_REQUETES + LogHelper.LOG_EXTENSION + "." + PropertyConfigurer.getProperty(LogHelper.PROP_LOG_FILENAME_PATTERN));
			policy.setTimeBasedFileNamingAndTriggeringPolicy(timeTrigger);
			// maximum d'archive
			if (StringUtils.isNotEmpty(PropertyConfigurer.getProperty(LogHelper.PROP_LOG_MAX_HISTORY))) {
				try {
					policy.setMaxHistory(Integer.parseInt(PropertyConfigurer.getProperty(LogHelper.PROP_LOG_MAX_HISTORY)));
				} catch (final Exception e) {
					LOGGER.error("La valeur de la propriétée \"" + LogHelper.PROP_LOG_MAX_HISTORY + "\" n'est pas numérique");
				}
			}
			// Application des triggers
			timeTrigger.setTimeBasedRollingPolicy(policy);
			fileAppender.setRollingPolicy(policy);
			fileAppender.setEncoder(encoder);
			fileAppender.setFile(folderApp.getAbsolutePath() + File.separator + FILE_NAME_REQUETES + LogHelper.LOG_EXTENSION);
			fileAppender.setAppend(true);
			policy.setParent(fileAppender);
			policy.start();
			encoder.start();
			fileAppender.start();
			asyncAppender.addAppender(fileAppender);
			//On augmente la taille du buffer pour éviter les blocages
			asyncAppender.setQueueSize(1500);
			asyncAppender.start();
			LOGGER.addAppender(asyncAppender);
			LOGGER.setLevel(Level.TRACE);
			LOGGER.setAdditive(false);
		}
		return Boolean.TRUE;
	}

	@Override
	public void init(final FilterConfig arg0) throws ServletException {
		// TODO Auto-generated method stub
	}
}
