/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.database;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.lang.CharEncoding;
import com.kportal.core.config.PropertyHelper;

/**
 * Un gestionnaire de pool minimum.
 */
public class BasicPoolMgr {

	private static final Logger LOGGER = LoggerFactory.getLogger(BasicPoolMgr.class);

	/** The Constant QUERY_TEST. */
	final static String QUERY_TEST = "SELECT 1";

	/** The liste connections. */
	private static Vector<InfosConnexion> listeConnections = new Vector<>();

	/** The i wait. */
	private static int iWait = 0;

	/**
	 * Tester connexion.
	 * 
	 * @param conn
	 *            the conn
	 * 
	 * @return true, if successful
	 */
	public static boolean testerConnexion(final Connection conn) {
		final boolean res = false;
		try {
			if (conn.isClosed()) {
				return false;
			}
			LOGGER.debug("BasicPoolMgr test connexion ");
			final Statement stmt = conn.createStatement();
			final ResultSet rs = stmt.executeQuery(BasicPoolMgr.QUERY_TEST);
			if (rs.next()) {
				LOGGER.debug("BasicPoolMgr test connexion OK");
				return true;
			}
			return false;
		} catch (final Exception e) {
			LOGGER.warn("BasicPoolMgr test connexion KO");
		}
		return res;
	}

	/**
	 * Dump.
	 */
	public static void dump() {
		LOGGER.debug("BasicPoolMgr dump");
		InfosConnexion infos = new InfosConnexion();
		int nbDispo = 0, nbLongues = 0;
		/* 1ère passe sur les connections libres */
		int iConnexion = 0;
		final ArrayList<InfosConnexion> lstConnexions = getCurrentConnections();
		while (iConnexion < lstConnexions.size()) {
			infos = (lstConnexions.get(iConnexion));
			if (infos.estDisponible()) {
				nbDispo++;
			}
			if (infos.estLongue()) {
				nbLongues++;
			}
			iConnexion++;
		}
		LOGGER.debug("  > " + nbDispo + " connection(s) libre(s)");
		LOGGER.debug("  > " + nbLongues + " connection(s) longue(s)");
	}

	/**
	 * Gets the nombre connexions disponibles.
	 * 
	 * @return the nombre connexions disponibles
	 */
	public static int getNombreConnexionsDisponibles() {
		int nbDispo = 0;
		final ArrayList<InfosConnexion> lstConnexions = getCurrentConnections();
		for (final Object element : lstConnexions) {
			if (((InfosConnexion) element).estDisponible()) {
				++nbDispo;
			}
		}
		return nbDispo;
	}

	/**
	 * Gets the nombre connexions ouvertes.
	 * 
	 * @return the nombre connexions ouvertes
	 */
	public static int getNombreConnexionsOuvertes() {
		return getCurrentConnections().size();
	}

	/* Récupération d'une connexion dans le pool */
	/**
	 * Gets the connection.
	 * 
	 * @param connectionLongue
	 *            the connection longue
	 * 
	 * @return the connection
	 */
	public static Connection getConnection(final boolean connectionLongue) {
		LOGGER.debug("Recherche connexion");
		InfosConnexion infos = new InfosConnexion();
		/*
		 * 1ère passe sur les connections libres
		 * Si on trouve une connexion invalide (hors timeout), on la supprime
		 */
		int iConnexion = 0;
		boolean fTrouve = false;
		synchronized (listeConnections) {
			int nbConnexions = listeConnections.size();
			final List<Integer> listeConnexionsHS = new ArrayList<Integer>(nbConnexions);
			// test de la connexion et suppression des connexions invalides trouvees
			while (!fTrouve && (iConnexion < nbConnexions)) {
				infos = (listeConnections.get(iConnexion));
				if (infos.estDisponible()) {
					calculeWait();
					// si libre depuis - de iWait minutes et valide, on recupere la connexion
					if ((infos.getTsLiberation() > System.currentTimeMillis() - iWait) && testerConnexion(infos.getConnection())) {
						fTrouve = true;
						// sinon on la supprime
					} else {
						listeConnexionsHS.add(new Integer(iConnexion));
					}
				}
				if (!fTrouve) {
					++iConnexion;
				} else {
					LOGGER.debug("Allocation connexion libre");
				}
			}
			// suppression : parcours inverse pour eviter decalage indice
			for (int i = listeConnexionsHS.size() - 1; i >= 0; --i) {
				listeConnections.remove(listeConnexionsHS.get(i).intValue());
				LOGGER.info("Suppression d'une connexion dépassée > " + iWait/60000 +" m");
			}
			nbConnexions = listeConnections.size();
			listeConnexionsHS.clear();
			/* 2eme passe sur les connections prises
			 * si on trouve une connexion invalide, on la supprime
			 * si on trouve une connexion valide mais non liberee, on la recupere
			 */
			if (!fTrouve) {
				iConnexion = 0;
				while (!fTrouve && (iConnexion < nbConnexions)) {
					infos = (listeConnections.get(iConnexion));
					if (!infos.estDisponible()) {
						// Si allouée depuis + de iWait minutes, on recupere la connexion car il s'agit d'un plantage
						// sauf pour les transactions longues (batch, mailing, ldap)
						if (!infos.estLongue()) {
							calculeWait();
							if (infos.getTsAllocation() < System.currentTimeMillis() - iWait) {
								if (testerConnexion(infos.getConnection())) {
									fTrouve = true;
								} else {
									// La connexion est plantée
									listeConnexionsHS.add(new Integer(iConnexion));
								}
							}
						} else {
							// 8 heures (28800000) pour les transations longues (batch, mailing, ldap)
							if (infos.getTsAllocation() < System.currentTimeMillis() - 28800000) {
								if (testerConnexion(infos.getConnection())) {
									fTrouve = true;
								} else {
									// La connexion est plantée
									listeConnexionsHS.add(new Integer(iConnexion));
								}
							}
						}
					}
					if (!fTrouve) {
						iConnexion++;
					} else {
						LOGGER.debug("Allocation connexion non libérée depuis " + (System.currentTimeMillis() - infos.getTsAllocation()) + " s");
					}
				}
			}
			// suppression connexion prise et invalide
			for (int i = listeConnexionsHS.size() - 1; i >= 0; --i) {
				InfosConnexion infosC = listeConnections.get(listeConnexionsHS.get(i).intValue());
				listeConnections.remove(listeConnexionsHS.get(i).intValue());
				LOGGER.info("Suppression d'une connexion bloquée > " + (infosC.estLongue() ? 28800000/3600000+ " h" : iWait/60000 +" m"));
			}
			// marquage de la connexion comme prise
			if (fTrouve) {
				infos.setLongue(connectionLongue);
				infos.setDisponible(false);
				infos.setTsAllocation(System.currentTimeMillis());
			}
		}
		if (!fTrouve) {
			/* Allocation d'une nouvelle connexion */
			try {
				infos = new InfosConnexion();
				// Load and register driver
				final String driver = PropertyHelper.getCoreProperty("datastore.default.driver");
				if (driver != null) {
					Class.forName(driver);
				}
				final Properties props = new Properties();
				final String user = PropertyHelper.getCoreProperty("datastore.default.user");
				if (user != null) {
					props.put("user", user);
				}
				final String password = PropertyHelper.getCoreProperty("datastore.default.password");
				if (password != null) {
					props.put("password", password);
				}
				String truncation = PropertyHelper.getCoreProperty("datastore.default.truncation");
				if (truncation == null) {
					truncation = "false";
				}
				props.put("characterEncoding", CharEncoding.DEFAULT.toLowerCase().replace("-", ""));
				//props.put("useOldUTF8Behavior","true");
				props.put("jdbcCompliantTruncation", truncation);
				props.put("dontTrackOpenResources", "true");
				props.put("connectTimeout", "5000");
				props.put("zeroDateTimeBehavior", "convertToNull");
				//props.put("enableQueryTimeouts","false");
				//props.put("queryTimeoutKillsConnection","false");
				//props.put("socketTimeout", "5000");
				// get a connection
				final String url = PropertyHelper.getCoreProperty("datastore.default.url");
				if (url != null) {
					//RP20091112 surcharge de la connection mysql avec un proxy pour nettoyer les ressources de type statement et resultset
					final Connection connection = SOSJDBCProxy.createJdbcProxy(DriverManager.getConnection(url, props));
					infos.setConnection(connection);
					infos.setLongue(connectionLongue);
					infos.setDisponible(false);
					infos.setTsAllocation(System.currentTimeMillis());
					synchronized (listeConnections) {
						listeConnections.add(infos);
						LOGGER.debug("BasicPoolMgr nouvelle connexion : " + listeConnections.size() + " connexions stockées.");
					}
				}
			} catch (final Exception e) {
				LOGGER.error(e.getMessage(), e);
				return null;
			}
		}
		infos.setStackTrace(Thread.currentThread().getStackTrace());
		return infos.getConnection();
	}

	/**
	 * Release connection.
	 * 
	 * @param _connection
	 *            the _connection
	 */
	public static void releaseConnection(final Connection _connection) {
		LOGGER.debug("BasicPoolMgr Remise d'une connexion dans le pool");
		//RP20091112 surcharge de la connection mysql avec un proxy pour nettoyer les ressources de type statement et resultset
		SOSJDBCProxy.cleanConnection(_connection);
		synchronized (listeConnections) {
			int iConnexion = 0;
			boolean fTrouve = false;
			while ((fTrouve == false) && (iConnexion < listeConnections.size())) {
				final InfosConnexion infos = (listeConnections.get(iConnexion));
				if (infos.getConnection().equals(_connection)) {
					if (infos.estDisponible()) {
						LOGGER.warn("BasicPoolMgr : la connection est deja liberee");
					}
					infos.setTsLiberation(System.currentTimeMillis());
					infos.setDisponible(true);
					infos.setStackTrace(null);
					fTrouve = true;
				}
				if (fTrouve == false) {
					iConnexion++;
				}
			}
		}
	}

	/**
	 * Calcule wait.
	 */
	private static void calculeWait() {
		// AM : on rend le timeout paramétrable
		if (iWait == 0) {
			String wait_timeout = PropertyHelper.getCoreProperty("datastore.default.wait_timeout");
			if (wait_timeout == null) {
				wait_timeout = "600000";
			}
			//on capture au cas où ce ne serait pas un chiffre qui est dans le properties
			try {
				iWait = Integer.parseInt(wait_timeout);
				if (iWait == 0) {
					iWait = 600000;
				}
			} catch (final Exception e) {
				iWait = 600000;
			}
		}
	}

	/**
	 * Renvoie une copie de la liste des contextes (accès synchronized en lecture de la liste originale)
	 * 
	 * @return
	 */
	public static ArrayList<InfosConnexion> getCurrentConnections() {
		synchronized (listeConnections) {
			final ArrayList<InfosConnexion> copie = new ArrayList<InfosConnexion>(listeConnections);
			return copie;
		}
	}
}