/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JDBCUtils {

	/** The Constant LOG. */
	private static final Logger LOGGER = LoggerFactory.getLogger(JDBCUtils.class);

	protected static Logger tempLogger = null;

	public static boolean existe(final Connection connection, final String nomTable) throws SQLException {
		boolean existe;
		final java.sql.DatabaseMetaData dmd = connection.getMetaData();
		final ResultSet tables = dmd.getTables(connection.getCatalog(), null, nomTable, null);
		existe = tables.next();
		tables.close();
		return existe;
	}

	/**
	 * Ajouter index.
	 * 
	 * @param connection
	 *            connection
	 * @param table
	 *            the table
	 * @param champ
	 *            the champ
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void ajouterIndex(final Connection connection, final String table, final String champ) throws Exception {
		PreparedStatement stmt = null;
		try {
			final String items[] = champ.split("\\*", -2);
			final String nomChamp = items[0];
			String lg = "";
			if (items.length == 2) {
				lg = "(" + items[1] + ")";
			}
			final String commande = "ALTER TABLE " + table + " ADD INDEX IDX_" + nomChamp + "(" + nomChamp + lg + ")";
			stmt = connection.prepareStatement(commande);
			stmt.executeUpdate();
			getLogger().info(commande);
		} finally {
			if (stmt != null) {
				stmt.close();
			}
		}
	}

	/**
	 * Lire indexs.
	 * 
	 * @param connection
	 *            connection
	 * @param table
	 *            the table
	 * 
	 * @return the list
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static List<String> lireIndexs(final Connection connection, final String table) throws Exception {
		final ArrayList<String> listIndexs = new ArrayList<String>();
		PreparedStatement stmt = null;
		ResultSet rs = null;
		final String commande = "SHOW INDEX FROM " + table;
		try {
			stmt = connection.prepareStatement(commande);
			rs = stmt.executeQuery();
			while (rs.next()) {
				listIndexs.add(rs.getString("Column_name").toUpperCase());
			}
		} finally {
			if (stmt != null) {
				stmt.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
		return listIndexs;
	}

	/**
	 * Mettre a jour indexs table.
	 * 
	 * @param connection
	 *            connection
	 * @param table
	 *            the table
	 * @param listeTotaleIndexs
	 *            the liste totale indexs
	 */
	public static void mettreAJourIndexsTable(final Connection connection, final String table, final List<String> listeTotaleIndexs) throws Exception {
		// Lecture des indexs courants
		final List<String> listeIndexsLus = lireIndexs(connection, table);
		// Boucle sur tous les indexs à mettre en place
		for (String indexCourant : listeTotaleIndexs) {
			final String items[] = indexCourant.split("\\*", -2);
			final String nomChamp = items[0];
			if ("DIFFUSION_PUBLIC_VISE".equals(nomChamp) || "DIFFUSION_PUBLIC_VISE_RESTRICTION".equals(nomChamp) || "META_DIFFUSION_PUBLIC_VISE".equals(nomChamp) || "META_DIFFUSION_PUBLIC_VISE_RESTRICTION".equals(nomChamp)) {
				final boolean misAjour = mettreAJourChamp(connection, table, nomChamp);
				if (misAjour && listeIndexsLus.contains(nomChamp)) {
					listeIndexsLus.remove(nomChamp);
				}
			}
			if (!listeIndexsLus.contains(nomChamp)) {
				ajouterIndex(connection, table, indexCourant);
			}
		}
	}

	/**
	 * Mettre a jour champ.
	 * 
	 * @param connexion
	 *            the connexion
	 * @param table
	 *            the table
	 * @param champ
	 *            the champ
	 * 
	 * @return true, if successful
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static boolean mettreAJourChamp(final Connection connexion, final String table, final String champ) throws Exception {
		PreparedStatement stmt = null;
		ResultSet rs = null;
		final String commande = "DESC " + table;
		boolean misAJour = false;
		try {
			stmt = connexion.prepareStatement(commande);
			rs = stmt.executeQuery();
			while (rs.next()) {
				final String nomColonne = rs.getString("Field");
				if (nomColonne.equalsIgnoreCase(champ)) {
					if (!rs.getString("Type").equalsIgnoreCase("text")) {
						if (!rs.getString("Key").equalsIgnoreCase("")) {
							final String nomIndex = lireNomIndex(connexion, table, champ);
							supprimerIndex(connexion, table, nomIndex);
						}
						changerTypeChamp(connexion, table, champ);
						misAJour = true;
					}
				}
			}
		} finally {
			if (stmt != null) {
				stmt.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
		return misAJour;
	}

	/**
	 * Changer type champ.
	 * 
	 * @param connexion
	 *            the connexion
	 * @param table
	 *            the table
	 * @param nomChamp
	 *            the nom champ
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void changerTypeChamp(final Connection connexion, final String table, final String nomChamp) throws Exception {
		PreparedStatement stmt = null;
		try {
			final String commande = "ALTER TABLE " + table + " CHANGE " + nomChamp + " " + nomChamp + " TEXT";
			stmt = connexion.prepareStatement(commande);
			stmt.executeUpdate();
			getLogger().info(commande);
		} finally {
			if (stmt != null) {
				stmt.close();
			}
		}
	}

	/**
	 * Supprimer index.
	 * 
	 * @param connexion
	 *            the ctx
	 * @param table
	 *            the table
	 * @param nomIndex
	 *            the nom index
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void supprimerIndex(final Connection connexion, final String table, final String nomIndex) throws Exception {
		PreparedStatement stmt = null;
		try {
			final String commande = "ALTER TABLE " + table + " DROP INDEX " + nomIndex;
			stmt = connexion.prepareStatement(commande);
			stmt.executeUpdate();
			getLogger().info(commande);
		} finally {
			if (stmt != null) {
				stmt.close();
			}
		}
	}

	/**
	 * Lire nom index.
	 * 
	 * @param connexion
	 *            the ctx
	 * @param table
	 *            the table
	 * @param champ
	 *            the champ
	 * 
	 * @return the string
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static String lireNomIndex(final Connection connexion, final String table, final String champ) throws Exception {
		PreparedStatement stmt = null;
		ResultSet rs = null;
		final String commande = "SHOW INDEX FROM " + table;
		String res = "";
		try {
			stmt = connexion.prepareStatement(commande);
			rs = stmt.executeQuery();
			while (rs.next()) {
				final String nomColonne = rs.getString("Column_name");
				if (nomColonne.equalsIgnoreCase(champ)) {
					res = rs.getString("Key_name");
				}
			}
		} finally {
			if (stmt != null) {
				stmt.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
		return res;
	}

	public static Logger getLogger() {
		return ((tempLogger != null) ? tempLogger : LOGGER);
	}

	public static void setTempLogger(final Logger _tempLogger) {
		tempLogger = _tempLogger;
	}

	/**
	 * Met à jour l'auto-increment en ajoutant à la valeur courante le nombre de lignes à insérer.
	 * @param connection
	 * @param table : Nom de la table pour laquelle on souhaite la valeur de l'auto-increment
	 * @param nbRow : nombre de ligne à insérer
	 * @return : la nouvelle valeur de l'autoincrément-1 (ie, la dernière valeur possible qui a été réservée)
	 * @throws Exception
	 */
	public static Long getNextAutoIncrement(final Connection connection, final String table, final int nbRow) throws Exception {
		PreparedStatement stmt = null;
		ResultSet rs = null;
		String commande = String.format("SELECT AUTO_INCREMENT FROM information_schema.TABLES WHERE TABLE_NAME = '%s' AND TABLE_SCHEMA = '%s'", table, connection.getCatalog());
		Long res = 0L;
		try {
			stmt = connection.prepareStatement(commande);
			rs = stmt.executeQuery();
			rs.next();
			res = rs.getLong(1);
			if (nbRow > 0) {
				commande = String.format("ALTER TABLE `%s` AUTO_INCREMENT = %d", table, res + nbRow);
				stmt.close();
				stmt = connection.prepareStatement(commande);
				stmt.execute();
			}
		} finally {
			if (stmt != null) {
				stmt.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
		return res + nbRow -1;
	}
}
