/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.database;

import java.sql.Connection;
import java.util.Hashtable;
import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.AbstractProcessusBean;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.exception.ErreurTechnique;
import com.jsbsoft.jtf.exception.JsbException;
import com.univ.multisites.InfosSite;
import com.univ.utils.ContexteUtil;

/**
 * Super-classe des processus.
 */
public abstract class ProcessusBean extends AbstractProcessusBean implements OMContext {

	/** The connection. */
	private Connection connection = null;

	/** The action. */
	protected String action = null;

	/** The infos site. */
	private InfosSite infosSite = null;

	/** The secure. */
	private boolean secure = false;

	/** The datas. */
	private Hashtable<String, Object> datas = null;

	private static final Logger LOG = LoggerFactory.getLogger(ProcessusBean.class);

	/**
	 * Interface processus d'aide à la codification.
	 *
	 * @param infoBean
	 *            com.jsbsoft.jtf.core.InfoBean
	 */
	public ProcessusBean(final InfoBean infoBean) {
		super(infoBean);
		datas = new Hashtable<String, Object>();
	}

	/**
	 * @see com.jsbsoft.jtf.database.OMContext#getDatas()
	 */
	@Override
	public Hashtable<String, Object> getDatas() {
		return datas;
	}

	/**
	 * Valorisation du DataStore associé au processus.
	 *
	 * @return the connection
	 */
	@Override
	public Connection getConnection() {
		return connection;
	}

	/**
	 * Valorisation du DataStore associBeané au processus.
	 *
	 * @return the locale
	 */
	@Override
	public Locale getLocale() {
		return ContexteUtil.getContexteUniv().getLocale();
	}

	/**
	 * Valorisation du DataStore associé au processus.
	 *
	 * @throws JsbException
	 *             the jsb exception
	 */
	@Override
	protected void initConnection() throws JsbException {
		if (connection == null) {
			try {
				// JSS 20040628 - gestion des connections longues
				connection = BasicPoolMgr.getConnection(false);
			} catch (final Exception e) {
				throw new ErreurTechnique(ErreurTechnique.TYPERR_SGBD, e.getMessage());
			}
		}
	}

	/**
	 * Valorisation du DataStore associé au processus.
	 *
	 * @throws JsbException
	 *             the jsb exception
	 */
	@Override
	protected void releaseConnection() throws JsbException {
		if (connection != null) {
			try {
				//connection.close();
				BasicPoolMgr.releaseConnection(connection);
				connection = null;
			} catch (final Exception e) {
				throw new ErreurTechnique(ErreurTechnique.TYPERR_SGBD, e.getMessage());
			}
		}
	}

	/**
	 * @see com.jsbsoft.jtf.core.AbstractProcessusBean#traiterAbandon()
	 */
	@Override
	public void traiterAbandon() {
		try {
			releaseConnection();
		} catch (final Exception e) {
			LOG.error("erreur lors de la libération de la connexion à la base", e);
		}
	}

	/**
	 * @see com.jsbsoft.jtf.database.OMContext#setInfosSite(com.univ.multisites.InfosSite)
	 */
	@Override
	public void setInfosSite(final InfosSite infosSite) {
		this.infosSite = infosSite;
	}

	/**
	 * Renvoie l'InfosSite correspondant au site courant.
	 *
	 * @return the infos site
	 */
	@Override
	public InfosSite getInfosSite() {
		return infosSite;
	}

	/**
	 * @see com.jsbsoft.jtf.database.OMContext#setSecure(boolean)
	 */
	@Override
	public void setSecure(final boolean secure) {
		this.secure = secure;
	}

	/**
	 * Renvoie true si on est en mode sécurisé (https).
	 *
	 * @return true, if checks if is secure
	 */
	@Override
	public boolean isSecure() {
		return secure;
	}
}
