/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Created on 22 févr. 2004
 *
 * To change the template for this generated file go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
package com.jsbsoft.jtf.database;

import java.text.SimpleDateFormat;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.email.JSBMailbox;
import com.jsbsoft.jtf.webutils.FormateurHTML;
import com.kportal.core.config.PropertyHelper;

/**
 *
 * Gestion des dumps de requete
 *
 * - détection des boucles - affichage sur exception
 *
 * Les requêtes sont identifiées par un idRequete au niveau des contextes
 *
 * Attention, toutes les requetes ne sont pas actuellement mémorisées car cela demandait d'internir à trop d'endroits (en particulier dans les processus).
 *
 * La variable idRequete est alors vide au niveau du contexte
 *
 * @author KOSMOS
 */
public class RequeteMgr {

	/** The id requete. */
	private static String idRequete = String.valueOf(System.currentTimeMillis());

	// Stockage des requetes
	/** The events. */
	private static TreeMap<String, EtatRequete> events = new TreeMap<String, EtatRequete>();

	private static final Logger LOG = LoggerFactory.getLogger(RequeteMgr.class);

	/**
	 * Creer requete.
	 *
	 * @return the string
	 */
	public static synchronized String creerRequete() {
		// Increment de l'identifiant de requete
		idRequete = String.valueOf(Long.parseLong(idRequete) + 1);
		events.put(idRequete, new EtatRequete());
		/*
		 *  On supprime les requetes terminees depuis 30 s.
		 *  On logge les requêtes durant plus de 1 minute
		 */
		final Vector<String> aSupprimer = new Vector<String>();
		final Vector<String> aLogger = new Vector<String>();
		final long maintenant = System.currentTimeMillis();
		final Iterator<String> iter = events.keySet().iterator();
		while (iter.hasNext()) {
			final String id = iter.next();
			final EtatRequete requete = events.get(id);
			//Log.debug("delai " + id + " : " + (maintenant - requete.getTimestampDebut()));
			if (requete.isTerminee()) {
				if (maintenant - requete.getTimestampDebut() > 30000) {
					aSupprimer.add(id);
				}
			} else {
				if (maintenant - requete.getTimestampDebut() > 60000) {
					aLogger.add(id);
					aSupprimer.add(id);
				}
			}
		}
		/* Détection de boucles :
		 * evènements à envoyer au webmaster
		 */
		if ("1".equals(PropertyHelper.getCoreProperty("dump.detection_boucle")) && !aLogger.isEmpty()) {
			String dump = "";
			final Enumeration<String> e1 = aLogger.elements();
			while (e1.hasMoreElements()) {
				final String id = e1.nextElement();
				//Log.debug("log requete " + id);
				dump += "\n***** DUMP REQUETE ************";
				dump += "\"" + dumpRequete(events.get(id));
				dump += "\n***** FIN DUMP ********************";
				LOG.info(dump);
			}
			try {
				//Initialisation du mailbox
				final JSBMailbox mailbox = new JSBMailbox(false);
				final String sujet = "[Requete non terminée ?] > A ANALYSER";
				mailbox.sendSystemMsg("debug@kosmos.fr", sujet, dump);
			} catch (final Exception e) {}
		}
		/* Evènements à supprimer */
		final Enumeration<String> e2 = aSupprimer.elements();
		while (e2.hasMoreElements()) {
			final String id = e2.nextElement();
			//Log.debug("suppression requete " + id);
			events.remove(id);
		}
		return idRequete;
	}

	/**
	 * Ajouter evenement.
	 *
	 * @param id
	 *            the id
	 * @param evenement
	 *            the evenement
	 */
	public static synchronized void ajouterEvenement(final String id, final String evenement) {
		//Log.debug(id + " " + evenement);
		final GregorianCalendar gCal = new GregorianCalendar();
		final SimpleDateFormat formateur = new SimpleDateFormat("hh:mm:ss:SSS");
		if ((id != null) && (id.length() > 0) && (events.get(id) != null)) {
			events.get(id).ajouterEvenement(formateur.format(gCal.getTime()) + "\n" + evenement);
		}
	}

	/**
	 * Terminer requete.
	 *
	 * @param id
	 *            the id
	 */
	public static synchronized void terminerRequete(final String id) {
		if ((id != null) && (id.length() > 0) && (events.get(id) != null)) {
			//((EtatRequete) events.get(id)).ajouterEvenement("fin");
			events.get(id).terminer();
		}
	}

	/**
	 * Affichage du dump instantané (appelé dans /adminsite/dump.jsp)
	 *
	 * @return the string
	 */
	public static synchronized String dump() {
		/* Durée moyenne des requêtes terminées */
		String res = "";
		long delai = 0;
		int nbRequetesTerminees = 0;
		Iterator<EtatRequete> iter = events.values().iterator();
		while (iter.hasNext()) {
			final EtatRequete requete = iter.next();
			if (requete.isTerminee()) {
				delai += requete.getTimestampFin() - requete.getTimestampDebut();
				nbRequetesTerminees++;
			}
		}
		if (nbRequetesTerminees > 0) {
			res += "<br /> Nb requetes OK (30 dernières sec)= " + nbRequetesTerminees;
			res += "<br /> Durée moyenne = " + (delai / nbRequetesTerminees) + " ms";
			res += "<br> Nombre de connexions ouvertes sur la base de données =" + BasicPoolMgr.getNombreConnexionsOuvertes();
			res += "<br> Nombre de connexions disponibles sur la base de données =" + BasicPoolMgr.getNombreConnexionsDisponibles();
		}
		/* Affichage des requetes suspendues */
		iter = events.values().iterator();
		res += "<font color=\"red\">";
		res += "<br /> Nb requêtes en cours (délai purge = 60 s.) = " + (events.size() - nbRequetesTerminees);
		while (iter.hasNext()) {
			final EtatRequete requete = iter.next();
			if (requete.isTerminee() == false) {
				res += "<br />-----REQUETE -------------";
				res += "<br />" + FormateurHTML.formaterEnHTML(dumpRequete(requete));
			}
		}
		res += "</font>";
		/* Affichage des requetes terminées */
		String requetesTerminees = "";
		iter = events.values().iterator();
		while (iter.hasNext()) {
			final EtatRequete requete = iter.next();
			if (requete.isTerminee()) {
				requetesTerminees = "<br />-----REQUETE -------------<br />" + FormateurHTML.formaterEnHTML(dumpRequete(requete)) + requetesTerminees;
			}
		}
		res += requetesTerminees;
		return res;
	}

	/**
	 * Affichage des des requêtes commencées depuis - de 3 s et non terminées Utilisé pour diagnostiquer les exceptions.
	 *
	 * @return the string
	 */
	public static synchronized String dumpDernieresRequetesNonTerminees() {
		String res = "";
		String logRequete = "";
		final long maintenant = System.currentTimeMillis();
		final Iterator<String> iter = events.keySet().iterator();
		while (iter.hasNext()) {
			final String id = iter.next();
			final EtatRequete requete = events.get(id);
			if (requete.isTerminee() == false) {
				if (maintenant - requete.getTimestampDebut() < 3000) {
					logRequete = "\n-----REQUETE -------------\n" + dumpRequete(requete) + logRequete;
				}
			}
		}
		res = res + logRequete;
		return res;
	}

	/**
	 * Suppression des requêtes commencées depuis - de 3 s et non terminées Permet de purger les erreurs applicatives et de ne pas les prendre en compte dans la détection de
	 * boucles (appelé depuis jsb/exception.jsp)
	 */
	public static synchronized void purgerRequetesNonTerminees() {
		final Vector<String> aSupprimer = new Vector<String>();
		final long maintenant = System.currentTimeMillis();
		final Iterator<String> iter = events.keySet().iterator();
		while (iter.hasNext()) {
			final String id = iter.next();
			final EtatRequete requete = events.get(id);
			if (requete.isTerminee() == false) {
				if (maintenant - requete.getTimestampDebut() < 3000) {
					aSupprimer.add(id);
				}
			}
		}
		/* On supprime ces requêtes */
		final Enumeration<String> e2 = aSupprimer.elements();
		while (e2.hasMoreElements()) {
			final String id = e2.nextElement();
			//Log.debug("suppression requete " + id);
			events.remove(id);
		}
	}

	/**
	 * Dump requete.
	 *
	 * @param requete
	 *            the requete
	 *
	 * @return the string
	 */
	public static synchronized String dumpRequete(final EtatRequete requete) {
		String res = "";
		final Enumeration<String> e = requete.getEvenements().elements();
		while (e.hasMoreElements()) {
			res += "\n" + e.nextElement() + "\n...............";
		}
		return res;
	}
}