/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.database;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * SOSJDBCDriver est un driver JDBC. En fait, il s'agit d'un wrapper de driver JDBC. Il fonctionne en utilisant la même url jdbc que le driver a wrapper en la prefixant de "sos:".<br>
 * Ex: jdbc:hsqldb:mem:sostest de vient sos:jdbc:hsqldb:mem:sostest.<br>
 * Le principe de ce driver est rendre des Proxy des connexions du driver orginal.<br>
 * La classe d'implementation des Proxy de connexions est {@link SOSJDBCProxy}.
 * 
 * @author IBM
 */
public class SOSJDBCDriver implements Driver {

	/**
	 * Logger sfl4j de la classe.
	 */
	private static final Logger LOG = LoggerFactory.getLogger(SOSJDBCDriver.class);
	/**
	 * Méthode exécutée lors du chargement en mémore de cette classe par la JVM. En fait, cette méthode enregistre ce driver auprès du driver manager JDBC.
	 */
	static {
		try {
			DriverManager.registerDriver(new SOSJDBCDriver());
		} catch (Exception ex) {
			throw (ex instanceof RuntimeException ? (RuntimeException) ex : new RuntimeException(ex));
		}
	}

	/**
	 * Surcharge de la méthode de Driver JDBC. Les urls acceptées doivent commencée par "sos:".
	 * 
	 * @see java.sql.Driver#acceptsURL(java.lang.String)
	 */
	@Override
	public boolean acceptsURL(String url) throws SQLException {
		return url.startsWith("sos:");
	}

	/**
	 * Surcharge de la méthode de Driver JDBC. Pour obtenir la connection, on retire le prefixe "sos:" de l'url et l'on demande à nouveau la connection au driver gérant l'url. La
	 * connexion retournée est un proxy de la connexion obtenue auprès du driver original.
	 * 
	 * @see java.sql.Driver#connect(java.lang.String, java.util.Properties)
	 */
	@Override
	public Connection connect(String url, Properties info) throws SQLException {
		url = url.substring(4);
		if (LOG.isDebugEnabled()) {
			LOG.debug(getClass().getName() + ".connect(" + url + ")");
		}
		return Connection.class.cast(SOSJDBCProxy.createJdbcProxy(DriverManager.getDriver(url).connect(url, info)));
	}

	/**
	 * retourne simplement 1.
	 * 
	 * @see java.sql.Driver#getMajorVersion()
	 */
	@Override
	public int getMajorVersion() {
		return 1;
	}

	/**
	 * retourne simplement 1.
	 * 
	 * @see java.sql.Driver#getMinorVersion()
	 */
	@Override
	public int getMinorVersion() {
		return 1;
	}

	/**
	 * Cette méthode retourne les DriverPropertyInfo du driver original (appel après retrait du prefixe "sos:").
	 * 
	 * @see java.sql.Driver#getPropertyInfo(java.lang.String, java.util.Properties)
	 */
	@Override
	public DriverPropertyInfo[] getPropertyInfo(String url, Properties info) throws SQLException {
		url = url.substring(4);
		return DriverManager.getDriver(url).getPropertyInfo(url, info);
	}

	/**
	 * Retourne simplement true.
	 * 
	 * @see java.sql.Driver#jdbcCompliant()
	 */
	@Override
	public boolean jdbcCompliant() {
		return true;
	}

	@Override
	public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
		return null;
	}
}
