/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.database;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Arrays;
import java.util.List;
import java.util.Vector;

import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cette classe est un Proxy Java. Elle est utilisée pour wrapper les classes JDBC en partant de la connexion, puis les différents statemtn et le resultset.</br> En fait l'objectif
 * est de conserver auprès de la connexion la liste des statements et resultset ouverts afin de pouvoir les fermer à la demande ou lors de la fermeture de la connexion.</br> En
 * procédant ainsi on peut libérer les ressources monopilisées par les statements et resultset qui n'ont pas été fermés.
 *
 * @author IBM
 */
public class SOSJDBCProxy implements InvocationHandler, SOSJDBCProxyInterface {

	/**
	 * Logger sfl4j de la classe.
	 */
	private static final Logger LOG = LoggerFactory.getLogger(SOSJDBCProxy.class);

	/**
	 * Méthode static de contruction du proxy de la connexion dont il faudra suivre les ressources (statement et resultset). En fait cette méthode appelle la méthode static
	 * générique de construction d'un proxy d'interface (méthode de même nom mais privée).
	 *
	 * @param cx
	 *            la connexion à wrapper.
	 * @return la connexion wrappée (le proxy)
	 */
	public static Connection createJdbcProxy(final Connection cx) {
		return (Connection) createJdbcProxy(null, Connection.class, cx);
	}

	/**
	 * Cette méthode permet de construire des proxys de l'objet 'wrapped' passé en paramètre. Le proxy construit ainsi implémentera l'interface fourni dan le paramètre 'clazz' et
	 * l'interface SOSJDBCProxyInterface (contenant les méthodes de cette classe (JDBCProxy) afin de pouvoir les appeler directement sur le proxy).<br>
	 * Les objets ainsi wrappés sont les ressources (statement et resultset) attachées à la connexion, la connexion à laquelle sont attachées les ressources est passé dans le
	 * paramètres 'wrappedcx'.
	 *
	 * @param wrappedcx
	 *            la connnexion à laquelle la ressource dont on doit faire le proxy est attachée.
	 * @param clazz
	 *            l'interface a implementée par le proxy (ex: Statement, ResultSet).
	 * @param wrapped
	 *            l'objet original dont on doit construire un proxy.
	 * @return Le proxy de l'objet wrapped.
	 */
	private static Object createJdbcProxy(final SOSJDBCProxy wrappedcx, final Class<?> clazz, final Object wrapped) {
		try {
			if (wrapped == null) {
				return null;
			}
			return Proxy.newProxyInstance( //
				wrapped.getClass().getClassLoader(), //
				new Class[] { clazz, SOSJDBCProxyInterface.class }, //
				new SOSJDBCProxy(wrappedcx, clazz, wrapped));
		} catch (final Exception ex) {
			throw ((RuntimeException) ex);
		}
	}

	/**
	 * Liste des interfaces pour lesquelles il faudra faire des proxys. En fait chaque fois qu'une méthode est appelée sur un proxy, si le retour est d'un type d'une des interface
	 * de cette liste, on construit un proxy de l'objet retourné. Par exemple, Statement est dans la liste, donc lors d'un appel a createStetement sur une connexion, comme la
	 * connexion est un proxy, la méthode invokde de cette classe est appelée et comme le retour est d'un type figurant dans la liste le retour sera lui même trnasformé en proxy
	 * avant d'être renvoyé.
	 */
	private static List<Class<?>> wrappedClasses = Arrays.asList(new Class<?>[] { Connection.class, CallableStatement.class, PreparedStatement.class, Statement.class, ResultSet.class });

	/**
	 * connexion associés à la ressource.
	 */
	private SOSJDBCProxy wrappedcx = null;

	/**
	 * liste des proxy des ressources associées à la connexion, si l'objet original wrappé (wrapped) n'est pas une connexion ce champs est sans signification est reste null.
	 */
	private List<Object> resourcesProxy = null;

	/**
	 * liste des objets originaux des ressources associées à la connexion, si l'objet original wrappé (wrapped) n'est pas une connexion ce champs est sans signification est reste
	 * null.
	 */
	private List<Object> resourcesOriginal = null;

	/**
	 * Interface à implementer par le proxy (ex: Statement).
	 */
	private Class<?> clazz = null;

	/**
	 * objet original wrappé par le proxy.
	 */
	private Object wrapped = null;

	/**
	 * Constructeur des proxys.
	 *
	 * @param wrappedproxy
	 *            connexion associés au proxy
	 * @param clazz
	 *            l'interface à implementer par le proxy (ex: Statement).
	 * @param wrapped
	 *            l'objet original wrappé par le proxy.
	 */
	public SOSJDBCProxy(final SOSJDBCProxy wrappedproxy, final Class<?> clazz, final Object wrapped) {
		// if (LOG.isDebugEnabled())
		// LOG.debug("            Wrapping class:"+clazz.getName());
		this.wrappedcx = wrappedproxy;
		if (wrappedcx == null) {
			if (clazz.equals(Connection.class)) {
				this.wrappedcx = this;
				resourcesProxy = new Vector<>();
				resourcesOriginal = new Vector<>();
			} else {
				throw new RuntimeException("SOSJDBC : Erreur Fatale : La connection associée ne doit pas être nulle!");
			}
		}
		this.clazz = clazz;
		this.wrapped = wrapped;
	}

	/**
	 * Cette méthode est appelée à chaque appel de méthode sur un objet proxy.<br>
	 * Si la méthode appelée est dans l'interface SOSJDBCProxyInterface, la méthode correspondante est directement appelée.<br>
	 * Dans les autres cas, les paramètres d'appels sont deproxyfiés (on remplace chaque paramètre de type proxy par son objet orginal), puis la méthode correspondante est appelé
	 * sur l'objet original.<br>
	 * Si le retour (objet résultat de l'appel) est d'un type de la liste des interface, on construit un proxy de l'objet retourné. Par exemple, Statement est dans la liste, donc
	 * lors d'un appel a createStetement sur une connexion, comme la connexion est un proxy, la méthode invokde de cette classe est appelée et comme le retour est d'un type
	 * figurant dans la liste le retour sera lui même trnasformé en proxy avant d'être renvoyé.<br>
	 * L'objet retourné est ajouté à la liste des ressources de la connexion (si il est dans la liste des classes wrappedClasses).< br>
	 * Lorsqu'une méthode close est appelée sur un proxy de ressources, la ressource est retirée des listes de resosources de la connexion associée.
	 *
	 * @see java.lang.reflect.InvocationHandler#invoke(java.lang.Object, java.lang.reflect.Method, java.lang.Object[])
	 */
	@Override
	public Object invoke(final Object proxy, Method method, Object[] args) throws Throwable {
		if (SOSJDBCProxyInterface.class.equals(method.getDeclaringClass())) {
			return method.invoke(this, args);
		}
		final Class<?> declaringClass = method.getDeclaringClass();
		if (LOG.isDebugEnabled() && (declaringClass.equals(SOSJDBCProxyInterface.class) || wrappedClasses.contains(declaringClass))) {
			LOG.debug("(" + clazz.getName() + ")" + wrapped + "." + method.getName() + "(" + (args == null ? "" : ArrayUtils.toString(args)) + ")");
		}
		if (method.getName().equals("close") && (args == null || args.length == 0)) {
			if (clazz.equals(Connection.class)) {
				cleanConnection(Connection.class.cast(proxy));
			} else {
				removeResource(proxy);
			}
		}
		if (args != null && args.length > 0) {
			for (int i = 0; i < args.length; i++) {
				if (isProxy(args[0])) {
					args[0] = ((SOSJDBCProxyInterface) args[0]).getWrapped();
				}
			}
		}
		if (method.getName().equals("prepareStatement")) {
			if (args != null && args.length == 1 && args[0] instanceof String && ((String) args[0]).toLowerCase().trim().startsWith("insert")) {
				method = declaringClass.getMethod("prepareStatement", String.class, int.class);
				args = new Object[] { args[0], Statement.RETURN_GENERATED_KEYS };
			}
		}
		try {
			Object result = method.invoke(wrapped, args);
			if (result != null && !isProxy(result)) {
				final Class<?> returnClass = method.getReturnType();
				for (Class<?> wrappedClass : wrappedClasses) {
					if (wrappedClass.isAssignableFrom(returnClass)) {
						final Object resultOriginal = result;
						result = createJdbcProxy(wrappedcx, returnClass, returnClass.cast(result));
						wrappedcx.addResource(resultOriginal, result);
						break;
					}
				}
			}
			return result;
		} catch (final InvocationTargetException e) {
			throw e.getTargetException();
		}
	}

	/**
	 * Méthode qui retourne l'objet original wrappé par le proxy.
	 *
	 * @see com.jsbsoft.jtf.database.SOSJDBCProxyInterface#getWrapped()
	 */
	@Override
	public Object getWrapped() {
		return wrapped;
	}

	/**
	 * Cette méthode ajoute le proxy et l'objet original dans la liste des ressources de la connexion associée (wrappedcx).
	 *
	 * @see com.jsbsoft.jtf.database.SOSJDBCProxyInterface#addResource(Object, Object)
	 */
	@Override
	public void addResource(final Object orginal, final Object proxy) {
		synchronized (wrappedcx) {
			if (!wrappedcx.resourcesOriginal.contains(orginal) && !(orginal instanceof Connection)) {
				wrappedcx.resourcesOriginal.add(orginal);
				wrappedcx.resourcesProxy.add(proxy);
			}
		}
	}

	/**
	 * Cette méthode retire la ressource passée en paramètre de la liste des ressources de la connexion assocées. L'objet ressource passé peut être un priginal ou un proxy.
	 *
	 * @see com.jsbsoft.jtf.database.SOSJDBCProxyInterface#removeResource(java.lang.Object)
	 */
	@Override
	public void removeResource(final Object obj) {
		Object wrapped = obj;
		if (wrapped instanceof SOSJDBCProxyInterface) {
			wrapped = ((SOSJDBCProxyInterface) wrapped).getWrapped();
		}
		synchronized (wrappedcx) {
			final int index = wrappedcx.resourcesOriginal.indexOf(wrapped);
			if (index >= 0) {
				wrappedcx.resourcesOriginal.remove(index);
				wrappedcx.resourcesProxy.remove(index);
			}
		}
	}

	/**
	 * Cette méthode retourne la liste des proxys des ressources associées à ce proxy. En fait c'est toujours null sauf si l'objet orginal est une connexion, l'ensemble des
	 * ressources de la connexion est géré au niveau du proxy de la conexion.
	 */
	@Override
	public List<Object> getResourcesProxy() {
		return resourcesProxy;
	}

	@Override
	public List<Object> getResourcesOriginales() {
		return resourcesOriginal;
	}

	/**
	 * Méthode static retournant true si l'objet passé en paramètre est un proxy de cette classe.
	 *
	 * @param obj
	 *            objet à tester
	 * @return true si l'objet est un proxy implémenté par cette classe.
	 */
	public static boolean isProxy(final Object obj) {
		return (obj instanceof SOSJDBCProxyInterface);
	}

	/**
	 * Cette méthode retourne l'interface implémenté par le proxy.
	 */
	@Override
	public Class<?> getClazz() {
		return clazz;
	}

	/**
	 * Cette méthode statique nettoie la connexion (proxy de connexion) passé en paramètre. En fait cela ferme (close) toutes les ressources encore associées à l'objet Proxy de
	 * connexion passé en paramètre.
	 *
	 * @param c la connexion à supprimer
	 */
	public static void cleanConnection(final Connection c) {
		if (c == null) {
			return;
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("cleanConnection de " + c);
		}
		if (isProxy(c)) {
			final SOSJDBCProxyInterface proxyC = (SOSJDBCProxyInterface) c;
			List<Object> resourcesCxProxy = null;
			List<Object> resourcesCxOriginales = null;
			synchronized (proxyC) {
				resourcesCxProxy = proxyC.getResourcesProxy();
				resourcesCxOriginales = proxyC.getResourcesOriginales();
			}
			final int sizeResource = resourcesCxProxy.size();
			int count = 0;
			while (resourcesCxProxy.size() > 0) {
				count++;
				final int index = resourcesCxProxy.size() - 1;
				final SOSJDBCProxyInterface obj = (SOSJDBCProxyInterface) resourcesCxProxy.get(index);
				try {
					final Method methClose = obj.getClazz().getMethod("close");
					methClose.invoke(obj);
				} catch (final NoSuchMethodException | SecurityException | InvocationTargetException | IllegalAccessException ex) {
					LOG.error("SOSJDBC : Ressource impossible a closer " + obj.toString(), ex);
				} finally {
					synchronized (proxyC) {
						if (index < resourcesCxProxy.size()) {
							LOG.error("SOSJDBC : Ressource impossible a cleaner (nb ressouces proxy/originale:" + resourcesCxProxy.size() + "/" + resourcesCxOriginales.size() + "): " + obj.toString());
							resourcesCxProxy.remove(index);
							resourcesCxOriginales.remove(index);
						}
					}
				}
				// forçage sortie de boucle si jamais
				if (count > 3 * sizeResource) {
					LOG.error("SOSJDBC : CleanConnection impossible");
					throw new RuntimeException("forçage sortie de boucle cleanConnection");
				}
			}
			if (LOG.isDebugEnabled()) {
				LOG.debug("cleanConnection terminé " + c);
			}
		} else {
			throw new RuntimeException("Cette connexion n'est pas un proxy, elle ne peut pas être nettoyée:" + c);
		}
	}
}
