/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.datasource.dao.impl.mysql;

import com.jsbsoft.jtf.datasource.dao.CommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public abstract class AbstractCommonDAO<T> implements CommonDAO<T> {

	/** The Constant LOG. */
	public static final Logger LOG = LoggerFactory.getLogger(AbstractCommonDAO.class);
	private static final String SELECT_DISTINCT = "select DISTINCT %s from `%s` T1 ";
	private static final String COLUMNS_PREFIX = "T1.";
	protected final Set<String> columns = new HashSet<>();
	protected DataSource dataSource;
	protected String tableName;
	private String columnsNameForRequest = null;

	public DataSource getDataSource() {
		return dataSource;
	}

	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}

	protected void getColumns() throws DataSourceException {
		ResultSet rs = null;
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s` T1", tableName))) {
			rs = stmt.executeQuery();
			final ResultSetMetaData md = rs.getMetaData();
			final int columnCount = md.getColumnCount();
			for (int i = 1; i <= columnCount; i++) {
				columns.add(md.getColumnName(i));
			}
		} catch (final SQLException e) {
			throw new DataSourceException(String.format("An error occured retrieving columns for table %s", tableName), e);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (final SQLException e) {
					LOG.error(String.format("An error occured closing ResultSet during getColumns() for table %s", tableName), e);
				}
			}
		}
	}

	@Deprecated
	@Override
	public List<T> select(final String request) throws DataSourceException {
		initColumns();
		ResultSet rs = null;
		final List<T> results = new ArrayList<>();
		String query = String.format(SELECT_DISTINCT, columnsNameForRequest, tableName);
		if (request != null) {
			query += request;
		}
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(query)) {
			rs = stmt.executeQuery();
			while (rs.next()) {
				results.add(fill(rs));
			}
		} catch (final SQLException e) {
			throw new DataSourceException(String.format("Selection request on table \"%s\" failed", tableName), e);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (final SQLException e) {
					LOG.error(String.format("An error occured closing ResultSet during select() for table %s", tableName), e);
				}
			}
		}
		return results;
	}

	private void initColumns() throws DataSourceException {
		if (columns.isEmpty() || StringUtils.isBlank(columnsNameForRequest)) {
			getColumns();
			@SuppressWarnings("unchecked")
			final Collection<String> columnsName = CollectionUtils.collect(columns, new Transformer() {
				@Override
				public String transform(final Object input) {
					return COLUMNS_PREFIX + input;
				}
			});
			columnsNameForRequest = StringUtils.join(columnsName, " , ");
		}
	}

	@Override
	public void delete(final Long id) throws DeleteFromDataSourceException {
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(String.format("delete from `%1$s` WHERE ID_%1$s = ?", tableName))) {
			stmt.setObject(1, id, Types.BIGINT);
			final int rowsAffected = stmt.executeUpdate();
			if (rowsAffected == 0) {
				throw new DeleteFromDataSourceException(String.format("Table \"%s\" doesn't contain any row width id %d", tableName, id));
			} else if (rowsAffected > 1) {
				throw new DeleteFromDataSourceException(String.format("Table \"%s\" contains more than one row width id %d", tableName, id));
			}
		} catch (final SQLException e) {
			throw new DeleteFromDataSourceException(String.format("An error occured during deletion of row with id %d from table \"%s\"", id, tableName), e);
		}
	}

	@Override
	public T getById(final Long id) throws DataSourceException {
		ResultSet rs = null;
		T object = null;
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%1$s` T1 WHERE T1.ID_%1$s = ?", tableName))) {
			stmt.setObject(1, id, Types.BIGINT);
			rs = stmt.executeQuery();
			if (!rs.first()) {
				throw new DataSourceException(String.format("No row with id %d from table %s could be retrieved", id, tableName));
			}
			object = fill(rs);
		} catch (final SQLException e) {
			throw new DataSourceException(String.format("An error occured retrieving object with id %d from table %s", id, tableName), e);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (final SQLException e) {
					LOG.error(String.format("An error occured closing ResultSet during getById() on table %s", tableName), e);
				}
			}
		}
		return object;
	}

	protected abstract T fill(ResultSet rs) throws DataSourceException;
}
