/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.monitoring;

import javax.servlet.http.HttpServletRequest;

import com.jsbsoft.jtf.database.OMContext;
import com.kportal.core.config.PropertyHelper;

/**
 * Classe abstraite définissant un module de monitoring.
 * 
 * Les parametres du module sont lus a partir du fichier de proprietes, les clefs de celles-ci sont prefixees par PARAM_MODULE_MONITORING + ".".
 * 
 * @see com.jsbsoft.jtf.monitoring.AbsScenario
 * @see com.jsbsoft.jtf.monitoring.GestionnaireScenarios
 * @author jbiard
 */
public abstract class AbsModuleMonitoring {

	/** The Constant CODE_ERREUR_RESULTAT_EXECUTION_OK. */
	public final static int CODE_ERREUR_RESULTAT_EXECUTION_OK = 0;

	/** The Constant CODE_ERREUR_RESULTAT_EXECUTION_ERREUR. */
	public final static int CODE_ERREUR_RESULTAT_EXECUTION_ERREUR = -1;

	/** The Constant CODE_ERREUR_RESULTAT_EXECUTION_WARNING. */
	public final static int CODE_ERREUR_RESULTAT_EXECUTION_WARNING = 1;

	/** The Constant PARAM_MODULE_MONITORING. */
	public final static String PARAM_MODULE_MONITORING = "monitoring.module";

	/** The Constant PARAM_MODULE_MONITORING_TPS_WARNING. */
	public final static String PARAM_MODULE_MONITORING_TPS_WARNING = "tps_warning";

	/** The Constant PARAM_MODULE_MONITORING_TPS_CRITIQUE. */
	public final static String PARAM_MODULE_MONITORING_TPS_CRITIQUE = "tps_critique";

	/** The intitule. */
	private final String code, intitule;

	/**
	 * Constructeur.
	 * 
	 * @param code
	 *            code du module(utilise pour la recuperation de propriete, donc unique)
	 * @param intitule
	 *            intitule du module
	 */
	public AbsModuleMonitoring(final String code, final String intitule) {
		this.code = code;
		this.intitule = intitule;
	}

	/**
	 * Gets the intitule.
	 * 
	 * @return the intitule
	 */
	public String getIntitule() {
		return intitule;
	}

	/**
	 * Execute le diagnostic du module.
	 * 
	 * @param intituleScenario
	 *            intitule du scenario auquel appartient le module
	 * @param ctx
	 *            contexte
	 * @param req
	 *            requete HTTP recue par la jsp de monitoring
	 * 
	 * @return resultat de l'execution @see ResultatExecutionMonitoring
	 */
	public abstract ResExec executer(String intituleScenario, OMContext ctx, HttpServletRequest req);

	/**
	 * Classe permettant de modeliser le resultat d'execution d'un module.
	 * 
	 * @author jbiard
	 */
	public class ResExec {

		/** The Constant STATUT_OK. */
		public final static String STATUT_OK = "_OK_";

		/** The Constant STATUT_ERREUR. */
		public final static String STATUT_ERREUR = "_ERREUR_";

		/** The Constant STATUT_WARNING. */
		public final static String STATUT_WARNING = "_WARNING_";

		/**
		 * Instantiates a new res exec.
		 * 
		 * @param intituleScenario
		 *            the intitule scenario
		 * @param intituleModuleMonitoring
		 *            the intitule module monitoring
		 * @param codeErreur
		 *            the code erreur
		 * @param msg
		 *            the msg
		 */
		public ResExec(final String intituleScenario, final String intituleModuleMonitoring, final int codeErreur, final String msg) {
			this(intituleScenario, intituleModuleMonitoring, codeErreur, msg, -1);
		}

		/**
		 * Instantiates a new res exec.
		 * 
		 * @param intituleScenario
		 *            the intitule scenario
		 * @param intituleModuleMonitoring
		 *            the intitule module monitoring
		 * @param codeErreur
		 *            the code erreur
		 * @param msg
		 *            the msg
		 * @param tpsExec
		 *            the tps exec
		 */
		public ResExec(final String intituleScenario, final String intituleModuleMonitoring, final int codeErreur, final String msg, final long tpsExec) {
			this.intituleScenario = intituleScenario;
			this.intituleModuleMonitoring = intituleModuleMonitoring;
			this.codeErreur = codeErreur;
			this.message = msg;
			this.temps = tpsExec;
			if (codeErreur == CODE_ERREUR_RESULTAT_EXECUTION_OK) {
				statut = STATUT_OK;
			} else if (codeErreur <= CODE_ERREUR_RESULTAT_EXECUTION_ERREUR) {
				statut = STATUT_ERREUR;
			} else if (codeErreur >= CODE_ERREUR_RESULTAT_EXECUTION_WARNING) {
				statut = STATUT_WARNING;
			}
		}

		/**
		 * Gets the statut.
		 * 
		 * @return the statut
		 */
		public String getStatut() {
			return statut;
		}

		/**
		 * Gets the code erreur.
		 * 
		 * @return the code erreur
		 */
		public int getCodeErreur() {
			return codeErreur;
		}

		/**
		 * Gets the intitule module monitoring.
		 * 
		 * @return the intitule module monitoring
		 */
		public String getIntituleModuleMonitoring() {
			return intituleModuleMonitoring;
		}

		/**
		 * Gets the intitule scenario.
		 * 
		 * @return the intitule scenario
		 */
		public String getIntituleScenario() {
			return intituleScenario;
		}

		/**
		 * Gets the message.
		 * 
		 * @return the message
		 */
		public String getMessage() {
			return message;
		}

		/**
		 * Gets the temps.
		 * 
		 * @return the temps
		 */
		public long getTemps() {
			return temps;
		}

		/** The message. */
		private String statut;

		private final String intituleScenario;

		private final String intituleModuleMonitoring;

		private final String message;

		/** The code erreur. */
		private final int codeErreur;

		/** The temps. */
		private final long temps;
	}

	/**
	 * Renvoie la valeur d'une propriete definie pour le module.
	 * 
	 * @param param
	 *            parametre permettant de constituer la clef dans le fichier de proprietes
	 * 
	 * @return valeur sous forme de chaine
	 */
	protected String getParametre(final String param) {
		return PropertyHelper.getCoreProperty(PARAM_MODULE_MONITORING + "." + code + "." + param);
	}

	/**
	 * Retourne le temps seuil du warning en ms.
	 * 
	 * @return temps seuil du warning en ms.
	 */
	protected long getTpsWarning() {
		final String param = getParametre(PARAM_MODULE_MONITORING_TPS_WARNING);
		if (param != null) {
			return Long.parseLong(param);
		}
		return -1;
	}

	/**
	 * Retourne le temps seuil critique en ms.
	 * 
	 * @return temps seuil critique en ms.
	 */
	protected long getTpsCritique() {
		final String param = getParametre(PARAM_MODULE_MONITORING_TPS_CRITIQUE);
		if (param != null) {
			return Long.parseLong(param);
		}
		return -1;
	}

	/**
	 * Retourne le code d'erreur associe suivant le temps d'execution et des temps seuil critique et de warning.
	 * 
	 * @param tps
	 *            d'execution
	 * 
	 * @return code d'erreur
	 */
	protected int getCodeErreurTpsExec(final long tps) {
		final long tpsWarning = getTpsWarning();
		final long tpsCritique = getTpsCritique();
		if ((tpsWarning == -1) || (tpsCritique == -1)) {
			return CODE_ERREUR_RESULTAT_EXECUTION_OK;
		} else if (tps < tpsWarning) {
			return CODE_ERREUR_RESULTAT_EXECUTION_OK;
		} else if ((tps >= tpsWarning) && (tps < tpsCritique)) {
			return CODE_ERREUR_RESULTAT_EXECUTION_WARNING;
		}
		return CODE_ERREUR_RESULTAT_EXECUTION_ERREUR;
	}
}
