/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.monitoring;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.kportal.core.config.PropertyHelper;

/**
 * Gestionnaire de scenarios de monitoring. Singleton.
 * 
 * Gere une liste d'IP de restriction d'acces, lue dans le fichier de proprietes (par defaut seule 127.0.0.1 est autorisée).
 * 
 * Lit les classes des scenarios depuis le fichier de proprietes. Ces classes doivent étendre AbsScenario. @see com.jsbsoft.jtf.monitoring.AbsScenario
 * 
 * @author jbiard
 */
public class GestionnaireScenarios {

	/** The Constant PARAM_REQUETE_NOM_SCENARIO. */
	public final static String PARAM_REQUETE_NOM_SCENARIO = "scenario";

	/** The Constant PARAM_JTF_SCENARIOS_CLASSES. */
	public final static String PARAM_JTF_SCENARIOS_CLASSES = "monitoring.scenarios_classes";

	/** The Constant PARAM_JTF_RESTRICTION_IPS. */
	public final static String PARAM_JTF_RESTRICTION_IPS = "monitoring.ips_restriction";

	/** The Constant PARAM_JTF_RAFRAICHISSEMENT_VUE. */
	public final static String PARAM_JTF_RAFRAICHISSEMENT_VUE = "monitoring.rafraichissement_vue";

	/** The Constant IP_AUTORISEE_DEFAUT. */
	public final static String IP_AUTORISEE_DEFAUT = "127.0.0.1";

	/** The Constant RAFRAICHISSEMENT_VUE_DEFAUT. */
	public final static long RAFRAICHISSEMENT_VUE_DEFAUT = 300;

	/** The _instance. */
	private static GestionnaireScenarios _instance;

	/** The _log. */
	private static final Logger LOG = LoggerFactory.getLogger(GestionnaireScenarios.class);

	/** The map scenarios. */
	private final Map<String, AbsScenario> mapScenarios = new HashMap<>();

	/** The set ips restriction. */
	private final Set<String> setIpsRestriction = new HashSet<>();

	/**
	 * Accesseur statique (singleton) .
	 * 
	 * @return the instance
	 * 
	 * @throws InstantiationException
	 *             levee si pb chargement scenario
	 * @throws IllegalAccessException
	 *             levee si pb chargement scenario
	 * @throws ClassNotFoundException
	 *             levee si pb chargement scenario
	 */
	public synchronized static GestionnaireScenarios getInstance() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		if (_instance == null) {
			_instance = new GestionnaireScenarios();
		}
		return _instance;
	}

	/**
	 * Retourne le temps de rafraichissement.
	 * 
	 * @return the tps rafraichissement
	 */
	public String getTpsRafraichissement() {
		final String tps = PropertyHelper.getCoreProperty(PARAM_JTF_RAFRAICHISSEMENT_VUE);
		if (tps != null) {
			return tps;
		}
		return "" + RAFRAICHISSEMENT_VUE_DEFAUT;
	}

	/**
	 * Enregistre un scenario de monitoring.
	 * 
	 * @param scenario
	 *            scenario a enregistrer
	 * 
	 * @return null si l'ajout n'a pu se faire, sinon le scenario
	 */
	public synchronized Object enregistreScenario(final AbsScenario scenario) {
		return mapScenarios.put(scenario.getCode(), scenario);
	}

	/**
	 * Supprime un scenario.
	 * 
	 * @param scenario
	 *            scenario a supprimer
	 * 
	 * @return null si la suppression n'a pu se faire, sinon le scenario
	 */
	public synchronized Object supprimeScenario(final AbsScenario scenario) {
		return mapScenarios.remove(scenario.getCode());
	}

	/**
	 * Lance l'execution du ou des scenarios. Si un parametre dans la requete HTTP precise un code de scenario a executer, seulement celui-ci sera execute, sinon ce sera l'ensemble
	 * des scenarios enregistres aupres du gestionnaire qui seront executes.
	 * 
	 * @param ctx
	 *            contexte
	 * @param req
	 *            requete HTTP
	 * 
	 * @return ensemble de tableaux de résultats @see ResultatExecutionMonitoring pour le ou les scenarios
	 * 
	 * @throws ExceptionMonitoring
	 *             the exception monitoring
	 */
	public synchronized Set<AbsModuleMonitoring.ResExec[]> executerScenarios(final OMContext ctx, final HttpServletRequest req) throws ExceptionMonitoring {
		final String remoteAddr = req.getRemoteAddr();
		// verification IP d'acces
		if (!setIpsRestriction.contains(remoteAddr)) {
			LOG.error("IP invalide : " + remoteAddr);
			throw new ExceptionMonitoring("IP invalide");
		}
		final Set<AbsModuleMonitoring.ResExec[]> listeResultats = new LinkedHashSet<>();
		AbsScenario scenario;
		// execution d'un seul scenario ?
		final String paramNomScenario = req.getParameter(PARAM_REQUETE_NOM_SCENARIO);
		if (paramNomScenario != null) {
			scenario = mapScenarios.get(paramNomScenario);
			if (scenario == null) {
				throw new ExceptionMonitoring("Scénario " + paramNomScenario + " inexistant");
			}
			listeResultats.add(scenario.executer(ctx, req));
			return listeResultats;
		}
		for (final String string : mapScenarios.keySet()) {
			scenario = mapScenarios.get(string);
			listeResultats.add(scenario.executer(ctx, req));
		}
		return listeResultats;
	}

	/**
	 * Constructeur prive.
	 * 
	 * @throws InstantiationException
	 *             levee si pb de chargement de scenario
	 * @throws IllegalAccessException
	 *             levee si pb de chargement de scenario
	 * @throws ClassNotFoundException
	 *             levee si pb de chargement de scenario
	 */
	private GestionnaireScenarios() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		String param = PropertyHelper.getCoreProperty(PARAM_JTF_SCENARIOS_CLASSES);
		if (param != null) {
			AbsScenario scenario;
			for (final StringTokenizer st = new StringTokenizer(param, ";"); st.hasMoreTokens();) {
				scenario = (AbsScenario) Class.forName(st.nextToken()).newInstance();
				mapScenarios.put(scenario.getCode(), scenario);
			}
		}
		setIpsRestriction.add(IP_AUTORISEE_DEFAUT);
		param = PropertyHelper.getCoreProperty(PARAM_JTF_RESTRICTION_IPS);
		if (param != null) {
			for (final StringTokenizer st = new StringTokenizer(param, ";"); st.hasMoreTokens();) {
				setIpsRestriction.add(st.nextToken());
			}
		}
	}
}
