/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.monitoring.impl;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.PatternMatcherInput;
import org.apache.oro.text.regex.Perl5Compiler;
import org.apache.oro.text.regex.Perl5Matcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.identification.GestionnaireIdentification;
import com.jsbsoft.jtf.identification.ValidateurCAS;
import com.jsbsoft.jtf.monitoring.AbsModuleMonitoring;

/**
 * Connecte un utilisateur sur le serveur CAS.
 *
 * @author jbiard
 */
public class ModuleConnexionCAS extends AbsModuleMonitoring {

	/** The Constant MODULE_CODE. */
	public final static String MODULE_CODE = "connexion_cas";

	/** The Constant MODULE_INTITULE. */
	public final static String MODULE_INTITULE = "Connexion serveur CAS";

	/** The Constant MODULE_PARAM_URI_LOGIN_CAS. */
	public final static String MODULE_PARAM_URI_LOGIN_CAS = "uri_login_cas";

	/** The Constant MODULE_PARAM_LOGIN_CAS. */
	public final static String MODULE_PARAM_LOGIN_CAS = "login_cas";

	/** The Constant MODULE_PARAM_PASSWORD_CAS. */
	public final static String MODULE_PARAM_PASSWORD_CAS = "password_cas";

	/** The Constant MODULE_PARAM_REGEXP_CONNEXION_REUSSIE. */
	public final static String MODULE_PARAM_REGEXP_CONNEXION_REUSSIE = "regexp_connexion_reussie";

	/** The _log. */
	private static final Logger LOG = LoggerFactory.getLogger(ModuleConnexionCAS.class);

	/** The Constant REG_EXP_LT. */
	private final static String REG_EXP_LT = "LT-\\d{1,}-[a-zA-Z0-9]{20}";

	/** The regexp connexion reussie. */
	private Pattern regexpPatternLT, regexpConnexionReussie;

	/** The hote cas. */
	private final String hoteCas;

	/** The port cas. */
	private final int portCas;

	/**
	 * Instantiates a new module connexion cas.
	 */
	public ModuleConnexionCAS() {
		super(MODULE_CODE, MODULE_INTITULE);
		final ValidateurCAS validateurCAS = GestionnaireIdentification.getInstance().getValidateurCAS();
		hoteCas = validateurCAS.getHoteCas();
		portCas = validateurCAS.getPortServeurCas();
		try {
			final Perl5Compiler compiler = new Perl5Compiler();
			regexpPatternLT = compiler.compile(REG_EXP_LT);
			regexpConnexionReussie = compiler.compile(getParametre(MODULE_PARAM_REGEXP_CONNEXION_REUSSIE));
		} catch (final MalformedPatternException e) {
			LOG.error("erreur de pattern", e);
		}
	}

	/**
	 * Flux2 string.
	 *
	 * @param is
	 *            the is
	 *
	 * @return the string
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	private String flux2String(final InputStream is) throws IOException {
		final BufferedInputStream bis = new BufferedInputStream(is);
		final byte[] buffer = new byte[1024 * 4];
		String res = "";
		int nbBytes;
		while ((nbBytes = bis.read(buffer)) > 0) {
			res += new String(buffer, 0, nbBytes);
		}
		bis.close();
		return res;
	}

	/* (non-Javadoc)
	 * @see com.jsbsoft.jtf.monitoring.AbsModuleMonitoring#executer(java.lang.String, com.jsbsoft.jtf.database.OMContext, javax.servlet.http.HttpServletRequest)
	 */
	@Override
	public ResExec executer(final String intiutleScenario, final OMContext ctx, final HttpServletRequest req) {
		long tps = System.currentTimeMillis();
		final HttpClient client = new HttpClient();
		client.getHostConfiguration().setHost(hoteCas, portCas, "https");
		final GetMethod authget = new GetMethod(getParametre(MODULE_PARAM_URI_LOGIN_CAS));
		ResExec res = null;
		try {
			client.executeMethod(authget);
			// extraction du LT : necessaire pour la soumission du formulaire
			String lt = null;
			PatternMatcherInput pmiPage = new PatternMatcherInput(flux2String(authget.getResponseBodyAsStream()));
			//new PatternMatcherInput( authget.getResponseBodyAsString( ) );
			final Perl5Matcher matcher = new Perl5Matcher();
			if (matcher.contains(pmiPage, regexpPatternLT)) {
				lt = matcher.getMatch().toString();
			} else {
				authget.releaseConnection();
				return new ResExec(intiutleScenario, MODULE_INTITULE, CODE_ERREUR_RESULTAT_EXECUTION_ERREUR,
					"Impossible de trouver le paramètre LT dans le corps lors de la connexion au serveur CAS " + hoteCas + " (serveur indisponible ?).");
			}
			// relachement connexion
			authget.releaseConnection();
			// soumission du formulaire
			final PostMethod authpost = new PostMethod(getParametre(MODULE_PARAM_URI_LOGIN_CAS));
			authpost.setRequestBody(new NameValuePair[] { new NameValuePair("username", getParametre(MODULE_PARAM_LOGIN_CAS)), new NameValuePair("password",
				getParametre(MODULE_PARAM_PASSWORD_CAS)), new NameValuePair("lt", lt) });
			client.executeMethod(authpost);
			// recherche d'une chaine dans la page pour valider la connexion
			pmiPage = new PatternMatcherInput(flux2String(authpost.getResponseBodyAsStream()));
			// new PatternMatcherInput( authpost.getResponseBodyAsString( ) ) ;
			tps = System.currentTimeMillis() - tps;
			if (matcher.contains(pmiPage, regexpConnexionReussie)) {
				res = new ResExec(intiutleScenario, MODULE_INTITULE, getCodeErreurTpsExec(tps),
					"Authentification sur le serveur CAS " + hoteCas + " validée<br/>pour l'utilisateur " + getParametre(MODULE_PARAM_LOGIN_CAS), tps);
			} else {
				res = new ResExec(intiutleScenario, MODULE_INTITULE, AbsModuleMonitoring.CODE_ERREUR_RESULTAT_EXECUTION_ERREUR, "Echec connexion au serveur CAS " + hoteCas, tps);
			}
			// relachement
			authpost.releaseConnection();
		} catch (final IOException e) {
			LOG.error("Echec connexion au serveur CAS " + hoteCas, e);
			res = new ResExec(intiutleScenario, MODULE_INTITULE, AbsModuleMonitoring.CODE_ERREUR_RESULTAT_EXECUTION_ERREUR, "Echec connexion au serveur CAS " + hoteCas);
		}
		return res;
	}
}
