/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.monitoring.impl;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.monitoring.AbsModuleMonitoring;

/**
 * The Class ModuleOccupationMemoire.
 */
public class ModuleOccupationMemoire extends AbsModuleMonitoring {

	/** The Constant MODULE_PARAM_POURENTAGE_WARNING. */
	public final static String MODULE_PARAM_POURENTAGE_WARNING = "pourcentage_occupation_warning";

	/** The Constant MODULE_PARAM_POURENTAGE_CRITIQUE. */
	public final static String MODULE_PARAM_POURENTAGE_CRITIQUE = "pourcentage_occupation_critique";

	/** The Constant MODULE_CODE. */
	public final static String MODULE_CODE = "occupation_memoire";

	/** The Constant MODULE_INTITULE. */
	public final static String MODULE_INTITULE = "Occupation mémoire";

	/** The _log. */
	private static final Logger LOG = LoggerFactory.getLogger(ModuleOccupationMemoire.class);

	/** The thread lance. */
	private static boolean threadLance = false;

	/** The Constant DELAI_THREAD_INFO_MEMOIRE. */
	private final static int DELAI_THREAD_INFO_MEMOIRE = 5000;

	/**
	 * Instantiates a new module occupation memoire.
	 */
	public ModuleOccupationMemoire() {
		super(MODULE_CODE, MODULE_INTITULE);
	}

	/**
	 * Thread qui affiche par intermitence l'occupation memoire.
	 */
	public static void lanceThreadMemoire() {
		if (LOG.isDebugEnabled() && !threadLance) {
			// pour qu'il n'y ait qu'un seul thread
			synchronized (ModuleOccupationMemoire.class) {
				// pour le 2eme thread
				if (!threadLance) {
					new Thread() {

						@Override
						public void run() {
							long heapSize, heapMaxSize, heapFreeSize;
							int pourcentageOccupation;
							while (true) {
								heapMaxSize = Runtime.getRuntime().maxMemory() / 1024;
								heapSize = Runtime.getRuntime().totalMemory() / 1024;
								heapFreeSize = Runtime.getRuntime().freeMemory() / 1024;
								pourcentageOccupation = (int) (((heapSize - heapFreeSize) * 100) / heapSize);
								LOG.debug("Taille du tas : " + heapSize + " ko | Taille maximale du tas : " + heapMaxSize + " ko | Occupation mémoire : " + pourcentageOccupation + "% (" + (heapSize - heapFreeSize) + " ko)");
								try {
									Thread.sleep(DELAI_THREAD_INFO_MEMOIRE);
								} catch (InterruptedException e) {
									LOG.error("Erreur lors de la mise en veille du thread", e);
								}
							}
						};
					}.start();
					threadLance = true;
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.jsbsoft.jtf.monitoring.AbsModuleMonitoring#executer(java.lang.String, com.jsbsoft.jtf.database.OMContext, javax.servlet.http.HttpServletRequest)
	 */
	@Override
	public ResExec executer(String intituleScenario, OMContext ctx, HttpServletRequest req) {
		// Get current size of heap in bytes
		long heapSize = Runtime.getRuntime().totalMemory() / 1024;
		// Get maximum size of heap in bytes. The heap cannot grow beyond this size.
		// Any attempt will result in an OutOfMemoryException.
		long heapMaxSize = Runtime.getRuntime().maxMemory() / 1024;
		// Get amount of free memory within the heap in bytes. This size will increase
		// after garbage collection and decrease as new objects are created.
		long heapFreeSize = Runtime.getRuntime().freeMemory() / 1024;
		int pourcentageOccupation = (int) (((heapSize - heapFreeSize) * 100) / heapSize);
		String infosMem = "Taille du tas : " + heapSize + " ko | Taille maximale du tas : " + heapMaxSize + " ko <br/>Occupation mémoire : " + pourcentageOccupation + "% (" + (heapSize - heapFreeSize) + " ko)";
		int pourcentageOccupationWarning = Integer.parseInt(getParametre(MODULE_PARAM_POURENTAGE_WARNING));
		int pourcentageOccupationCritique = Integer.parseInt(getParametre(MODULE_PARAM_POURENTAGE_CRITIQUE));
		if ((pourcentageOccupation >= pourcentageOccupationWarning) && (pourcentageOccupation < pourcentageOccupationCritique)) {
			LOG.warn(infosMem);
			return new ResExec(intituleScenario, MODULE_INTITULE, 1, infosMem);
		} else if (pourcentageOccupation >= pourcentageOccupationWarning) {
			LOG.error(infosMem);
			return new ResExec(intituleScenario, MODULE_INTITULE, -1, infosMem);
		}
		return new ResExec(intituleScenario, MODULE_INTITULE, 0, infosMem);
	}
}
