/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.stats;

import java.util.HashMap;
import java.util.Map;

import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.PatternMatcherInput;
import org.apache.oro.text.regex.Perl5Compiler;
import org.apache.oro.text.regex.Perl5Matcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurTechnique;
import com.kportal.core.config.PropertyHelper;

/**
 * Classe permettant les logs de statistiques. On manipule des switchs (prefixe par %) permettant de construire l'affichage en fonction d'un format, defini dans le fichier de
 * proprietes, pour l'identifiant de log. Des informations peuvent provenir du contexte, le mapping (code switch, valeur switch) est realise par InfosStats.
 * 
 * ex : stats.service.format=[%ksession][%adresseip][%profildsi][%codeservice]
 * 
 * @see com.jsbsoft.jtf.stats.IStats
 * @see com.jsbsoft.jtf.stats.InfosStats
 * @author jbiard
 */
public class LogStats implements IStats {

	/** The Constant PROP_STATS. */
	public final static String PROP_STATS = "stats";

	/** The Constant PROP_STATS_ACTIVATION. */
	public final static String PROP_STATS_ACTIVATION = PROP_STATS + ".activation";

	/** The Constant PROP_STATS_FORMAT. */
	public final static String PROP_STATS_FORMAT = "format";

	/** The Constant ID_LOGGER. */
	public final static String ID_LOGGER = "stats";

	/** The _log. */
	private static final Logger LOG = LoggerFactory.getLogger(LogStats.class);

	/** The _log stats. */
	private static final Logger LOG_STATS = LoggerFactory.getLogger(ID_LOGGER);

	/** The _instance. */
	private static LogStats _instance;

	/** The _regexp pattern. */
	private static Pattern _regexpPattern;

	/** The activation stats. */
	private boolean activationStats;

	/** The infos stats. */
	private InfosStats infosStats;

	/**
	 * Indique si les statistiques sont activees ou non. A appeler systematiquement avant l'appel de log.
	 * 
	 * @return vrai si activees
	 */
	public static boolean isLogStats() {
		init();
		return _instance.activationStats;
	}

	/**
	 * Affiche un message de statistiques suivant le format defini pour l'identifiant de log (stats.id.format). Des informations peuvent provenir du contexte.
	 * 
	 * @param ctx
	 *            contexte
	 * @param idLog
	 *            identifiant de log (ex service)
	 * @param mapInfos
	 *            map dans laquelle on retrouve les valeurs des switchs
	 * 
	 * @throws ErreurTechnique
	 *             the erreur technique
	 */
	public static void log(final OMContext ctx, final String idLog, final Map<String, String> mapInfos) throws ErreurTechnique {
		init();
		if (!_instance.activationStats) {
			return;
		}
		final String formatLog = PropertyHelper.getCoreProperty(PROP_STATS + "." + idLog + "." + PROP_STATS_FORMAT);
		if (formatLog == null) {
			throw new ErreurTechnique(-1, "Identifiant de log de statistiques invalide.");
		}
		String switch_;
		String switchValeur;
		String message = formatLog;
		// recherche des switchs
		PatternMatcherInput pmiLigne;
		pmiLigne = new PatternMatcherInput(formatLog);
		final Perl5Matcher matcher = new Perl5Matcher();
		// boucle sur les occurrences d'switch
		while (matcher.contains(pmiLigne, _regexpPattern)) {
			switch_ = matcher.getMatch().toString();
			switchValeur = mapInfos.get(switch_);
			// recuperation d'infos depuis le contexte
			if (switchValeur == null) {
				switchValeur = _instance.infosStats.getInfos(ctx, switch_);
			}
			if (switchValeur == null) {
				switchValeur = "";
			}
			message = message.replaceFirst(switch_, switchValeur);
		}
		LOG_STATS.info(message);
	}

	/**
	 * Log.
	 * 
	 * @param idLog
	 *            the id log
	 * @param mapInfos
	 *            the map infos
	 * 
	 * @throws ErreurTechnique
	 *             the erreur technique
	 */
	public static void log(final String idLog, final Map<String, String> mapInfos) throws ErreurTechnique {
		log(null, idLog, mapInfos);
	}

	/**
	 * Inits the.
	 */
	private static void init() {
		if (_instance == null) {
			synchronized (LogStats.class) {
				// pour le 2eme thread
				if (_instance == null) {
					_instance = new LogStats();
				}
			}
		}
	}

	/**
	 * Constructeur prive assurant l'unicite du logger.
	 */
	private LogStats() {
		activationStats = "1".equals(PropertyHelper.getCoreProperty(PROP_STATS_ACTIVATION));
		if (!activationStats) {
			return;
		}
		final Perl5Compiler compiler = new Perl5Compiler();
		try {
			_regexpPattern = compiler.compile(SWITCH_REGEXP);
		} catch (final MalformedPatternException e) {
			LOG.error("Pattern invalide : " + _regexpPattern, e);
		}
		infosStats = new InfosStats();
	}

	/**
	 * The main method.
	 * 
	 * @param args
	 *            the arguments
	 * 
	 * @throws ErreurTechnique
	 *             the erreur technique
	 */
	public static void main(final String[] args) throws ErreurTechnique {
		final java.util.Map<String, String> mapInfosStats = new HashMap<>();
		mapInfosStats.put("%codeservice", "CODE_SERVICE");
		LogStats.log(null, ID_LOG_SERVICE, mapInfosStats);
		mapInfosStats.put("%coderub", "CODE_RUB");
		mapInfosStats.put("%modeconsult", "MODE");
		LogStats.log(null, ID_LOG_CONTRIB, mapInfosStats);
	}
}
