/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import javax.servlet.jsp.JspWriter;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexWriterConfig.OpenMode;
import org.apache.lucene.index.Term;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.util.Version;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.TikaMetadataKeys;
import org.apache.tika.parser.AutoDetectParser;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.sax.BodyContentHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.textsearch.util.IndexationHelper;
import com.kportal.cms.objetspartages.annotation.FicheAnnotationHelper;
import com.kportal.core.config.PropertyHelper;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.collaboratif.om.InfosEspaceCollaboratif;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheObjet;
import com.univ.objetspartages.om.FicheRattachementsSecondaires;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Rubriquepublication;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.StructureModele;
import com.univ.utils.FicheUnivMgr;
import com.univ.xhtml.HTMLParser;

/**
 * Indexation des documents joints, et des fiches Utilisation des projets open-source suivants: Lucène (http://jakarta.apache.org/lucene) PDFBox (http://www.pdfbox.org) POI
 * (http://jakarta.apache.org/poi) Textmining (http://www.textmining.org)
 *
 * @author malice
 */
public class Indexer {

	/** The Constant DOSSIER_GLOBAL. */
	public final static String DOSSIER_GLOBAL = "interne";

	/** The analyzer. */
	public static Analyzer analyzer = new FrenchAnalyzer(false);

	/** The repertoire indexation. */
	private static File repertoireIndexation = null;

	/** The writer. */
	private IndexWriter writer;

	/** The _instance. */
	private static Indexer _instance = null;

	private static final Logger LOGGER = LoggerFactory.getLogger(Indexer.class);

	private static final String DEFAULT_ETATS_NON_INDEXABLES = EtatFiche.APERCU+";"+EtatFiche.SAUVEGARDE_AUTO+";"+EtatFiche.A_SUPPRIMER;
	private Set<String> etatsNonIndexables = null;

	/**
	 * Gets the repertoire indexation.
	 *
	 * @return the repertoire indexation
	 */
	public static File getRepertoireIndexation() {
		if (repertoireIndexation == null) {
			try {
				final String indexDir = IndexationHelper.getWorkDirectory() + File.separator + Indexer.DOSSIER_GLOBAL;
				repertoireIndexation = new File(indexDir);
				if (!repertoireIndexation.exists()) {
					repertoireIndexation.mkdir();
				}
			} catch (final Exception e) {
				LOGGER.error("erreur lors de la création du dossier", e);
			}
		}
		return repertoireIndexation;
	}

	/**
	 * Constructeur, initialise les dossiers Lucène et les booléens.
	 *
	 * @throws Exception
	 *             the exception
	 */
	protected Indexer() throws Exception {
		getRepertoireIndexation();
		final String property = StringUtils.defaultString( PropertyHelper.getCoreProperty( "index.fiche.etatsnonindexables"),DEFAULT_ETATS_NON_INDEXABLES);
		etatsNonIndexables = new HashSet<>( Arrays.asList( property.split(";+") ) );
	}

	/**
	 * Gets the single instance of Indexer.
	 *
	 * @return single instance of Indexer
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Indexer getInstance() throws Exception {
		if (_instance == null) {
			synchronized (Indexer.class) {
				if (_instance == null) {
					_instance = new Indexer();
				}
			}
		}
		return _instance;
	}

	// méthode permettant d'initialiser les writers
	/**
	 * Delock.
	 *
	 * @param out
	 *            the out
	 *
	 * @throws Exception
	 *             the exception
	 */
	public void delock(final JspWriter out) throws Exception {
		final long timeOut = IndexWriterConfig.WRITE_LOCK_TIMEOUT;
		IndexWriterConfig.WRITE_LOCK_TIMEOUT = 60000;
		try {
			out.println("Ouverture de l'index...<br/>");
			out.flush();
			//si l'index existe, on supprime le répertoire
			initWriter();
		} catch (final Exception e) {
			out.println("Timeout atteint !<br/>");
			out.flush();
			out.print("Suppression du lock...");
			out.flush();
			final File ledossier = new File(System.getProperty("org.apache.lucene.lockdir", System.getProperty("java.io.tmpdir")));
			if (ledossier.exists() && ledossier.isDirectory() && ledossier.canWrite()) {
				final String[] listenoms = ledossier.list(new LockLuceneFilter());
				for (final String nomLock : listenoms) {
					final File fichier = new File(ledossier.getAbsolutePath() + File.separator + nomLock);
					if (fichier.exists()) {
						if (!fichier.delete()) {
							out.println("impossible de supprimer le lock " + nomLock);
						}
					}
				}
			}
			out.println("OK<br/>");
			out.flush();
			out.println("Réouverture de l'index<br/>");
			out.flush();
			initWriter();
		} finally {
			out.println("Optimisation de l'index<br/>");
			out.flush();
			optimisation();
			if (writer != null) {
				try {
					writer.close();
					writer = null;
				} catch (final CorruptIndexException e) {
					out.println("L'index est corrompu, vider le répertoire /textsearch et ré-indexer les contenus<br/>");
				}
			}
			IndexWriterConfig.WRITE_LOCK_TIMEOUT = timeOut;
			out.println("Fin<br/>");
			out.flush();
		}
	}

	// méthode permettant d'initialiser les writers
	/**
	 * Inits the writer.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void newWriter() throws Exception {
		final Directory directory = Searcher.getInstance().getDirectory(repertoireIndexation);
		final IndexWriterConfig conf = new IndexWriterConfig(Version.LUCENE_36, analyzer);
		conf.setOpenMode(OpenMode.CREATE_OR_APPEND);
		writer = new IndexWriter(directory, conf);
	}

	// méthode permettant d'initialiser les writers
	/**
	 * Release the writer.
	 *
	 * @throws Exception
	 *             the exception
	 */
	public void forceInitWriter() throws Exception {
		// suppression du fichier de lock
		final File fLock = new File(repertoireIndexation.getAbsolutePath() + File.separator + IndexWriter.WRITE_LOCK_NAME);
		if (fLock.exists()) {
			if (!fLock.delete()) {
				throw new ErreurApplicative("impossible de supprimer le lock sur l'indexer");
			}
		}
		// reinitialisation du writer
		initWriter();
	}

	/**
	 * Close writer.
	 *
	 * @throws Exception
	 *             the exception
	 */
	public void closeWriter() throws Exception {
		if (writer != null) {
			writer.close();
			writer = null;
		}
	}

	/**
	 * Indexe une fiche dans l'index global et dans l'index de l'objet (si cette option est active dans le jtf) synchronized pour gérer l'accès aux fichiers qui gèrent l'index.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _fiche
	 *            the _fiche
	 * @param deleteBefore
	 *            the delete before
	 *
	 * @throws Exception
	 *             the exception
	 */
	public synchronized void indexFiche(final OMContext _ctx, final FicheUniv _fiche, final boolean deleteBefore) throws Exception {
		if (_fiche == null || !FicheAnnotationHelper.isIndexable(_fiche)) {
			if ( _fiche!=null ) {
				LOGGER.debug("_fiche {} id:[{}] non indexée (fiche non indexable)",_fiche.getClass().getSimpleName(),_fiche.getIdFiche());
			}
			return;
		}
		//Filtre sur les états
		if ( etatsNonIndexables!=null && etatsNonIndexables.contains(_fiche.getEtatObjet()) ) {
			LOGGER.debug("_fiche {} id:[{}] non indexée (état non indexable)",_fiche.getClass().getSimpleName(),_fiche.getIdFiche());
			return;
		}
		// filtre les fiches soumises à dsi par leur rubrique
		final String codeRubrique = _fiche.getCodeRubrique();
		final ArrayList<String> lstRubriqueRestriction = creerLstRubriqueRestriction();
		if (lstRubriqueRestriction.contains(codeRubrique)) {
			return;
		}
		// filtre les structures non visibles en front
		if (_fiche instanceof StructureModele) {
			final Metatag meta = FicheUnivMgr.lireMeta(_fiche);
			// on indexe pas uniquement si le parametre jtf structure.inTree est non active
			if (meta.getMetaInTree().equals("0") && !"1".equals(PropertyHelper.getCoreProperty("structure.inTree"))) {
				return;
			}
		}
		// creation de l'index de la fiche
		final Index index = creerIndex(_ctx, _fiche);
		// creation document à partir de l'index
		final Document document = index.creerDocument();
		if (deleteBefore) {
			// update = delete + add
			final Term term = new Term(Index.IDENTIFIANT_UNIQUE, _fiche.getClass().getName() + _fiche.getIdFiche());
			writer.updateDocument(term, document);
		} else {
			// ajout du document dans le writer
			writer.addDocument(document);
		}
	}

	/**
	 * BA 2005/07/26.
	 *
	 */
	public void optimisation() {
		try {
			if (writer != null) {
				writer.forceMerge(1);
			}
		} catch (final Exception e) {
			LOGGER.error("Exception lors de l'optimisation de l'index", e);
		}
	}

	/**
	 * Delete fiche.
	 *
	 * @param _fiche
	 *            the _fiche
	 *
	 * @return true, if successful
	 *
	 * @throws Exception
	 *             the exception
	 */
	public boolean deleteFiche(final FicheUniv _fiche) throws Exception {
		if (_fiche == null || !FicheAnnotationHelper.isIndexable(_fiche)) {
			if ( _fiche!=null ) {
				LOGGER.debug("fiche {} id:[{}] non desindexée (fiche non indexable)",_fiche.getClass().getSimpleName(),_fiche.getIdFiche());
			}
			return true;
		}
		boolean res = false;
		boolean init = false;
		try {
			final Term term = new Term(Index.IDENTIFIANT_UNIQUE, _fiche.getClass().getName() + _fiche.getIdFiche());
			if (writer == null) {
				initWriter();
				init = true;
			}
			writer.deleteDocuments(term);
			res = true;
		} catch (final Exception e) {
			LOGGER.error("Impossible de supprimer l'index de la fiche id=" + _fiche.getIdFiche() +" ("+_fiche.getClass().getSimpleName()+")", e);
		} finally {
			if (init && writer != null) {
				closeWriter();
			}
		}
		return res;
	}

	/**
	 * Suppression définitive d'une fiche dans l'index (appelée par ScanSite).
	 *
	 * @param type
	 *            the type
	 *
	 * @return true, if delete fiche
	 *
	 * @throws Exception
	 *             the exception
	 */
	public boolean deleteFicheByType(final String type) throws Exception {
		boolean res = false;
		boolean init = false;
		try {
			final Term term = new Term(Index.OBJET, ReferentielObjets.getNomObjet(ReferentielObjets.getCodeObjet(type)));
			if (writer == null) {
				initWriter();
				init = true;
			}
			writer.deleteDocuments(term);
			res = true;
		} catch (final Exception e) {
			LOGGER.error("Impossible de supprimer l'index des fiches " + type, e);
			res = false;
		} finally {
			if (init && writer != null) {
				closeWriter();
			}
		}
		return res;
	}

	/**
	 * appelée par ajouteDocument : On supprime les crochets sinon, on ne pourra pas faire une recherche. En effet, il n'est pas possible de faire une wildcardquery (recherche avec
	 * *, ??) et en + des caracteres speciaux comme le crochet
	 *
	 * @param _chaine
	 *            the _chaine
	 *
	 * @return the string
	 */
	private String supprimeCrochets(final String _chaine) {
		String chaine = _chaine;
		int idx = chaine.indexOf("[");
		while (idx != -1) {
			chaine = chaine.substring(0, idx) + chaine.substring(idx + 1);
			idx = chaine.indexOf("[");
		}
		idx = chaine.indexOf("]");
		while (idx != -1) {
			chaine = chaine.substring(0, idx) + chaine.substring(idx + 1);
			idx = chaine.indexOf("]");
		}
		return chaine;
	}

	/**
	 * Creer lst rubrique restriction.
	 *
	 * @return the array list
	 *
	 * @throws Exception
	 *             the exception
	 */
	private ArrayList<String> creerLstRubriqueRestriction() throws Exception {
		final ArrayList<String> lstRubriqueRestriction = new ArrayList<>();
		final String chaineCodeRubrique = PropertyHelper.getCoreProperty("lucene.restriction_rubrique");
		if (chaineCodeRubrique != null && chaineCodeRubrique.trim().length() > 0) {
			final String[] lstCodeRubrique = chaineCodeRubrique.trim().split(";", -2);
			for (final String codeRubrique : lstCodeRubrique) {
				final InfosRubriques info = Rubrique.renvoyerItemRubrique(codeRubrique);
				if (info != null && info.getCode().length() > 0) {
					lstRubriqueRestriction.add(info.getCode());
					for (InfosRubriques infosRubriques : info.getListeSousRubriquesTousNiveaux()) {
						final String cr = infosRubriques.getCode();
						if (!lstRubriqueRestriction.contains(cr)) {
							lstRubriqueRestriction.add(cr);
						}
					}
				}
			}
		}
		return lstRubriqueRestriction;
	}

	/**
	 * Creer index.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _fiche
	 *            the _fiche
	 *
	 * @return the index
	 *
	 * @throws Exception
	 *             the exception
	 */
	private Index creerIndex(final OMContext _ctx, final FicheUniv _fiche) throws Exception {
		final String nomObjet = ReferentielObjets.getNomObjet(ReferentielObjets.getCodeObjet(_fiche));
		final String idunique = _fiche.getClass().getName() + _fiche.getIdFiche();
		final Index index = new Index();
		//on réajoute la fiche
		index.setIdentifiantUnique(idunique);
		//nom de la classe
		index.setObjet(nomObjet);
		//id de la fiche
		index.setIdFiche(_fiche.getIdFiche().toString());
		//code de la fiche
		index.setCodeFiche(_fiche.getCode());
		// code redacteur
		index.setCodeRedacteur(_fiche.getCodeRedacteur());
		//langue
		index.setLangue(_fiche.getLangue());
		//etat de la fiche
		index.setEtatFiche(_fiche.getEtatObjet());
		//titre de la fiche
		index.setTitle(_fiche.getLibelleAffichable());
		//keywords de la fiche
		String metaKeywords = "";
		if (_fiche.getMetaKeywords().length() > 0) {
			metaKeywords = _fiche.getMetaKeywords() + " ";
		}
		metaKeywords += _fiche.getLibelleAffichable();
		index.setKeywords(RechercheFmt.formaterTexteRecherche(metaKeywords, true));
		//keywords de la fiche
		metaKeywords = "";
		if (_fiche.getMetaDescription().length() > 0) {
			metaKeywords = _fiche.getMetaDescription() + " ";
		}
		metaKeywords += _fiche.getLibelleAffichable();
		index.setDescription(RechercheFmt.formaterTexteRecherche(metaKeywords, true));
		// structure de la fiche
		final List<String> listeStructures = new ArrayList<>();
		listeStructures.add(_fiche.getCodeRattachement());
		if (_fiche instanceof FicheRattachementsSecondaires) {
			final String[] codesRattachementsAutres = StringUtils.split(((FicheRattachementsSecondaires) _fiche).getCodeRattachementAutres(), ";");
			listeStructures.addAll(Arrays.asList(codesRattachementsAutres));
		}
		index.setCodeRattachement(listeStructures);
		//date de modification
		if (Formateur.estSaisie(_fiche.getDateModification())) {
			index.setLastModified(new java.text.SimpleDateFormat("yyyyMMdd").format(_fiche.getDateModification()));
		}
		// date de mise en ligne
		final Metatag meta = FicheUnivMgr.lireMeta(_fiche);
		Date dateMiseEnLigne = meta.getMetaDateMiseEnLigne();
		if (!Formateur.estSaisie(dateMiseEnLigne)) {
			dateMiseEnLigne = new Date(0);
		}
		index.setMiseEnLigne(new java.text.SimpleDateFormat("yyyyMMdd").format(dateMiseEnLigne));
		/* DSI */
		String activationDsi = PropertyHelper.getCoreProperty("dsi.activation");
		if (activationDsi == null) {
			activationDsi = "0";
		}
		//on stocke également les infos de restriction sur la fiche
		if ((activationDsi.equals("1")) && (_fiche instanceof DiffusionSelective)) {
			final DiffusionSelective fDsi = (DiffusionSelective) _fiche;
			index.setModeRestriction(fDsi.getDiffusionModeRestriction());
			//On supprime les crochets sinon, on ne pourra pas faire une recherche.
			//En effet, il n'est pas possible de faire une wildcardquery (recherche avec *, ??)
			// et en + des caracteres speciaux comme le crochet
			String pVise = fDsi.getDiffusionPublicVise();
			if (StringUtils.isNotEmpty(pVise)) {
				pVise = supprimeCrochets(pVise);
				index.setPublicVise(";" + pVise + ";");
			}
			String pViseRestriction = fDsi.getDiffusionPublicViseRestriction();
			if (StringUtils.isNotEmpty(pViseRestriction)) {
				pViseRestriction = supprimeCrochets(pViseRestriction);
				index.setPublicViseRestriction(";" + pViseRestriction + ";");
			}
		} else {
			//si ce type de fiche n'a pas de DSI, on met le mode à 0. Sinon, elle n'apparaitra pas dans les resultats
			//car la recherche inclut la DSI, et on ne connait pas le type de fiche à l'avance
			index.setModeRestriction("0");
		}
		// indexation des rubriques de publication
		FicheUniv fichePouIndexRubrique = _fiche;
		if (_fiche instanceof FicheObjet) {
			fichePouIndexRubrique = ((FicheObjet) _fiche).renvoyerFicheParente();
		}
		final List<String> listeRubriques = new ArrayList<>();
		listeRubriques.add(fichePouIndexRubrique.getCodeRubrique());
		listeRubriques.addAll(Rubriquepublication.getListeRubriquesPublication(_ctx, fichePouIndexRubrique));
		if (FicheUnivMgr.isFicheCollaborative(fichePouIndexRubrique)) {
			final InfosEspaceCollaboratif infosEspace = Espacecollaboratif.renvoyerItemEspace(((DiffusionSelective) fichePouIndexRubrique).getDiffusionPublicViseRestriction());
			listeRubriques.add(infosEspace.getCodeRubrique());
		}
		index.setCodeRubrique(listeRubriques);
		final InfosSite infosSite = Site.determinerSiteRubrique(_fiche.getCodeRubrique());
		if (infosSite != null) {
			index.setCodeSite(infosSite.getAlias());
		} else {
			index.setCodeSite("");
		}
		index.setUrl("");
		final StringBuilder toutcontenu = new StringBuilder(_fiche.getFullTextString());
		/*TODO 6.0 indexation des plugins ed contenus)*/
		toutcontenu.append(" ").append(_fiche.getLibelleAffichable());
		if (_fiche.getMetaKeywords().length() > 0) {
			toutcontenu.append(" ").append(_fiche.getMetaKeywords());
		}
		if (_fiche.getMetaDescription().length() > 0) {
			toutcontenu.append(" ").append(_fiche.getMetaDescription());
		}
		toutcontenu.append(" ").append(Structure.getLibelleAffichable(_fiche.getCodeRattachement(), _fiche.getLangue()));
		toutcontenu.append(" ").append(_fiche.getContenuEncadre());
		final HTMLParser parser = new HTMLParser();
		parser.setInputHtml(RechercheFmt.formaterEnHTML(_ctx, toutcontenu.toString()));
		//contenu de la fiche
		index.setContent(RechercheFmt.formaterTexteRecherche(parser.extractString(false), false, false));
		StringBuffer contenuFichiers = new StringBuffer();
		int poidsTotal = 0;
		final Vector<Ressource> lstFichier = Ressource.getListeTotale(_fiche);
		for (final Ressource fichiergw : lstFichier) {
			if (fichiergw.getMedia() != null) {
				poidsTotal += fichiergw.getMedia().getPoids();
				// au dessus de 100Mo de données, on indexe plus les pièces jointes pour des questions de performance
				if (poidsTotal > 100000) {
					break;
				}
				final String path = fichiergw.getPathAbsolu();
				File fichierJoint = new File(path);
				if (fichierJoint.exists()) {
					try {
						final StringBuffer contenuFichiersTemp = indexFile(fichierJoint, fichiergw.getMedia().getFormat());
						if (contenuFichiersTemp != null) {
							if (contenuFichiers.length() < contenuFichiersTemp.length()) {
								contenuFichiers = contenuFichiersTemp.append(contenuFichiers);
							} else {
								contenuFichiers = contenuFichiers.append(contenuFichiersTemp);
							}
						}
						contenuFichiers.append(" ").append(fichiergw.getFullTextString());
					} catch (final Exception e) {
						LOGGER.error("Exception lors de l'indexation du fichier " + fichierJoint.getName(), e);
					}
				}
			}
		}
		index.setContentFile(RechercheFmt.formaterTexteRecherche(contenuFichiers.toString(), false, false));
		return index;
	}

	/**
	 * Indexe le fichier joint.
	 *
	 * @param _f
	 *            the _f
	 * @param format
	 *            the format
	 *
	 * @return the string buffer
	 *
	 * @throws Exception
	 *             the exception
	 */
	private StringBuffer indexFile(final File _f, String format) throws Exception {
		if (_f == null) {
			return null;
		}
		if (format == null) {
			format = "";
		}
		if (format.contains("image/")) {
			return null;
		}
		final String filename = _f.getName().toLowerCase();
		final FileInputStream fis = new FileInputStream(_f);
		final Parser parser = new AutoDetectParser();
		final Metadata metadata = new Metadata();
		metadata.add(TikaMetadataKeys.RESOURCE_NAME_KEY, filename);
		final StringWriter writer = new StringWriter();
		parser.parse(fis, new BodyContentHandler(writer), metadata, new ParseContext());
		fis.close();
		return writer.getBuffer();
	}

	/**
	 * Gets the writer.
	 *
	 * @return the writer
	 */
	public IndexWriter getWriter() {
		return writer;
	}

	public void deleteAll() throws Exception {
        try {
            initWriter();
            getWriter().deleteAll();
            closeWriter();
        } catch (IOException ioe) {
            //L'index est certainement corrompu - on va nettoyer manuellement le répertoire
            LOGGER.warn("Suppression manuelle des indexes Lucene : ", ioe );
            File indexDir = Indexer.getRepertoireIndexation();
            if ( indexDir!=null && indexDir.exists() && indexDir.isDirectory() && indexDir.canWrite()) {
                File[] indexFiles = (File[])ArrayUtils.nullToEmpty(indexDir.listFiles());
                for(File indexFile : indexFiles) {
                    if ( indexFile.isFile() && indexFile.canWrite() ) {
                        indexFile.delete();
                    }
                }
            }
            
        }
	}

	protected void initWriter() throws Exception {
		try {
			// initialisation des writers
			newWriter();
		} catch (final LockObtainFailedException e) {
			forceInitWriter();
		}
	}
}
