/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Vector;
import java.util.List;

import com.univ.objetspartages.bean.MetatagBean;
import net.sf.ehcache.Cache;
import org.quartz.JobExecutionContext;

import com.jsbsoft.jtf.webutils.ContextePage;
import com.kportal.core.cluster.ClusterHelper;
import com.kportal.core.config.PropertyHelper;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.kportal.scheduling.spring.quartz.SequenceJob;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUtil;
import com.univ.utils.sql.Operateur;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * The Class IndexerScheduler.
 */
public class IndexerJob extends LogReportJob {

	public static final String FIRST_INDEX = "firstIndex";
	
	private static final int FICHE_SUPPRIMEE = 1;
	private static final int FICHE_INDEXEE = 2;

	/** The delay. */
	private Long delay = new Long(300000);

	/** The first index. */
	private boolean firstIndex = false;

	private int indexerFiche(Indexer indexer, Metatag meta) {
		int result = 0;
		try {
			ContextePage ctxFiche = ContexteUtil.setContexteSansRequete();

			FicheUniv ficheUniv = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(meta.getMetaCodeObjet()));
			ficheUniv.setCtx(ctxFiche);
			ficheUniv.init();
			ficheUniv.setIdFiche(meta.getMetaIdFiche());
			ficheUniv.retrieve();
			logger.info("Traitement de la fiche : " + meta.getMetaLibelleFiche() + ", objet=" + meta.getMetaLibelleObjet() + ", id=" + meta.getMetaIdFiche());
			if ( EtatFiche.A_SUPPRIMER.getEtat().equals(ficheUniv.getEtatObjet()) ) {
				indexer.deleteFiche(ficheUniv);
				result = FICHE_SUPPRIMEE;
				logger.debug("Fiche supprimée");
			} else {
				indexer.indexFiche(ctxFiche, ficheUniv, true);
				result = FICHE_INDEXEE;
				logger.debug("Fiche indexée");
			}
		} catch (final Exception e) {
			logger.error("Exception lors du traitement de la fiche : objet=" + meta.getMetaLibelleObjet() + ", id=" + meta.getMetaIdFiche(), e);
		} finally {
			ContexteUtil.releaseContexteUniv();
		}
		return result;
	}
	
	/**
	 * Run.
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void run() {
		final ContextePage ctx = new ContextePage("");
		int count = 0;
		int countSuppr = 0;
		try {
			if (firstIndex) {
				ctx.setConnectionLongue(true);
			}
			final CacheIndexerManager cacheIndexerManager = CacheIndexerManager.getInstance();
			final Cache fichesSupprimees = cacheIndexerManager.getCache();
			final List cacheKeys = new ArrayList(fichesSupprimees.getKeys());

			final ClauseWhere where = new ClauseWhere();
			if (!firstIndex) {
				final GregorianCalendar gc = new GregorianCalendar();
				gc.setTime(new Date(System.currentTimeMillis()));
				gc.setTimeInMillis(gc.getTimeInMillis() - delay);
				final SimpleDateFormat formatDateTime = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				where.setPremiereCondition(ConditionHelper.genericCondition("DATE_FORMAT(META_DATE_OPERATION, '%Y-%m-%d %H:%i:%s')", TypeOperande.NON_ECHAPABLE,
						"'" + formatDateTime.format(gc.getTime()) + "'", TypeOperande.NON_ECHAPABLE, Operateur.GREATER_EQUALS));
			} else {
				where.setPremiereCondition(ConditionHelper.genericCondition(1, TypeOperande.INTEGER, 1, TypeOperande.INTEGER, Operateur.EQUALS));
			}
			final Vector<String> objetsExclus = Chaine.getVecteurPointsVirgules(PropertyHelper.getCoreProperty("lucene.restriction_objet"));
			if (objetsExclus.size() > 0) {
				where.and(ConditionHelper.notIn("META_CODE_OBJET", objetsExclus));
			}
			final Metatag meta = new Metatag();
			meta.setCtx(ctx);
			meta.init();
			boolean initWriter = false;
			final int nbMeta = meta.select(where.formaterSQL());
			if (nbMeta > 0  || cacheKeys.size()>0 ) {
				logger.info(nbMeta + " fiche(s) à traiter");
				final Indexer indexer = Indexer.getInstance();
				try {
					// le writer est disponible
					if (indexer.getWriter() == null) {
						if (firstIndex) {
							indexer.deleteAll();
						}
						indexer.initWriter();
						initWriter = true;

						for(Object key : cacheKeys ) {
							MetatagBean mb = (MetatagBean)cacheIndexerManager.get(key);
							if ( mb!=null ) {
								FicheUniv ficheUniv = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(mb.getMetaCodeObjet()));
								ficheUniv.setIdFiche(mb.getMetaIdFiche());
								indexer.deleteFiche(ficheUniv);
								cacheIndexerManager.evict(key);
							}
						}

						while (meta.nextItem()) {
							int result = indexerFiche(indexer, meta);
							if ( result==FICHE_SUPPRIMEE ) {
								countSuppr++;
							} else if (result==FICHE_INDEXEE) {
								count++;
							}
						}
					} else {
						logger.error("Writer en cours d'exécution...");
					}
				} catch (final Exception e) {
					logger.error("Exception lors de l'indexation des fiches : " + e.getMessage(), e);
				} finally {
					if (initWriter) {
						indexer.closeWriter();
						// rechargement du singleton Searcher
						ClusterHelper.refresh(Searcher.getInstance(), null);
					}
				}
				// traitement du fichier de report
				if (firstIndex) {
					firstIndex = false;
					logger.info("Nombre de fiche(s) indexée(s) : " + count);
				} else {
					logger.info("Nombre de fiche(s) indexée(s) : " + count);
					logger.info("Nombre de fiche(s) supprimée(s) de l'index : " + countSuppr);
				}
			}
		} catch (final Exception e) {
			logger.error("Exception dans le job d'indexation", e);
		} finally {
			ctx.release();
		}
	}

	@Override
	public void perform() {
		run();
	}

	@Override
	public void init(final JobExecutionContext jobCtx) {
		firstIndex = jobCtx.getMergedJobDataMap().getBooleanValue(FIRST_INDEX);
		if (!firstIndex) {
			Date date1;
			Date date2;
			// les paramètres sont passées depuis la sequence
			if (jobCtx.getMergedJobDataMap().get(SequenceJob.PREVIOUS_FIRE_TIME) != null) {
				date1 = (Date) jobCtx.getMergedJobDataMap().get(SequenceJob.PREVIOUS_FIRE_TIME);
				date2 = (Date) jobCtx.getMergedJobDataMap().get(SequenceJob.FIRE_TIME);
			} else {
				date1 = (Date) jobCtx.getMergedJobDataMap().get(SequenceJob.FIRE_TIME);
				date2 = (Date) jobCtx.getMergedJobDataMap().get(SequenceJob.NEXT_FIRE_TIME);
			}
			if (date1 != null && date2 != null) {
				delay = date2.getTime() - date1.getTime();
			}
		}
	}

	@Override
	public String getLogFileName() {
		String res = IndexerJob.class.getSimpleName().toLowerCase();
		if (firstIndex) {
			res += "-firstindex";
		}
		return res;
	}
}
