/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiReader;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.store.SimpleFSLockFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.core.config.PropertyHelper;

import fr.kosmos.cluster.api.Cluster;
import fr.kosmos.cluster.api.MessageHandler;

/**
 * The Class Searcher.
 */
public class Searcher implements MessageHandler<Serializable> {

	private static final String ID_BEAN = "searcher";

	/** The lst recherche. */
	private ArrayList<Recherche> lstRecherche = null;

	/** The analyzer. */
	private final FrenchAnalyzer analyzer = new FrenchAnalyzer(false);

	/** The reader. */
	private IndexReader indexReader = null;

	/** The searcher. */
	private IndexSearcher indexSearcher = null;

	/** The is nfs. */
	private boolean isNFS = false;

	/** The need to re open. */
	private boolean needToReOpen = false;

	/** The Constant MAX_RESULT. */
	public final static int MAX_RESULT = 10000;

	private static final Logger LOG = LoggerFactory.getLogger(Searcher.class);

	private Cluster cluster;

	public void init() {
		cluster.registerService(this);
		start();
	}

	/**
	 * Gets the single instance of Searcher.
	 *
	 * @return single instance of Searcher
	 *
	 * 
	 */
	public static Searcher getInstance() {
		return (Searcher) ApplicationContextManager.getCoreContextBean(ID_BEAN);
	}

	/**
	 * Check init.
	 */
	private void start() {
		if (lstRecherche == null) {
			lstRecherche = new ArrayList<>();
			int i = 1;
			String param;
			while ((param = PropertyHelper.getCoreProperty("lucene.recherche" + i)) != null) {
				try {
					//Vérification qu'on implémente Recherche
					final Class<?> recherche = Class.forName("com.jsbsoft.jtf.textsearch.Recherche");
					if (!recherche.isAssignableFrom(Class.forName(param))) {
						throw new Exception("La classe " + param + " n'implémente pas com.jsbsoft.jtf.textsearch.Recherche");
					}
					lstRecherche.add((Recherche) Class.forName(param).newInstance());
				} catch (final Exception e) {
					LOG.error("Erreur dans la configuration de la recherche : " + param, e);
				}
				i++;
			}
			final ArrayList<File> lstRepertoire = new ArrayList<>();
			for (Recherche aLstRecherche : lstRecherche) {
				lstRepertoire.addAll(aLstRecherche.getLstRepertoireRecherche());
			}
			final ArrayList<IndexReader> lstReader = new ArrayList<>(lstRepertoire.size());
			for (final File f : lstRepertoire) {
				try {
					if (f.exists()) {
						final Directory directory = getDirectory(f);
						if (IndexReader.indexExists(directory)) {
							lstReader.add(IndexReader.open(directory));
						} else {
							LOG.warn("Index de recherche inexistant pour le path " + f.getAbsolutePath());
						}
					}
				} catch (final Exception e) {
					LOG.error("Erreur lors de la création de l'index searcher pour le répertoire : " + f.getAbsolutePath(), e);
				}
			}
			if (lstReader.size() > 0) {
				try {
					indexReader = new MultiReader(lstReader.toArray(new IndexReader[lstReader.size()]));
					indexSearcher = new IndexSearcher(indexReader);
				} catch (final Exception e) {
					LOG.error("Erreur dans la configuration de la recherche", e);
				}
			} else {
				LOG.warn("Aucun index de recherche trouvé");
			}
			isNFS = "nfs".equalsIgnoreCase(PropertyHelper.getCoreProperty("recherche.directory"));
		}
	}


	private synchronized void testReader() {
		if (indexReader == null) {
			forcerInit();
		}
		if (needToReOpen) {
			reOpen();
		}
	}

	/**
	 * Force le rechargement du searcher / reader pour la recherche.
	 */
	private synchronized void reOpen() {
		if (needToReOpen) {
			try {
				final IndexReader newReader = IndexReader.openIfChanged(indexReader);
				if (newReader != indexReader) {
					closeAll();
				}
				indexReader = newReader;
				indexSearcher = new IndexSearcher(indexReader);
			} catch (final Exception e) {
				LOG.error("erreur sur l'indexer", e);
			}
			needToReOpen = false;
		}
	}

	/**
	 * Force le rechargement du searcher / reader pour la recherche.
	 */
	public void forcerInit() {
		lstRecherche = null;
		try {
			closeAll();
		} catch (final IOException e) {
			LOG.error("impossible de fermer les index", e);
		}
		start();
		needToReOpen = false;
	}

	/**
	 * Gets the directory.
	 *
	 * @param f
	 *            the f
	 * @return the directory
	 * @throws IOException
	 *             the exception
	 */
	public Directory getDirectory(final File f) throws IOException {
		Directory directory;
		if (isNFS) {
			directory = new SimpleFSDirectory(f, new SimpleFSLockFactory());
		} else {
			directory = FSDirectory.open(f);
		}
		return directory;
	}

	public void closeAll() throws IOException {
		if (indexReader != null) {
			indexReader.close();
			indexReader = null;
		}
		if (indexSearcher != null) {
			indexSearcher.close();
			indexSearcher = null;
		}
	}

	public IndexSearcher getSearcher() {
		testReader();
		return indexSearcher;
	}

	public void setSearcher(final IndexSearcher searcher) {
		this.indexSearcher = searcher;
	}

	public IndexReader getIndexReader() {
		testReader();
		return indexReader;
	}

	public void setIndexReader(final IndexReader indexReader) {
		this.indexReader = indexReader;
	}

	/**
	 * @deprecated Préférer la méthode {@link #getSearcher()} qui check l'existence de l'IndexSearcher et son état pour en retourner un utilisable.
	 */
	@Deprecated
	public IndexSearcher getIndexSearcher() {
		return indexSearcher;
	}

	public void setIndexSearcher(final IndexSearcher indexSearcher) {
		this.indexSearcher = indexSearcher;
	}

	public FrenchAnalyzer getAnalyzer() {
		return analyzer;
	}

	@Override
	public void handleMessage(final Serializable message) {
		needToReOpen = Boolean.TRUE;
	}

	public Cluster getCluster() {
		return cluster;
	}

	public void setCluster(final Cluster cluster) {
		this.cluster = cluster;
	}
}
