/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch.sitesdistants;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.Vector;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexWriterConfig.OpenMode;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.Version;
import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.Perl5Compiler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.lang.CharEncoding;
import com.jsbsoft.jtf.textsearch.Index;
import com.jsbsoft.jtf.textsearch.Indexer;
import com.jsbsoft.jtf.textsearch.RechercheFmt;
import com.jsbsoft.jtf.textsearch.Searcher;
import com.jsbsoft.jtf.webutils.ContextePage;
import com.kportal.core.cluster.ClusterHelper;
import com.kportal.core.config.PropertyHelper;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.Site;
import com.univ.utils.Chaine;
import com.univ.xhtml.HTMLParser;

/**
 * Singleton prennant en charge l'aspiration de sites distants, et l'indexation des pages. Code inspire de http://moguntia.ucd.ie/programming/webcrawler/ On a un index par site et
 * par langue. Pour eviter de referencer des pages qui n'existent plus, on repart de zéro.
 * 
 * @author jbiard
 */
public class IndexeurSitesDistants {

	/** The PARA m_ jt f_ nbthreads. */
	public static String PARAM_JTF_NBTHREADS = "lucene.nb_threads_aspiration";

	/** The PARA m_ jt f_ timeout. */
	public static String PARAM_JTF_TIMEOUT = "lucene.timeout_threads";

	/** */
	public static String PARAM_JTF_MAXSIZE_HTMLQUEUE = "lucene.max_size_queue";

	/** */
	public static String PARAM_JTF_TIME_SLEEP = "lucene.time_sleep";

	/** The N b_ ma x_ thread s_ defaut. */
	public static int NB_MAX_THREADS_DEFAUT = 4;

	/** The TIMEOU t_ defaut. */
	public static int TIMEOUT_DEFAUT = 10; // en s

	private Logger logger = LoggerFactory.getLogger(IndexeurSitesDistants.class);

	/** The _instance. */
	private static IndexeurSitesDistants _instance;

	/** The queue sites. */
	private final QueueSiteAIndexer queueSites;

	/** The n niveau courant profondeur. */
	private int nNiveauCourantProfondeur;

	/** The n niveau max profondeur. */
	private int nNiveauMaxProfondeur;

	/** The n nb threads courants. */
	private int nNbThreadsCourants;

	/** The n nb max threads. */
	private int nNbMaxThreads;

	/** The n timeout. */
	private int nTimeout;

	/** Le nombre max de page html contenu dans la queue de page html */
	private int maxSizeQueue;

	/** Le temps en millisecond d'attente du thread lorsque la queue de page est pleinne */
	private long timeToSleep;

	/** The b fin indexation. */
	private boolean bFinAspiration, bFinIndexation;

	/** The queue url. */
	private URLQueue queueURL;

	/** The queue html. */
	private QueueFluxHTML queueHTML;

	/** The pattern url refus. */
	private Pattern patternURLAcceptation, patternURLRefus;

	/** The sz url site. */
	private String szUrlSite;

	/** The asz url disallows. */
	private String[] aszUrlDisallows;

	public void setLogger(final Logger logger) {
		this.logger = logger;
	}

	/**
	 * Renvoie l'unique instance (singleton).
	 * 
	 * @return l'instance
	 */
	public synchronized static IndexeurSitesDistants getInstance() {
		if (_instance == null) {
			_instance = new IndexeurSitesDistants();
		}
		return _instance;
	}

	/**
	 * Lance l'indexation d'un site.
	 * 
	 * @param site
	 *            site a indexer
	 * @throws Exception
	 */
	public void indexeSite(final Site site, final boolean join) throws Exception {
		try {
			queueSites.push(site);
			lanceIndexations(join);
		} catch (final InterruptedException e) {
			logger.error(e.getMessage(), e);
		} finally {
			// rechargement du singleton Searcher
			ClusterHelper.refresh(Searcher.getInstance(), null);
		}
	}

	/**
	 * Retourne pour chacun des sites son etat.
	 * 
	 * @return map dont la clef est le libelle du site
	 * 
	 * @see EtatsSitesIndexation
	 */
	public Map<Long, String> getEtatsSites() {
		return queueSites.getEtats();
	}

	/**
	 * Indexe l'ensemble des sites en base.
	 * 
	 * @param ctx
	 *            contexte
	 * 
	 * @throws Exception
	 *             levee si probleme de chargement
	 */
	public void indexeSites(final ContextePage ctx) throws Exception {
		final Site site = new Site();
		site.init();
		site.setCtx(ctx);
		site.select("");
		while (site.nextItem()) {
			indexeSite((Site) site.clone(), true);
		}
	}

	/**
	 * Indexation en cours.
	 * 
	 * @return true, if successful
	 */
	public boolean indexationEnCours() {
		return !bFinIndexation;
	}

	/**
	 * Gets the nb threads courants.
	 * 
	 * @return the nb threads courants
	 */
	public int getNbThreadsCourants() {
		return nNbThreadsCourants;
	}

	/**
	 * Gets the nb max threads.
	 * 
	 * @return the nb max threads
	 */
	public int getNbMaxThreads() {
		return nNbMaxThreads;
	}

	/**
	 * Gets the niveau courant profondeur.
	 * 
	 * @return the niveau courant profondeur
	 */
	public int getNiveauCourantProfondeur() {
		return nNiveauCourantProfondeur;
	}

	/**
	 * Gets the niveau max profondeur.
	 * 
	 * @return the niveau max profondeur
	 */
	public int getNiveauMaxProfondeur() {
		return nNiveauMaxProfondeur;
	}

	/**
	 * Indique si une url doit etre aspiree ou non (suivant le robots.txt)
	 * 
	 * @param szUrl
	 *            url a valider ou non
	 * 
	 * @return vrai si l'url est acceptable
	 */
	public boolean accepteUrlRobots(final String szUrl) {
		if (aszUrlDisallows == null) {
			return true;
		}
		for (final String aszUrlDisallow : aszUrlDisallows) {
			if (szUrl.indexOf(aszUrlDisallow) != -1) {
				logger.debug("\t!!URL refusee (robots.txt) : " + szUrl);
				return false;
			}
		}
		return true;
	}

	/**
	 * Renvoie le pattern correspondant a l'expression reguliere validant les url.
	 * 
	 * @return pattern, null si non definie
	 */
	public Pattern getPatternURLAcceptation() {
		return patternURLAcceptation;
	}

	/**
	 * Renvoie le pattern correspondant a l'expression reguliere refusant les url.
	 * 
	 * @return pattern, null si non definie
	 */
	public Pattern getPatternURLRefus() {
		return patternURLRefus;
	}

	/**
	 * Lance le thread dedie a l'indexation. Permet un traitement en tâche de fond.
	 * 
	 * @throws InterruptedException
	 */
	protected void lanceIndexations(final boolean join) throws InterruptedException {
		if (bFinIndexation) {
			bFinIndexation = false;
			final Thread threadIndexation = new Thread() {

				@Override
				public void run() {
					int nbPages = 0;
					Date dateDeb = null;
					Date dateFin = null;
					for (Site site = queueSites.pop(); site != null; site = queueSites.pop()) {
						dateDeb = new Date();
						logger.info("%%%% Début de l'indexation de " + site.getUrl() + " à " + dateDeb);
						try {
							nbPages = indexe(site);
						} catch (final Exception e) {
							logger.error("erreur lors de l'indexation", e);
							nbPages = 0;
						}
						queueSites.finIndexation(site);
						dateFin = new Date();
						logger.info("%%%% Fin de l'indexation de " + site.getUrl() + " en " + ((dateFin.getTime() - dateDeb.getTime()) / 1000) + "s - " + nbPages + " pages indexées.");
						RechercheSitesDistants.init();
					}
					bFinIndexation = true;
				}
			};
			threadIndexation.start();
			if (join) {
				threadIndexation.join();
			}
		}
	}

	/**
	 * Indexe le site : lance les threads prennant en charge l'aspiration et indexe en parallele les pages remontees.
	 * 
	 * @param siteDistant
	 *            site a indexer
	 * 
	 * @return nombre de pages indexees
	 * 
	 * @throws MalformedURLException
	 *             levee si l'url est incorrecte
	 * @throws MalformedPatternException
	 *             levee si l'expression reguliere sur les urls est incorrecte
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	protected int indexe(final Site siteDistant) throws MalformedURLException, MalformedPatternException, IOException {
		IndexWriter writer = null;
		int nb = 0;
		try {
			writer = prepareIndexation(siteDistant);
			// suppression de l'index existant
			writer.deleteAll();
			writer.commit();
			// ajout du point d'entree sur le site
			queueURL.push(new URL(szUrlSite), 0);
			lanceThreadsAspiration();
			nb = indexePagesHTML(writer, siteDistant);
		} finally {
			if (writer != null) {
				writer.forceMerge(1, true);
				writer.close();
			}
		}
		return nb;
	}

	/**
	 * Prepare les elements necessaires a l'indexation.
	 * 
	 * @param siteDistant
	 *            site a indexer
	 * 
	 * @return index
	 * 
	 * @throws MalformedPatternException
	 *             levee si la construction du pattern de l'expression reguliere a echoue
	 * @throws IOException
	 *             levee si probleme d'acces a l'index
	 */
	protected IndexWriter prepareIndexation(final Site siteDistant) throws MalformedPatternException, IOException {
		this.nNiveauCourantProfondeur = 0;
		this.nNiveauMaxProfondeur = siteDistant.getNiveauProfondeur().intValue();
		this.szUrlSite = siteDistant.getUrl();
		this.bFinAspiration = false;
		this.queueHTML = new QueueFluxHTML();
		this.queueURL = new URLQueue();
		queueURL.setMaxElements(-1);
		litRobotsTxt();
		// patterns (threadsafe) pour les expressions regulieres
		// celles-ci ont ete verifiees a la saisie
		String szRegExpUrl = siteDistant.getRegExpAccepte();
		if ((szRegExpUrl != null) && !szRegExpUrl.equals("")) {
			patternURLAcceptation = new Perl5Compiler().compile(szRegExpUrl);
		} else {
			patternURLAcceptation = null;
		}
		szRegExpUrl = siteDistant.getRegExpRefuse();
		if ((szRegExpUrl != null) && !szRegExpUrl.equals("")) {
			patternURLRefus = new Perl5Compiler().compile(szRegExpUrl);
		} else {
			patternURLRefus = null;
		}
		final File repertoireIndexation = siteDistant.getRepertoireIndexation();
		if (!repertoireIndexation.exists()) {
			repertoireIndexation.mkdir();
		}
		final Directory directory = Searcher.getInstance().getDirectory(repertoireIndexation);
		final IndexWriterConfig conf = new IndexWriterConfig(Version.LUCENE_36, Indexer.analyzer);
		conf.setOpenMode(OpenMode.CREATE_OR_APPEND);
		return new IndexWriter(directory, conf);
	}

	/**
	 * Lance les thread sur le site courant. Appelee par les threads lorsqu'ils constatent que le nombre de threads courants est inferieur au nombre maximal.
	 */
	protected synchronized void lanceThreadsAspiration() {
		// Lance les threads suivant le maximum autorise, et s'il reste des url
		// pour le niveau courant
		int nNbThreads = nNbMaxThreads - nNbThreadsCourants;
		final int nTailleQueueURL = queueURL.getQueueSize(nNiveauCourantProfondeur);
		if (nTailleQueueURL < nNbThreads || nNbMaxThreads == -1) {
			nNbThreads = nTailleQueueURL;
		}
		// lance les threads nécessaires
		ThreadAspirateur threadAspi;
		for (int n = 1; n <= nNbThreads; ++n) {
			threadAspi = new ThreadAspirateur(this, szUrlSite, nNbThreadsCourants++, nNiveauCourantProfondeur, queueURL, queueHTML, maxSizeQueue, timeToSleep);
			threadAspi.start();
		}
	}

	/**
	 * Appelee par les threads pour indiquer qu'ils sont arrives en fin de traitement.
	 * 
	 * @param threadId
	 *            identifiant du thread
	 */
	protected synchronized void finTraitementThreadAspiration(final int threadId) {
		--nNbThreadsCourants;
		//receiver.finished(threadId);
		if (nNbThreadsCourants == 0) {
			++nNiveauCourantProfondeur;
			// si le niveau courant est superieur au niveau maximal
			// ou s'il n'y a pas d'url pour le niveau courant, il est inutile
			// de relancer d'autres threads
			if (nNiveauCourantProfondeur > nNiveauMaxProfondeur) {
				return;
			}
			if (queueURL.getQueueSize(nNiveauCourantProfondeur) == 0) {
				// BUG jamais atteint voir gestion timeout dans indexePagesHTML
				bFinAspiration = true;
				return;
			}
			lanceThreadsAspiration();
		}
	}

	/**
	 * Lance l'indexation des pages HTML aspirees. On recree complement l'index a chaque lancement.
	 * 
	 * @param writer
	 *            index
	 * @param indexation
	 *            site a indexer
	 * 
	 * @return the int
	 * 
	 * @throws IOException
	 *             levee si probleme lors de l'indexation d'un document
	 */
	protected int indexePagesHTML(final IndexWriter writer, final Site indexation) throws IOException {
		QueueFluxHTML.FluxHTML fluxHTML;
		int nIterationVide = 0;
		Document docPageHTML;
		Index index;
		int nbPage = 0;
		// on effectue l'indexation jusqu'a ce que la fin du traitement soit
		// effectuee
		while (!bFinAspiration) {
			// recuperation du flux aspire
			fluxHTML = queueHTML.pop();
			if (fluxHTML != null) {
				try {
					index = creeIndex(fluxHTML, indexation);
					docPageHTML = index.creerDocument();
					writer.addDocument(docPageHTML);
					logger.debug("==Indexation de " + fluxHTML.getUrl() + " terminee.");
					++nbPage;
				} catch (final Exception e) {
					logger.error("Exception lors de l'indexation de : " + fluxHTML.getUrl(), e);
				}
			} else {
				try {
					// attente qu'une url soit pushee
					Thread.sleep(1000);
					// gestion du timeout : pas de timeout sur les connexions +
					// pb détection fin aspiration
					if ((queueHTML.getTaille() != 0) || (nNbThreadsCourants != 0)) {
						nIterationVide = 0;
					} else {
						++nIterationVide;
					}
					// si au bout de n iterations on a rien dans la queue : on
					// s'arrete.
					if (nIterationVide == nTimeout) {
						bFinAspiration = true;
						// logger.debug( "-_-_-_- Timeout de " + nTimeout + "s
						// atteint" );
					}
				} catch (final Exception e) {
					logger.error("Erreur los de l'indexation des pages HTML", e);
				}
			}
		}
		return nbPage;
	}

	/**
	 * Cree index.
	 * 
	 * @param fluxHTML
	 *            the flux html
	 * @param indexation
	 *            the indexation
	 * 
	 * @return the index
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private Index creeIndex(final QueueFluxHTML.FluxHTML fluxHTML, final Site indexation) throws Exception {
		final HTMLParser htmlParser = new HTMLParser();
		htmlParser.setInputHtml(fluxHTML.getPage());
		String chaine = htmlParser.extractString(false);
		String title = htmlParser.getTitle();
		if (title.trim().length() == 0) {
			title = indexation.getLibelle();
		}
		String keywords = htmlParser.getMetaTag("keywords");
		String description = htmlParser.getMetaTag("description");
		chaine = Chaine.encode(chaine, CharEncoding.DEFAULT);
		title = Chaine.encode(title, CharEncoding.DEFAULT);
		keywords = Chaine.encode(keywords, CharEncoding.DEFAULT);
		description = Chaine.encode(description, CharEncoding.DEFAULT);
		final Index index = new Index();
		final String url = fluxHTML.getUrl();
		final Vector<String> codeRubrique = new Vector<String>();
		codeRubrique.add(indexation.getCode());
		index.setCodeRubrique(codeRubrique);
		index.setLangue("0");
		index.setContent(RechercheFmt.formaterTexteRecherche(chaine, Boolean.FALSE, Boolean.FALSE));
		index.setContentFile("");
		index.setTitle(title);
		index.setIdentifiantUnique(url);
		index.setUrl(url);
		index.setEtatFiche(EtatFiche.EN_LIGNE.getEtat());
		index.setLastModified(new SimpleDateFormat("yyyyMMdd").format(new Date()));
		index.setMiseEnLigne(new SimpleDateFormat("yyyyMMdd").format(new Date()));
		index.setKeywords(keywords);
		index.setDescription(description);
		return index;
	}

	/**
	 * Lit le fichier robots.txt.
	 */
	protected void litRobotsTxt() {
		try {
			final URL urlRobotsTxt = new URL(szUrlSite.substring(0, szUrlSite.lastIndexOf('/') + 1) + "robots.txt");
			//			logger.debug( "Robots.txt url = " + urlRobotsTxt );
			final BufferedReader bReader = new BufferedReader(new InputStreamReader(urlRobotsTxt.openConnection().getInputStream()));
			aszUrlDisallows = HTMLParser.parseRobots(bReader);
			/*
			 * for( int i = 0; i < aszUrlDisallows.length; ++i ) logger.debug(
			 * "\tURL DISALLOW : " + aszUrlDisallows[ i ] );
			 */
		} catch (final Exception e) {
			aszUrlDisallows = null;
		}
	}

	/**
	 * Constructeur prive. La classe est en fait un singleton.
	 */
	private IndexeurSitesDistants() {
		bFinIndexation = true;
		queueSites = new QueueSiteAIndexer();
		// lecture du nombre de threads
		final String szNbThreadJTF = PropertyHelper.getCoreProperty(PARAM_JTF_NBTHREADS);
		if (szNbThreadJTF != null) {
			try {
				nNbMaxThreads = Integer.parseInt(szNbThreadJTF);
			} catch (final Exception e) {
				nNbMaxThreads = NB_MAX_THREADS_DEFAUT;
			}
		} else {
			nNbMaxThreads = NB_MAX_THREADS_DEFAUT;
		}
		final String szTimeout = PropertyHelper.getCoreProperty(PARAM_JTF_TIMEOUT);
		if (szTimeout != null) {
			try {
				nTimeout = Integer.parseInt(szTimeout);
			} catch (final Exception e) {
				nTimeout = TIMEOUT_DEFAUT;
			}
		} else {
			nTimeout = TIMEOUT_DEFAUT;
		}
		final String sMaxSizeQueue = PropertyHelper.getCoreProperty(PARAM_JTF_MAXSIZE_HTMLQUEUE);
		if (sMaxSizeQueue != null) {
			maxSizeQueue = Integer.parseInt(sMaxSizeQueue);
		} else {
			maxSizeQueue = 1000;
		}
		final String sTimeToSleep = PropertyHelper.getCoreProperty(PARAM_JTF_TIME_SLEEP);
		if (sTimeToSleep != null) {
			timeToSleep = Long.parseLong(sTimeToSleep);
		} else {
			timeToSleep = 30000L;
		}
		logger.debug("%%%% Nombre de threads affectes a l'aspiration: " + nNbMaxThreads + " - Timeout: " + nTimeout + "s");
	}
}
