/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch.sitesdistants;

import java.util.Collections;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Map;

import com.univ.objetspartages.om.Site;

/**
 * Queue permettant de stocker les sites a indexer.
 * 
 * @author jbiard
 */
class QueueSiteAIndexer implements EtatsSitesIndexation {

	/**
	 * Instantiates a new queue site a indexer.
	 */
	QueueSiteAIndexer() {
		llSites = new LinkedList<Site>();
		mapSites = new Hashtable<Long, String>();
	}

	/**
	 * Ajoute un site et le passe en etat attente.
	 * 
	 * @param site
	 *            site a indexer
	 */
	synchronized void push(final Site site) {
		final Long lgIdSite = site.getIdSite();
		final String szEtat = mapSites.get(lgIdSite);
		// si on a deja le site on ne le reinjecte que si son indexation est terminee
		if (szEtat != null) {
			if (szEtat == ETAT_TERMINE) {
				mapSites.remove(lgIdSite);
			} else {
				return;
			}
		}
		// recopie necessaire : l'utilisation du nextItem nous fait perdre l'objet
		llSites.addLast((Site) site.clone());
		mapSites.put(lgIdSite, ETAT_ATTENTE);
	}

	/**
	 * Recupere le prochain site a indexer. Passage de son etat en "indexation en cours".
	 * 
	 * @return site
	 */
	synchronized Site pop() {
		if (llSites.size() > 0) {
			final Site site = llSites.removeFirst();
			final Long lgIdSite = site.getIdSite();
			mapSites.remove(site);
			mapSites.put(lgIdSite, ETAT_INDEXATION_EN_COURS);
			// en cours indexation
			return site;
		}
		return null;
	}

	/**
	 * Indique pour un site que son indexation est terminee.
	 * 
	 * @param site
	 *            site pour lequel l'indexation est terminee
	 */
	synchronized void finIndexation(final Site site) {
		final Long lgIdSite = site.getIdSite();
		if (mapSites.get(lgIdSite) != null) {
			mapSites.remove(lgIdSite);
			mapSites.put(lgIdSite, ETAT_TERMINE);
		}
	}

	/**
	 * Renvoie les etats courant des sites.
	 * 
	 * @return map dont la clef est le libelle du site
	 */
	Map<Long, String> getEtats() {
		return Collections.unmodifiableMap(mapSites);
	}

	/** The ll sites. */
	private final LinkedList<Site> llSites;

	/** The map sites. */
	private final Map<Long, String> mapSites;
}
