/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jsbsoft.jtf.textsearch.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.textsearch.Index;
import com.jsbsoft.jtf.textsearch.ResultatRecherche;
import com.jsbsoft.jtf.textsearch.Searcher;
import com.kportal.cms.objetspartages.Objetpartage;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.StructureModele;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.sql.ConstanteSQL;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.CritereRubriquePublicationHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class RechercheUtilBO {

	private static Logger LOG = LoggerFactory.getLogger(RechercheUtilBO.class);

	public static List<ResultatRecherche> traiterRecherche(final AutorisationBean autorisation, final String request) throws Exception {
		final Searcher searcher = Searcher.getInstance();
		if (searcher.getSearcher() == null) {
			return new ArrayList<>();
		}
		return searchAll(autorisation, searcher, request);
	}

	/**
	 * Traite la recherche.
	 * 
	 * @param autorisation
	 *            the autorisation
	 * @param searcher
	 *            the searcher
	 * @param request
	 *            the request
	 * 
	 * @return the vector
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private static List<ResultatRecherche> searchAll(final AutorisationBean autorisation, final Searcher searcher, final String request) throws Exception {
		final String codeObjet = "";
		final String codeRubrique = "";
		final String action = "M";
		final BooleanQuery query = getQueryDsi(autorisation, action, codeObjet, codeRubrique);
		final List<SortField> sortFields = new ArrayList<SortField>();
		sortFields.add(SortField.FIELD_SCORE);
		final Sort sort = new Sort(sortFields.toArray(new SortField[sortFields.size()]));
		final List<ResultatRecherche> listeFiches = new ArrayList<ResultatRecherche>();
		final int maxDocs = searcher.getSearcher().maxDoc();
		if (maxDocs > 0) {
			final TopFieldCollector tfc = TopFieldCollector.create(sort, maxDocs, false, true/*trackDocScores */, true/* trackMaxScore */, false/* docsInOrder */);
			searcher.getSearcher().search(query, tfc);
			final int totalHits = tfc.getTotalHits();
			final TopDocs results = tfc.topDocs();
			final int nbHits = results.scoreDocs.length;
			Document doc = null;
			for (int i = 0; i < nbHits; i++) {
				doc = searcher.getSearcher().doc(results.scoreDocs[i].doc);
				final ResultatRecherche resultatRecherche = new ResultatRecherche();
				resultatRecherche.setObjet(doc.get(Index.OBJET));
				resultatRecherche.setCodeFiche(doc.get(Index.CODE_FICHE));
				resultatRecherche.setIdFiche(doc.get(Index.ID_FICHE));
				resultatRecherche.setLangue(doc.get(Index.LANGUE));
				resultatRecherche.setCodeRubrique(doc.get(Index.CODE_RUBRIQUE));
				resultatRecherche.setCodeRattachement(doc.get(Index.CODE_RATTACHEMENT));
				resultatRecherche.setPublicVise(doc.get(Index.PUBLIC_VISE));
				resultatRecherche.setModeRestriction(doc.get(Index.MODE_RESTRICTION));
				resultatRecherche.setPublicViseRestriction(doc.get(Index.PUBLIC_VISE_RESTRICTION));
				resultatRecherche.setTitre(doc.get(Index.TITLE));
				resultatRecherche.setUrl(doc.get(Index.URL));
				resultatRecherche.setDateModification(doc.get(Index.LAST_MODIFIED));
				resultatRecherche.setDescription(doc.get(Index.DESCRIPTION));
				resultatRecherche.setRank(i);
				resultatRecherche.setTotal(nbHits);
				resultatRecherche.setTotalHits(totalHits);
				listeFiches.add(resultatRecherche);
			}
		} else {
			LOG.warn("Aucun document dans l'index de recherche");
		}
		return listeFiches;
	}

	private static BooleanQuery getQueryDsi(final AutorisationBean autorisation, final String action, final String codeObjet, final String codeRubrique) throws Exception {
		final BooleanQuery query = new BooleanQuery();
		if (isNotEmpty(codeObjet) && !"0000".equals(codeObjet)) {
			final FicheUniv fiche = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(codeObjet));
			addQuery(query, traiterConditionDsiMeta(autorisation, fiche, action, codeRubrique), BooleanClause.Occur.MUST);
		} else {
			final BooleanQuery queryAll = new BooleanQuery();
			for (final Objetpartage objet : ReferentielObjets.getObjetsPartagesTries()) {
				final FicheUniv fiche = ReferentielObjets.instancierFiche(objet.getNomObjet());
				if (!objet.isStrictlyCollaboratif() && fiche != null) {
					fiche.setCodeRubrique(codeRubrique);
					addQuery(queryAll, traiterConditionDsiMeta(autorisation, fiche, action, codeRubrique), BooleanClause.Occur.SHOULD);
				}
			}
			addQuery(query, queryAll, BooleanClause.Occur.MUST);
		}
		addQuery(query, getConditionRubPubSuivantAction(ContexteUtil.getContexteUniv(), action, codeRubrique), BooleanClause.Occur.MUST);
		return query;
	}

	private static BooleanQuery getConditionRubPubSuivantAction(final ContexteUniv ctx, final String action, final String codeRubrique) throws Exception {
		final Collection<String> listeCodeRubriques = CritereRubriquePublicationHelper.getCodesRubriquesSuivantAction(ctx, action, codeRubrique);
		final BooleanQuery conditionRubPub = new BooleanQuery();
		for (final String code : listeCodeRubriques) {
			conditionRubPub.add(new TermQuery(new Term(Index.CODE_RUBRIQUE, code)), BooleanClause.Occur.SHOULD);
		}
		return conditionRubPub;
	}

	/*
	 * ATTENTION CODE SENSIBLE QUI EST UNE COPIE DE LA METHODE DE LA CLASSE RequeteSQLHelper
	 */
	private static BooleanQuery traiterConditionDsiMeta(final AutorisationBean autorisation, final FicheUniv fiche, final String action, final String rubriqueRecherche)
		throws Exception {
		final BooleanQuery conditionComplete = new BooleanQuery();
		final BooleanQuery conditionSurFiche = new BooleanQuery();
		boolean restriction = Boolean.FALSE;
		boolean aucuneRestriction = Boolean.FALSE;
		boolean aucunePermission = Boolean.TRUE;
		// Controle des droits sur les actions de modification
		final String codeObjet = ReferentielObjets.getCodeObjet(fiche);
		final Hashtable<String, String> actionsPossibles = RequeteSQLHelper.recupererActionsPossibles(action);
		final Collection<String> codesEspacesCollab = new HashSet<>();
		if (autorisation != null) {
			for (final Entry<String, String> actionPossible : actionsPossibles.entrySet()) {
				final String lettreAction = actionPossible.getKey();
				for (final Entry<String, Vector<Perimetre>> permission : autorisation.getListePermissions().entrySet()) {
					if (RequeteSQLHelper.isPermissionValide(permission.getKey(), codeObjet, lettreAction)) {
						aucunePermission = Boolean.FALSE;
						for (final Perimetre perimetre : permission.getValue()) {
							final BooleanQuery conditionPerimetre = new BooleanQuery();
							if (RequeteSQLHelper.isAcuneRestriction(perimetre)) {
								aucuneRestriction = Boolean.TRUE;
							} else {
								final BooleanQuery conditionRubrique = traiterCodeRubriquePerimetre(perimetre.getCodeRubrique(), rubriqueRecherche);
								final BooleanQuery conditionStructure = traiterStructurePerimetre(perimetre.getCodeStructure(), fiche);
								if (StringUtils.isNotEmpty(perimetre.getCodeEspaceCollaboratif())) {
									codesEspacesCollab.add(perimetre.getCodeEspaceCollaboratif());
								}
								if (!conditionRubrique.clauses().isEmpty()) {
									restriction = Boolean.TRUE;
									conditionPerimetre.add(conditionRubrique, BooleanClause.Occur.MUST);
								}
								if (!conditionStructure.clauses().isEmpty()) {
									restriction = Boolean.TRUE;
									conditionPerimetre.add(conditionStructure, BooleanClause.Occur.MUST);
								}
							}
							addQuery(conditionSurFiche, conditionPerimetre, BooleanClause.Occur.SHOULD);
						}
					}
				}
			}
		}
		if (!codesEspacesCollab.isEmpty()) {
			final BooleanQuery conditionEspaceCollab = new BooleanQuery();
			conditionEspaceCollab.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "4")), BooleanClause.Occur.MUST);
			final BooleanQuery conditionCode = new BooleanQuery();
			for (final String codeEspace : codesEspacesCollab) {
				conditionCode.add(new TermQuery(new Term(Index.PUBLIC_VISE_RESTRICTION, codeEspace)), BooleanClause.Occur.SHOULD);
			}
			addQuery(conditionEspaceCollab, conditionCode, BooleanClause.Occur.MUST);
			addQuery(conditionSurFiche, conditionEspaceCollab, BooleanClause.Occur.SHOULD);
		}
		if (aucunePermission) {
			if (autorisation == null) {
				final BooleanQuery conditionImpossible = new BooleanQuery();
				conditionImpossible.add(new TermQuery(new Term(Index.OBJET, "0000")), BooleanClause.Occur.MUST);
				return conditionImpossible;
			}
			final BooleanQuery ficheDeLUtilisateur = new BooleanQuery();
			ficheDeLUtilisateur.add(new TermQuery(new Term(Index.CODE_REDACTEUR, autorisation.getCode())), BooleanClause.Occur.MUST);
			if (StringUtils.isNotBlank(codeObjet)) {
				ficheDeLUtilisateur.add(new TermQuery(new Term(Index.OBJET, ReferentielObjets.getNomObjet(codeObjet))), BooleanClause.Occur.MUST);
			}
			return ficheDeLUtilisateur;
		}
		if (conditionSurFiche.clauses().isEmpty() && !aucuneRestriction) {
			conditionSurFiche.add(new TermQuery(new Term(Index.OBJET, null)), BooleanClause.Occur.MUST_NOT);
			return conditionSurFiche;
		}
		if (aucuneRestriction && !restriction && !codesEspacesCollab.isEmpty()) {
			conditionSurFiche.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "4")), BooleanClause.Occur.MUST_NOT);
		}
		conditionComplete.add(new TermQuery(new Term(Index.OBJET, ReferentielObjets.getNomObjet(codeObjet))), BooleanClause.Occur.MUST);
		addQuery(conditionComplete, conditionSurFiche, BooleanClause.Occur.MUST);
		return conditionComplete;
	}

	private static BooleanQuery traiterStructurePerimetre(String codeStructure, final FicheUniv fiche) throws Exception {
		final BooleanQuery conditionStructure = new BooleanQuery();
		if (StringUtils.isNotEmpty(codeStructure)) {
			if ("-".equals(codeStructure)) {
				conditionStructure.add(new TermQuery(new Term(Index.CODE_RATTACHEMENT, "")), BooleanClause.Occur.MUST);
			} else {
				codeStructure = codeStructure.replaceAll("\\+", ";");
				if (ConditionHelper.isStructureNoArbo(codeStructure)) {
					codeStructure = StringUtils.substringBefore(codeStructure, ConstanteSQL.NO_ARBO);
					conditionStructure.add(new TermQuery(new Term(Index.CODE_RATTACHEMENT, codeStructure)), BooleanClause.Occur.MUST);
					return conditionStructure;
				}
				final Collection<String> listeDesCodesStructures = ConditionHelper.getListeDesCodesStructures(codeStructure);
				final BooleanQuery conditionCode = new BooleanQuery();
				for (final String code : listeDesCodesStructures) {
					conditionCode.add(new TermQuery(new Term(Index.CODE_RATTACHEMENT, code)), BooleanClause.Occur.SHOULD);
				}
				if (fiche instanceof StructureModele) {
					conditionCode.add(new TermQuery(new Term(Index.CODE_FICHE, codeStructure)), BooleanClause.Occur.SHOULD);
				}
				addQuery(conditionStructure, conditionCode, BooleanClause.Occur.MUST);
			}
		}
		return conditionStructure;
	}

	private static BooleanQuery traiterCodeRubriquePerimetre(final String codeRubrique, final String rubriqueRecherche) throws Exception {
		final BooleanQuery conditionRubrique = new BooleanQuery();
		if (StringUtils.isNotEmpty(codeRubrique)) {
			if ("-".equals(codeRubrique)) {
				conditionRubrique.add(new TermQuery(new Term(Index.CODE_RUBRIQUE, "")), BooleanClause.Occur.MUST);
			} else {
				final Collection<String> codesRubriquesValides = RequeteSQLHelper.recupererListeCodeRubriqueParRubriqueRecherche(codeRubrique, rubriqueRecherche);
				final BooleanQuery conditionCode = new BooleanQuery();
				for (final String code : codesRubriquesValides) {
					conditionCode.add(new TermQuery(new Term(Index.CODE_RUBRIQUE, code)), BooleanClause.Occur.SHOULD);
				}
				addQuery(conditionRubrique, conditionCode, BooleanClause.Occur.MUST);
			}
		}
		return conditionRubrique;
	}

	private static void addQuery(final BooleanQuery query, final BooleanQuery queryToAdd, final Occur occur) {
		if (!queryToAdd.clauses().isEmpty()) {
			query.add(queryToAdd, occur);
		}
	}
}
