/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.card.dao;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.layout.card.bean.CardBean;
import com.kosmos.layout.utils.LayoutJacksonMapper;
import com.univ.utils.json.Views;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.UUID;

public class CardDAO extends AbstractCommonDAO<CardBean> {

    public CardDAO() {
        this.tableName = "CARD";
    }

    @Override
    public CardBean add(CardBean card) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into CARD (DATAS_CARD, KEY_CARD) values (?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            String datas = LayoutJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(card);
            stmt.setString(1, datas);
            stmt.setString(2, card.getKey().toString());
            int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", card.toString(), this.tableName, Integer.valueOf(rowsAffected)));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            card.setId(Long.valueOf(rs.getLong(1)));
        } catch (Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", card.toString(), this.tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occured closing Resultset during add() on table %s", this.tableName), e);
                }
            }
        }
        return card;
    }

    public CardBean update(CardBean card) throws UpdateToDataSourceException {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update CARD set DATAS_CARD = ?, KEY_CARD = ? WHERE ID_CARD = ?")) {
            String datas = LayoutJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(card);
            stmt.setString(1, datas);
            stmt.setString(2, card.getKey().toString());
            stmt.setLong(3, card.getId());
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", card.toString(), this.tableName), e);
        }
        return card;
    }

    protected CardBean fill(ResultSet rs) throws DataSourceException {
        CardBean card = new CardBean();
        try {
            card = LayoutJacksonMapper.getMapper().readValue(rs.getString("DATAS_CARD"), CardBean.class);
            card.setId(rs.getLong("ID_CARD"));
            card.setKey(UUID.fromString(rs.getString("KEY_CARD")));
        } catch (Exception e) {
            LOG.error("An error occured trying to map card bean", e);
        }
        return card;
    }
}
