/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.card.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.layout.Layout;
import com.kosmos.layout.card.bean.CardBean;
import com.kosmos.layout.card.dao.CardDAO;
import com.kosmos.layout.card.util.CardDescription;
import com.kosmos.layout.card.util.CardList;
import com.kosmos.layout.exception.CardNotFoundException;
import com.kosmos.layout.exception.CardsNotFoundException;
import com.kosmos.layout.meta.bean.CardMetaBean;
import com.kosmos.layout.meta.dao.CardMetaDAO;
import com.kosmos.layout.slot.Slot;
import com.kportal.core.config.MessageHelper;
import com.kportal.extension.module.AbstractBeanManager;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.utils.FicheUnivMgr;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceCard extends AbstractBeanManager {

    private static final Logger LOG = LoggerFactory.getLogger(ServiceCard.class);

    public static final String ID_BEAN = "serviceCard";

    private final CardList cardsPool;
    private Map<Class<? extends CardBean>, CardDescription> availableCards;
    private CardDAO cardDao;

    private CardMetaDAO cardMetaDao;

    public ServiceCard() {
        this.availableCards = new HashMap<>();
        this.cardsPool = new CardList();
    }

    public Map<Class<? extends CardBean>, CardDescription> getAvailableCards() {
        return availableCards;
    }

    public void setCardDao(CardDAO cardDao) {
        this.cardDao = cardDao;
    }

    public void setCardMetaDao(CardMetaDAO cardMetaDao) {
        this.cardMetaDao = cardMetaDao;
    }

    public CardList getAllowedCardsPool(Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes) {
        final CardList allowedPool = new CardList();
        final Collection<Class<? extends CardBean>> allowedCardTypesSet = getAllowedCardTypesSet(allowedCardTypes);
        for (CardBean currentCardBean : cardsPool) {
            if (allowedCardTypesSet.contains(currentCardBean.getClass())) {
                allowedPool.add(currentCardBean);
            }
        }
        return allowedPool;
    }

    private Collection<Class<? extends CardBean>> getAllowedCardTypesSet(Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes) {
        final Collection<Class<? extends CardBean>> allowedTypesSet = new HashSet<>();
        for (Map.Entry<String, Collection<Class<? extends CardBean>>> currentEntry : allowedCardTypes.entrySet()) {
            allowedTypesSet.addAll(currentEntry.getValue());
        }
        return allowedTypesSet;
    }

    public Map<String, String> getAllowedCardBoView(Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes) {
        final Map<String, String> allowedPool = new HashMap<>();
        final Collection<Class<? extends CardBean>> allowedCardTypesSet = getAllowedCardTypesSet(allowedCardTypes);
        for (CardBean currentCardBean : cardsPool) {
            if (allowedCardTypesSet.contains(currentCardBean.getClass())) {
                allowedPool.put(currentCardBean.getClass().getName(), currentCardBean.getViewBo());
            }
        }
        return allowedPool;
    }

    /**
     * Useful method to save a card and bind it to a {@link com.univ.objetspartages.om.Metatag}.
     *
     * @param card
     * 		the {@link com.kosmos.layout.card.bean.CardBean} to save.
     * @param meta
     * 		the {@link com.univ.objetspartages.om.Metatag} to bind the {@link com.kosmos.layout.card.bean.CardBean} "card"
     * 		to.
     * @throws DataSourceException
     * 		thrown for one the following reason : <ul> <li>The "card" could not be persisted in the datasource</li> <li>The
     * 		"card" could not be updated ine the data source</li> <li>The binding could not be persisted in the data
     * 		source</li> </ul>
     */
    public void addCardForMeta(CardBean card, Metatag meta) throws DataSourceException {
        if (card.getId() != null) {
            this.cardDao.update(card);
        } else {
            this.cardDao.add(card);
        }
        // Ajout du card meta s'il n'est pas déjà présent
        CardMetaBean cardMeta = this.cardMetaDao.getCardMeta(card.getId(), meta.getIdMetatag());
        if (cardMeta == null) {
            cardMeta = new CardMetaBean();
            cardMeta.setIdCard(card.getId());
            cardMeta.setIdMeta(meta.getIdMetatag());
            this.cardMetaDao.add(cardMeta);
        }
    }

    /**
     * Delete a {@link com.kosmos.layout.card.bean.CardBean} for the given {@link com.univ.objetspartages.om.Metatag}.
     *
     * @param card
     * 		the {@link com.kosmos.layout.card.bean.CardBean} to delete.
     * @param meta
     * 		the {@link com.univ.objetspartages.om.Metatag} to delete the card for.
     * @throws DeleteFromDataSourceException
     * 		thrown for one of the following reasons : <ul> <li>The {@link com.kosmos.layout.card.bean.CardBean} card could
     * 		not be removed</li> <li>The {@link com.kosmos.layout.meta.bean.CardMetaBean} corresponding could not be
     * 		removed</li> </ul>
     * @see com.kosmos.layout.meta.bean.CardMetaBean
     */
    public void deleteCardForMeta(CardBean card, Metatag meta) throws DeleteFromDataSourceException {
        this.cardDao.delete(card.getId());
        this.cardMetaDao.deleteCardMeta(card.getId(), meta.getIdMetatag());
    }

    public void updateCard(CardBean card) throws UpdateToDataSourceException {
        this.cardDao.update(card);
    }

    public void updateCardModel(Map<String, CardBean> model) throws UpdateToDataSourceException {
        for (CardBean currentCardBean : model.values()) {
            updateCard(currentCardBean);
        }
    }

    /**
     * Retrieve a {@link java.util.Map} of {@link com.kosmos.layout.card.bean.CardBean} bound to a {@link
     * com.univ.objetspartages.om.FicheUniv} object.
     *
     * @param fiche
     * 		the {@link com.univ.objetspartages.om.FicheUniv} to search from.
     * @return a {@link java.util.Map} of {@link com.kosmos.layout.card.bean.CardBean} full-filling the binding
     * conditions.
     * @throws com.kosmos.layout.exception.CardsNotFoundException
     * 		thrown if no cards could be found.
     */
    public Map<String, CardBean> getCards(FicheUniv fiche) throws CardsNotFoundException {
        try {
            final Metatag meta = FicheUnivMgr.lireMeta(fiche);
            return getCards(meta.getIdMetatag());
        } catch (Exception e) {
            throw new CardsNotFoundException(String.format(MessageHelper.getCoreMessage("BO.CARD.EXCEPTION.NOT_FOUND_FICHE"), fiche.getCode()), e);
        }
    }

    /**
     * Retrieve a {@link java.util.Map} of {@link com.kosmos.layout.card.bean.CardBean} bound to a {@link
     * com.univ.objetspartages.om.Metatag}.
     *
     * @param metaId
     * 		the id of the {@link com.univ.objetspartages.om.Metatag} to search from.
     * @return a {@link java.util.Map} of {@link com.kosmos.layout.card.bean.CardBean} bean full-filling the binding
     * conditions.
     * @throws com.kosmos.layout.exception.CardsNotFoundException
     * 		thrown if no cards could be found.
     */
    public Map<String, CardBean> getCards(Long metaId) throws CardsNotFoundException {
        Collection<CardMetaBean> cardMeta = this.cardMetaDao.getByMeta(metaId);
        Map<String, CardBean> cards = new HashMap<>();
        for (CardMetaBean currentCardMeta : cardMeta) {
            try {
                CardBean currentCardBean = this.cardDao.getById(currentCardMeta.getIdCard());
                cards.put(currentCardBean.getKey().toString(), currentCardBean);
            } catch (DataSourceException e) {
                LOG.warn(String.format(MessageHelper.getCoreMessage("BO.CARD.EXCEPTION.NOT_FOUND_META"), metaId), e);
            }
        }
        return cards;
    }

    /**
     * Retrieve a card by its id.
     *
     * @param cardId
     * 		the id of the card to look for.
     * @return the card found as a {@link com.kosmos.layout.card.bean.CardBean}.
     * @throws com.kosmos.layout.exception.CardNotFoundException
     * 		thrown if the card could not be found.
     */
    public CardBean getCardById(Long cardId) throws CardNotFoundException {
        try {
            return this.cardDao.getById(cardId);
        } catch (DataSourceException e) {
            throw new CardNotFoundException(String.format(MessageHelper.getCoreMessage("BO.CARD.EXCEPTION.NOT_FOUND_ID"), cardId), e);
        }
    }

    /**
     * Useful method to compute and retrieve a {@link java.util.Map} of edit fragment according to a {@link
     * com.kosmos.layout.card.bean.CardBean} Map.
     *
     * @param cards
     * 		the {@link java.util.Map} of {@link com.kosmos.layout.card.bean.CardBean} from which the edit fragments should
     * 		be computed.
     * @return a {@link java.util.Map} of edit fragments.
     */
    public Map<String, String> getAllowedEditFragments(Collection<CardBean> pool) {
        final Map<String, String> editFragments = new HashMap<>();
        for (CardBean currentCard : pool) {
            editFragments.put(currentCard.getClass().getName(), currentCard.getEditFragment());
        }
        return editFragments;
    }

    /**
     * Useful method to assert consistency between a provided model and a data source model. The provided model stand as
     * the reference and shall remain intact. Only the data source model will be modified during the comparison.
     *
     * @param ficheUniv
     * 		the {@link com.univ.objetspartages.om.FicheUniv} object carrying the model to clean up.
     * @param providedModel
     * 		the model as a {@link java.util.Map} used to compare the persisted model to.
     * @throws Exception
     * 		thrown if the {@link com.univ.objetspartages.om.Metatag} could not be retrieved from the {@link
     * 		com.univ.objetspartages.om.FicheUniv} object
     */
    public void cleanUpModel(FicheUniv ficheUniv, Map<String, CardBean> providedModel) throws Exception {
        final Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
        try {
            final Map<String, CardBean> recordedModel = getCards(ficheUniv);
            for (Map.Entry<String, CardBean> currentEntry : recordedModel.entrySet()) {
                if (!providedModel.containsKey(currentEntry.getKey())) {
                    try {
                        final CardMetaBean cardMeta = cardMetaDao.getCardMeta(currentEntry.getValue().getId(), meta.getIdMetatag());
                        cardMetaDao.delete(cardMeta.getId());
                        cardDao.delete(currentEntry.getValue().getId());
                    } catch (DeleteFromDataSourceException e) {
                        LOG.error(String.format("La carte portant l'id \"%d\" n'a pas pu être supprimée.", currentEntry.getValue().getId()), e);
                    }
                }
            }
        } catch (CardsNotFoundException e) {
            LOG.info("Aucune carte trouvée pour le nettoyage du modèle.");
        }
    }

    @Override
    public void refresh() {
        final Map<String, CardDescription> retrievedCardDescription = ApplicationContextManager.getAllBeansOfType(CardDescription.class);
        if (MapUtils.isNotEmpty(retrievedCardDescription)) {
            for (CardDescription currentCardDescription : retrievedCardDescription.values()) {
                availableCards.put(currentCardDescription.getType(), currentCardDescription);
                try {
                    cardsPool.add(currentCardDescription.getType().newInstance());
                } catch (InstantiationException | IllegalAccessException e) {
                    LOG.error("Une erreur est survenue lors de la création du pool de carte", e);
                }
            }
        }
    }

    public Map<String, Collection<Class<? extends CardBean>>> getAllowedCardTypes(Layout layout) {
        final Map<String, Collection<Class<? extends CardBean>>> cardsAllowed = new HashMap<>();
        for (Slot currentSlot : layout.getSlots()) {
            final Collection<Class<? extends CardBean>> allowedClasses = new ArrayList<>();
            if (CollectionUtils.isEmpty(currentSlot.getAllowedCardTypes())) {
                if (CollectionUtils.isEmpty(layout.getAllowedCardTypes())) {
                    allowedClasses.addAll(getAllCardClasses());
                } else {
                    allowedClasses.addAll(layout.getAllowedCardTypes());
                }
            } else {
                allowedClasses.addAll(currentSlot.getAllowedCardTypes());
            }
            cardsAllowed.put(currentSlot.getKey().toString(), allowedClasses);
        }
        return cardsAllowed;
    }

    public Collection<CardDescription> getAllowedCardsDescriptions(Layout layout) {
        final List<CardDescription> descriptions = new ArrayList<>();
        final Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes = getAllowedCardTypes(layout);
        for (Collection<Class<? extends CardBean>> cards : allowedCardTypes.values()) {
            for (Class<? extends CardBean> currentClass : cards) {
                final CardDescription description = availableCards.get(currentClass);
                if (!descriptions.contains(description)) {
                    descriptions.add(description);
                }
            }
        }
        Collections.sort(descriptions);
        return descriptions;
    }

    private Collection<Class<? extends CardBean>> getAllCardClasses() {
        return availableCards.keySet();
    }

}
