/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.card.util;

import java.io.IOException;

import com.kosmos.layout.card.bean.CardBean;
import com.kosmos.layout.utils.LayoutJacksonMapper;
import com.univ.utils.ContexteUtil;
import com.univ.utils.UnivWebFmt;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CardUtil {

	public static final Logger LOG = LoggerFactory.getLogger(CardUtil.class);

	/**
	 * Méthode utilisée pour obtenir la chaîne JSON correspondant à la CardBean passée en paramètre.
	 *
	 * @param card
	 * 		the {@link com.kosmos.layout.card.bean.CardBean} to serialize.
	 * @return the JSON serialization of the {@link com.kosmos.layout.card.bean.CardBean} "card".
	 * @see #getCardFromJson(String)
	 */
	public static String getJsonModel(CardBean card) throws IOException {
		if (card != null) {
			return LayoutJacksonMapper.getMapper().writeValueAsString(card);
		}
		return null;
	}

	/**
	 * Méthode utilisé pour désérialiser une {@link com.kosmos.layout.card.bean.CardBean} à partir d'une chaîne JSON.
	 *
	 * @param jsonCard
	 * 		la chaîne à parser
	 * @return la {@link com.kosmos.layout.card.bean.CardBean} décrite par la chaîne JSON.
	 * @see #getJsonModel(com.kosmos.layout.card.bean.CardBean)
	 */
	public static CardBean getCardFromJson(String jsonCard) throws IOException {
		return LayoutJacksonMapper.getMapper().readValue(jsonCard, CardBean.class);
	}

	/**
	 * Méthode utilisée pour le calcul de liens issus de composants KmonoSelect.
	 *
	 * @param link
	 * 		: la chaine "code, langue, type | libellé"
	 * @return le lien http calculé ou vide si l'url n'a pas pu être calculée.
	 */
	public static String getLink(String link) {
		final String[] datas = StringUtils.split(StringUtils.substringBefore(link, "|"), ",");
		if (datas.length == 3) {
			try {
				final String code = datas[0];
				final String langue = datas[1].replace("LANGUE=", StringUtils.EMPTY);
				final String objet = datas[2].replace("TYPE=", StringUtils.EMPTY);

				return UnivWebFmt.determinerUrlFiche(ContexteUtil.getContexteUniv(), objet, code, langue, true, ContexteUtil.getContexteUniv().getCodeRubriqueFicheCourante());
			} catch (Exception e) {
				LOG.error(String.format("Une erreur est survenue lors du calcul de l'url %s", link), e);
				return StringUtils.EMPTY;
			}
		}
		return StringUtils.EMPTY;
	}

}
