/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.content.impl;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kosmos.components.media.bean.ComponentMediaFile;
import com.kosmos.components.media.bean.ComponentMediaFileList;
import com.kosmos.layout.annotations.ContentTypes;
import com.kosmos.layout.annotations.MediaContent;
import com.kosmos.layout.card.bean.CardBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Ressource;
import com.univ.utils.ContexteDao;

/**
 * Created by Fabien on 11/11/2014.
 */
public class MediaContentHandler extends AbstractContentHandler<MediaContent> {

    public static final Logger LOG = LoggerFactory.getLogger(MediaContentHandler.class);

    @Override
    protected int handleModel(final Map<String, CardBean> inputedModel, final FicheUniv ficheUniv, final Map<String, Object> datas, final int startIndex) {
        final ComponentMediaFileList mediaFiles = getMediaFiles(inputedModel);
        int index = 0;
        for (ComponentMediaFile currentComponentMediaFile : mediaFiles) {
            synchronizeMedia(currentComponentMediaFile, ficheUniv);
            index++;
        }
        return index;
    }

    @Override
    protected void handleSwitch(final Map<String, CardBean> inputedModel, FicheUniv ficheUniv, final Map<String, Object> datas) {
        final ComponentMediaFileList mediaFiles = getMediaFiles(inputedModel);
        for (ComponentMediaFile currentComponentMediaFile : mediaFiles) {
            switchRessource(currentComponentMediaFile, ficheUniv);
        }
    }

    private ComponentMediaFileList getMediaFiles(final Map<String, CardBean> inputedModel) {
        final ComponentMediaFileList mediaFiles = new ComponentMediaFileList();
        for (CardBean currentCard : inputedModel.values()) {
            final Collection<Field> fields = getAnnotatedFields(currentCard);
            for (Field currentField : fields) {
                final MediaContent contentType = currentField.getAnnotation(MediaContent.class);
                if (contentType.type() == ContentTypes.MEDIA) {
                    try {
                        currentField.setAccessible(true);
                        Object fieldValue = currentField.get(currentCard);
                        if (fieldValue instanceof ComponentMediaFile) {
                            mediaFiles.add((ComponentMediaFile) fieldValue);
                        } else if (fieldValue instanceof ComponentMediaFileList) {
                            mediaFiles.addAll((ComponentMediaFileList) fieldValue);
                        }
                    } catch (IllegalAccessException e) {
                        LOG.error(String.format("An error occured trying to handle card %s", currentCard.toString()), e);
                    }
                }
            }
        }
        sortMediaFiles(mediaFiles);
        return mediaFiles;
    }

    private void sortMediaFiles(final ComponentMediaFileList mediaFiles) {
        int index = getMaxIndex(mediaFiles);
        Collections.sort(mediaFiles, new Comparator<ComponentMediaFile>() {
            @Override
            public int compare(ComponentMediaFile o1, ComponentMediaFile o2) {
                return o2.getFileNumber() - o1.getFileNumber();
            }
        });
        for (ComponentMediaFile currentComponentMediaFile : mediaFiles) {
            if (currentComponentMediaFile.getFileNumber() == -1) {
                currentComponentMediaFile.setFileNumber(++index);
            }
        }
    }

    private int getMaxIndex(ComponentMediaFileList componentMediaFileList) {
        int index = 0;
        for (ComponentMediaFile currentComponentMediaFile : componentMediaFileList) {
            if (index < currentComponentMediaFile.getFileNumber()) {
                index = currentComponentMediaFile.getFileNumber();
            }
        }
        return index;
    }

    private void synchronizeMedia(ComponentMediaFile componentMediaFile, FicheUniv ficheUniv) {
        try (final ContexteDao ctx = new ContexteDao()) {
            Ressource.synchroniserFichier(ctx, ficheUniv, componentMediaFile.toString(), "PHOTO", componentMediaFile.getFileNumber());
        } catch (Exception e) {
            LOG.error(String.format("An error occured trying to save mediaFile %s", componentMediaFile.toString()), e);
        }
    }

    private void switchRessource(final ComponentMediaFile componentMediaFile, final FicheUniv ficheUniv) {
        final Ressource newRessource = Ressource.getFichier(ficheUniv, Integer.toString(componentMediaFile.getFileNumber()));
        if(newRessource != null && !Objects.equals(newRessource.getIdRessource(), componentMediaFile.getId())) {
            componentMediaFile.setId(newRessource.getIdRessource());
            componentMediaFile.setFormat(newRessource.getFormat());
            componentMediaFile.setLegende(newRessource.getLegende());
            componentMediaFile.setTitre(newRessource.getLibelle());
        }
    }
}
