/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.layout.Layout;
import com.kosmos.layout.impl.AbstractLayout;
import com.kosmos.layout.slot.util.SlotList;
import com.kosmos.layout.utils.LayoutJacksonMapper;
import com.univ.utils.json.Views;

/**
 * Created by Fabien on 10/11/2014.
 */
public class LayoutDAO extends AbstractCommonDAO<Layout> {

    public LayoutDAO() {
        this.tableName = "LAYOUT";
    }

    @Override
    public Layout add(Layout layout) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into LAYOUT (NAME_LAYOUT, HAS_CONTEXT_LAYOUT, DESCRIPTION_LAYOUT, SLOTS_LAYOUT, DATAS_LAYOUT) values (?, ?, ?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            String datas = LayoutJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(layout);
            stmt.setString(1, layout.getName());
            stmt.setBoolean(2, layout.hasContext());
            stmt.setString(3, layout.getDescription());
            stmt.setString(4, LayoutJacksonMapper.getMapper().writeValueAsString(layout.getSlots()));
            stmt.setString(5, datas);
            int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", layout.toString(), this.tableName, rowsAffected));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            layout.setId(rs.getLong(1));
        } catch (Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", layout.toString(), this.tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occurred closing Resultset during add() on table %s", this.tableName), e);
                }
            }
        }
        return layout;
    }

    @Override
    public Layout update(Layout layout) throws UpdateToDataSourceException {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update LAYOUT set DESCRIPTION_LAYOUT = ?, HAS_CONTEXT_LAYOUT = ?, SLOTS_LAYOUT = ?, DATAS_LAYOUT = ?  WHERE NAME_LAYOUT = ?")) {

            String datas = LayoutJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(layout);
            stmt.setString(1, layout.getDescription());
            stmt.setBoolean(2, layout.hasContext());
            stmt.setString(3, LayoutJacksonMapper.getMapper().writeValueAsString(layout.getSlots()));
            stmt.setString(4, datas);
            stmt.setString(5, layout.getName());
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", layout.toString(), this.tableName), e);
        }
        return layout;
    }

    @Override
    protected Layout fill(ResultSet rs) throws DataSourceException {
        AbstractLayout layout = null;
        try {
            layout = LayoutJacksonMapper.getMapper().readValue(rs.getString("DATAS_LAYOUT"), AbstractLayout.class);
            layout.setSlots(LayoutJacksonMapper.getMapper().readValue(rs.getString("SLOTS_LAYOUT"), SlotList.class));
            layout.setId(Long.valueOf(rs.getLong("ID_LAYOUT")));
            layout.setName(rs.getString("NAME_LAYOUT"));
            layout.setDescription(rs.getString("DESCRIPTION_LAYOUT"));
            layout.setContext(rs.getBoolean("HAS_CONTEXT_LAYOUT"));
        } catch (Exception e) {
            LOG.error("An error occurred trying to map layout bean", e);
        }
        return layout;
    }

    public Layout getByName(String name) throws DataSourceException {
        ResultSet rs = null;
        Layout layout = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%1$s` T1 WHERE T1.NAME_%1$s = ?", tableName))) {
            stmt.setObject(1, name, Types.VARCHAR);
            rs = stmt.executeQuery();
            if (!rs.first()) {
                return null;
            }
            layout = fill(rs);
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving object with name %s from table %s", name, tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getByName() on table %s", tableName), e);
                }
            }
        }
        return layout;
    }

    public Collection<Layout> getAllLayouts() throws DataSourceException {
        ResultSet rs = null;
        Collection<Layout> layouts = new ArrayList<>();
        try (final Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s`", tableName))) {
            rs = stmt.executeQuery();
            while (rs.next()) {
                layouts.add(fill(rs));
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getAllLayouts() on table %s", tableName), e);
                }
            }
        }
        return layouts;
    }
}
