/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.meta.dao;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.layout.meta.bean.CardMetaBean;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;

public class CardMetaDAO extends AbstractCommonDAO<CardMetaBean> {

    public CardMetaDAO() {
        this.tableName = "CARD_META";
    }

    public CardMetaBean add(CardMetaBean cardMeta) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("insert into %s (ID_CARD, ID_META) values (?, ?)", this.tableName), Statement.RETURN_GENERATED_KEYS)) {
            stmt.setLong(1, cardMeta.getIdCard());
            stmt.setLong(2, cardMeta.getIdMeta());
            int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", cardMeta.toString(), this.tableName, Integer.valueOf(rowsAffected)));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            cardMeta.setId(Long.valueOf(rs.getLong(1)));
        } catch (Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", cardMeta.toString(), this.tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occurred closing Resultset during add() on table %s", this.tableName), e);
                }
            }
        }
        return cardMeta;
    }

    public CardMetaBean update(CardMetaBean cardMeta) throws UpdateToDataSourceException {

        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("update %s set ID_CARD = ?, ID_META = ? WHERE ID_CARD_META = ?", this.tableName))) {
            stmt.setLong(1, cardMeta.getIdCard());
            stmt.setLong(2, cardMeta.getIdMeta());
            stmt.setLong(3, cardMeta.getId());
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", cardMeta.toString(), this.tableName), e);
        }
        return null;
    }

    protected CardMetaBean fill(ResultSet rs) throws DataSourceException {
        CardMetaBean cardMeta = new CardMetaBean();
        try {
            cardMeta.setId(rs.getLong("ID_CARD_META"));
            cardMeta.setIdCard(rs.getLong("ID_CARD"));
            cardMeta.setIdMeta(rs.getLong("ID_META"));
        } catch (Exception e) {
            LOG.error("An error occurred trying to map cardMeta bean", e);
        }
        return cardMeta;
    }

    public Collection<CardMetaBean> getByMeta(Long metaId) throws DataSourceException {
        final Collection<CardMetaBean> metas = new ArrayList<>();
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s` T1 WHERE T1.ID_META = ?", this.tableName))) {
            stmt.setLong(1, metaId);
            rs = stmt.executeQuery();
            while (rs.next()) {
                metas.add(fill(rs));
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving object with ids [%d] from table %s", metaId, this.tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getById() on table %s", this.tableName), e);
                }
            }
        }
        return metas;
    }

    public CardMetaBean getCardMeta(Long cardId, Long metaId) throws DataSourceException {
        ResultSet rs = null;
        CardMetaBean object = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s` T1 WHERE T1.ID_CARD = ? AND T1.ID_META = ?", tableName))) {
            stmt.setLong(1, cardId);
            stmt.setLong(2, metaId);
            rs = stmt.executeQuery();
            if (!rs.first()) {
                return null;
            }
            object = fill(rs);
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving object with ids [%d, %d] from table %s", cardId, metaId, tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getById() on table %s", tableName), e);
                }
            }
        }
        return object;
    }

    public void deleteCardMeta(Long cardId, Long metatagId) throws DeleteFromDataSourceException {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("delete from `%s` WHERE ID_CARD = ? AND ID_META = ?", tableName))) {
            stmt.setLong(1, cardId);
            stmt.setLong(2, metatagId);
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected == 0) {
                throw new DeleteFromDataSourceException(String.format("Table \"%s\" doesn't contain any row width ids [%d, %d]", tableName, cardId, metatagId));
            } else if (rowsAffected > 1) {
                throw new DeleteFromDataSourceException(String.format("Table \"%s\" contains more than one row width ids [%d, %d]", tableName, cardId, metatagId));
            }
        } catch (final SQLException e) {
            throw new DeleteFromDataSourceException(String.format("An error occurred during deletion of row with ids [%d, %d] from table \"%s\"", cardId, metatagId, tableName), e);
        }
    }
}
