/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.meta.dao;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.layout.meta.bean.LayoutMetaBean;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;

public class LayoutMetaDAO extends AbstractCommonDAO<LayoutMetaBean> {

    public LayoutMetaDAO() {
        this.tableName = "LAYOUT_META";
    }

    public LayoutMetaBean add(LayoutMetaBean layoutMeta) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into LAYOUT_META (ID_LAYOUT, ID_META) values (?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            stmt.setObject(1, layoutMeta.getIdLayout(), 4);
            stmt.setObject(2, layoutMeta.getIdMeta(), 4);
            int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", layoutMeta.toString(), this.tableName, Integer.valueOf(rowsAffected)));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            layoutMeta.setId(Long.valueOf(rs.getLong(1)));
        } catch (Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", layoutMeta.toString(), this.tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occurred closing Resultset during add() on table %s", this.tableName), e);
                }
            }
        }
        return layoutMeta;
    }

    public LayoutMetaBean update(LayoutMetaBean layoutMeta) throws UpdateToDataSourceException {

        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update LAYOUT_META set ID_LAYOUT = ?, ID_META = ? WHERE ID_LAYOUT_META = ?")) {
            stmt.setObject(1, layoutMeta.getIdLayout(), 4);
            stmt.setObject(2, layoutMeta.getIdMeta(), 4);
            stmt.setObject(3, layoutMeta.getId(), -5);
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", layoutMeta.toString(), this.tableName), e);
        }
        return null;
    }

    protected LayoutMetaBean fill(ResultSet rs) throws DataSourceException {
        LayoutMetaBean cardMeta = new LayoutMetaBean();
        try {
            cardMeta.setId(Long.valueOf(rs.getLong("ID_LAYOUT_META")));
            cardMeta.setIdLayout(Long.valueOf(rs.getLong("ID_LAYOUT")));
            cardMeta.setIdMeta(Long.valueOf(rs.getLong("ID_META")));
        } catch (Exception e) {
            LOG.error("An error occurred trying to map layoutMeta bean", e);
        }
        return cardMeta;
    }

    public LayoutMetaBean getByMetaId(Long metaId) throws DataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s` T1 WHERE T1.ID_META = ?", tableName))) {
            stmt.setObject(1, metaId, Types.BIGINT);
            rs = stmt.executeQuery();
            if (!rs.first()) {
                return null;
            }
            return fill(rs);
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving object with id %d from table %s", metaId, tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getById() on table %s", tableName), e);
                }
            }
        }
    }

    public LayoutMetaBean getLayoutMeta(Long layoutId, Long metaId) throws DataSourceException {
        ResultSet rs = null;
        LayoutMetaBean object = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `%s` T1 WHERE T1.ID_LAYOUT = ? AND T1.ID_META = ?", tableName))) {
            stmt.setLong(1, layoutId);
            stmt.setLong(2, metaId);
            rs = stmt.executeQuery();
            if (!rs.first()) {
                return null;
            }
            object = fill(rs);
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving object with ids [%d, %d] from table %s", layoutId, metaId, tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occurred closing ResultSet during getById() on table %s", tableName), e);
                }
            }
        }
        return object;
    }
}
