/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.plugin;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kosmos.layout.Layout;
import com.kosmos.layout.card.bean.CardBean;
import com.kosmos.layout.card.service.ServiceCard;
import com.kosmos.layout.card.util.CardList;
import com.kosmos.layout.card.util.CardUtil;
import com.kosmos.layout.card.view.model.CardViewModel;
import com.kosmos.layout.content.ContentHandler;
import com.kosmos.layout.content.manager.ContentHandlerManager;
import com.kosmos.layout.service.ServiceLayout;
import com.kosmos.layout.slot.Slot;
import com.kosmos.layout.slot.util.SlotState;
import com.kosmos.layout.slot.view.model.SlotViewModel;
import com.kosmos.layout.utils.LayoutJacksonMapper;
import com.kosmos.layout.view.model.LayoutManagerViewModel;
import com.kosmos.layout.view.model.LayoutViewModel;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.utils.EscapeString;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created on 13/11/14.
 */
public class LayoutFichePluginHelper {

	public static final Logger LOG = LoggerFactory.getLogger(LayoutFichePluginHelper.class);

	public static final String LAYOUT_VIEW_MODEL = "LAYOUT_VIEW_MODEL";
	public static final String LAYOUT_MANAGER_VIEW_MODEL = "LAYOUT_MANAGER_VIEW_MODEL";
	public static final String CARD_VIEW_MODEL = "CARD_VIEW_MODEL";
	public static final String SLOT_VIEW_MODEL = "SLOT_VIEW_MODEL";

	private static ContentHandlerManager contentHandlerManager;

	private static ServiceCard serviceCard;

	private static ServiceLayout serviceLayout;

	static {
		contentHandlerManager = (ContentHandlerManager) ApplicationContextManager.getBean("core", ContentHandlerManager.ID_BEAN);
		serviceCard = (ServiceCard) ApplicationContextManager.getBean("core", ServiceCard.ID_BEAN);
		serviceLayout = (ServiceLayout) ApplicationContextManager.getBean("core", ServiceLayout.ID_BEAN);
	}

	public static void prepareBoDatas(final Map<String, Object> datas, final FicheUniv ficheUniv) throws Exception {
		final Layout layout = serviceLayout.getLayout(ficheUniv);
		prepareBoDatas(datas, ficheUniv, layout);
	}

	public static void prepareBoDatas(final Map<String, Object> datas, final FicheUniv ficheUniv, final Long layoutId) throws Exception {
		final Layout layout = serviceLayout.getLayout(layoutId);
		prepareBoDatas(datas, ficheUniv, layout);
	}

	public static void prepareBoDatas(final Map<String, Object> data, final FicheUniv ficheUniv, final Layout layout) throws Exception {
		if (layout != null) {
			final Map<String, CardBean> cardModel = serviceCard.getCards(ficheUniv);
			prepareBoDatas(data, ficheUniv, layout, cardModel);
		}
	}

	public static void prepareBoDatas(final Map<String, Object> datas, final FicheUniv ficheUniv, final Layout layout, Map<String, CardBean> cardModel) throws Exception {
		if (layout != null) {
			prepareContent(cardModel, ficheUniv, datas);
			serviceCard.updateCardModel(cardModel);
			assertModelConsistency(layout, cardModel);

			final LayoutViewModel layoutViewModel = prepareLayoutViewModel(layout, cardModel);
			final LayoutManagerViewModel layoutManagerViewModel = prepareLayoutManagerViewModel(layout);
			layoutManagerViewModel.setLayoutViewModel(layoutViewModel);
			datas.put(LAYOUT_MANAGER_VIEW_MODEL, layoutManagerViewModel);
		}
	}

	protected static void prepareFoDatas(final Map<String, Object> datas, final FicheUniv ficheUniv) throws Exception {
		final Layout layout = serviceLayout.getLayout(ficheUniv);
		if (layout != null) {
			final Map<String, CardBean> cardModel = serviceCard.getCards(ficheUniv);
			prepareContent(cardModel, ficheUniv, datas);
			serviceCard.updateCardModel(cardModel);
			assertModelConsistency(layout, cardModel);

			final LayoutViewModel layoutViewModel = prepareLayoutViewModel(layout, cardModel);
			datas.put(LAYOUT_VIEW_MODEL, layoutViewModel);
		}
	}

	private static LayoutManagerViewModel prepareLayoutManagerViewModel(final Layout layout) throws Exception {
		final LayoutManagerViewModel layoutManagerViewModel = new LayoutManagerViewModel();
		final Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes = serviceCard.getAllowedCardTypes(layout);
		final CardList cardPool = serviceCard.getAllowedCardsPool(allowedCardTypes);
		final String jsonCardPool = LayoutJacksonMapper.getMapper().writeValueAsString(cardPool);

		layoutManagerViewModel.setCardBoViewPool(serviceCard.getAllowedCardBoView(allowedCardTypes));
		layoutManagerViewModel.setAvailableLayouts(serviceLayout.getAvailableLayouts());
		layoutManagerViewModel.setCardsPool(EscapeString.escapeAttributHtml(jsonCardPool));
		layoutManagerViewModel.setEditFragments(serviceCard.getAllowedEditFragments(cardPool));
		layoutManagerViewModel.setCardsDescriptions(serviceCard.getAllowedCardsDescriptions(layout));
		layoutManagerViewModel.setAllowedCardTypes(allowedCardTypes);

		return layoutManagerViewModel;
	}

	private static LayoutViewModel prepareLayoutViewModel(final Layout layout, Map<String, CardBean> cardModel) throws Exception {
		final LayoutViewModel layoutViewModel = new LayoutViewModel();

		layoutViewModel.setLayout(layout);
		layoutViewModel.setSlotViewModels(prepareSlotViewModels(layout, cardModel));

		return layoutViewModel;
	}

	private static Map<String, SlotViewModel> prepareSlotViewModels(final Layout layout, final Map<String, CardBean> cards) throws Exception {
		final Map<String, SlotViewModel> slotViewModels = new HashMap<>();
		final Map<String, Collection<Class<? extends CardBean>>> allowedCardTypes = serviceCard.getAllowedCardTypes(layout);
		for (Slot currentSlot : layout.getSlots()) {
			final String currentKey = currentSlot.getKey().toString();
			final SlotViewModel currentSlotViewModel = new SlotViewModel();
			final String jsonAllowedCardTypes = LayoutJacksonMapper.getMapper().writeValueAsString(allowedCardTypes.get(currentKey));
			final CardViewModel cardViewModel = prepareCardViewModel(cards.get(currentKey));
			currentSlot.setState(cardViewModel.getCardBean() == null ? SlotState.EMPTY : SlotState.FILLED);
			currentSlotViewModel.setSlot(currentSlot);
			currentSlotViewModel.setAllowedCardTypes(EscapeString.escapeAttributHtml(jsonAllowedCardTypes));
			currentSlotViewModel.setCardViewModel(cardViewModel);
			slotViewModels.put(currentKey, currentSlotViewModel);
		}
		return slotViewModels;
	}

	private static CardViewModel prepareCardViewModel(CardBean card) {
		final CardViewModel cardViewModel = new CardViewModel();
		String jsonCardModel = StringUtils.EMPTY;
		try {
			jsonCardModel = StringUtils.defaultIfBlank(CardUtil.getJsonModel(card), StringUtils.EMPTY);
		} catch (IOException e) {
			LOG.error(String.format("Une erreur est survenue lors de la sérialisation de la carte %s", card.getKey()), e);
		}
		cardViewModel.setCardBean(card);
		cardViewModel.setJsonCard(jsonCardModel);
		return cardViewModel;
	}

	private static void assertModelConsistency(Layout layout, Map<String, CardBean> cardModel) {
		for (Slot currentSlot : layout.getSlots()) {
			final String currentKey = currentSlot.getKey().toString();
			if (cardModel.containsKey(currentKey)) {
				final CardBean currentCard = cardModel.get(currentKey);
				if (CollectionUtils.isNotEmpty(currentSlot.getAllowedCardTypes()) && !currentSlot.getAllowedCardTypes().contains(currentCard.getClass())) {
					cardModel.remove(currentKey);
				}
			}
		}
	}

	// Permet de gérér les changements de ressources qui interviennent lors des changements d'états de fiche
	// ex : media de toolbox et media
	private static void prepareContent(Map<String, CardBean> cards, FicheUniv ficheUniv, Map<String, Object> datas) {
		for (ContentHandler currentHandler : contentHandlerManager.getContentHandlers()) {
			currentHandler.prepareContent(cards, ficheUniv, datas);
		}
	}

	public static Map<String, CardBean> retrieveModelFromDatas(Map<String, Object> datas, Layout layout, boolean resetIds) {
		final Map<String, CardBean> inputedModel = new HashMap<>();
		for (Slot currentSlot : layout.getSlots()) {
			final String currentModel = (String) datas.get(String.format("jsonModel-%s", currentSlot.getKey().toString()));
			if (StringUtils.isNotBlank(currentModel)) {
				final CardBean currentInputedCard;
				try {
					currentInputedCard = CardUtil.getCardFromJson(currentModel);
					if (resetIds) {
						currentInputedCard.setId(null);
					}
					inputedModel.put(currentInputedCard.getKey().toString(), currentInputedCard);
				} catch (IOException e) {
					LOG.error(String.format("Une erreur est survenue lors de la récupération de la carte depuis le json \"%s\"", currentModel), e);
				}
			}
		}
		return inputedModel;
	}

	public static void propagateInputedModel(Map<String, Object> datas, Long layoutId) throws Exception {
		final Layout layout = serviceLayout.getLayout(layoutId);
		final Map<String, CardBean> cardModel = retrieveModelFromDatas(datas, layout, false);
		final LayoutViewModel layoutViewModel = prepareLayoutViewModel(layout, cardModel);
		final LayoutManagerViewModel layoutManagerViewModel = prepareLayoutManagerViewModel(layout);
		layoutManagerViewModel.setLayoutViewModel(layoutViewModel);
		datas.put(LAYOUT_MANAGER_VIEW_MODEL, layoutManagerViewModel);
	}

}
