/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.layout.service;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.kosmos.layout.Layout;
import com.kosmos.layout.dao.LayoutDAO;
import com.kosmos.layout.exception.DefaultLayoutNotFoundException;
import com.kosmos.layout.exception.LayoutNotFoundException;
import com.kosmos.layout.manager.LayoutManager;
import com.kosmos.layout.meta.bean.LayoutMetaBean;
import com.kosmos.layout.meta.dao.LayoutMetaDAO;
import com.kportal.core.config.MessageHelper;
import com.kportal.extension.module.AbstractBeanManager;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.utils.FicheUnivMgr;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ServiceLayout extends AbstractBeanManager {

    public static final String ID_BEAN = "serviceLayout";
    private final Map<Class<?>, LayoutManager> layoutManagers = new HashMap<>();
    private LayoutDAO layoutDao;
    private LayoutMetaDAO layoutMetaDao;
    private Map<String, String> defaultLayouts = new HashMap<>();

    public LayoutMetaDAO getLayoutMetaDao() {
        return layoutMetaDao;
    }

    public void setLayoutMetaDao(LayoutMetaDAO layoutMetaDao) {
        this.layoutMetaDao = layoutMetaDao;
    }

    public LayoutDAO getLayoutDao() {
        return layoutDao;
    }

    public void setLayoutDao(LayoutDAO layoutDao) {
        this.layoutDao = layoutDao;
    }

    public Map<String, String> getDefaultLayouts() {
        return defaultLayouts;
    }

    public void setDefaultLayouts(Map<String, String> defaultLayouts) {
        this.defaultLayouts = defaultLayouts;
    }

    public void refresh() {
        Collection<LayoutManager> managers = ApplicationContextManager.getAllBeansOfType(LayoutManager.class).values();
        for (LayoutManager currentManager : managers) {
            this.layoutManagers.put(currentManager.getClass(), currentManager);
        }
    }

    /**
     * Method used to retrieve a Collection of layout.
     *
     * @return a layout Collection
     */
    public Collection<Layout> getAvailableLayouts() {
        try {
            return layoutDao.getAllLayouts();
        } catch (DataSourceException e) {
            return null;
        }
    }

    /**
     * Allow to retrieve a given layout.
     *
     * @param layoutId
     *         : the id of the layout to retrieve.
     * @return found layout.
     * @throws LayoutNotFoundException
     *         : thrown when no eligible layout was found.
     */
    public Layout getLayout(Long layoutId) throws LayoutNotFoundException {
        try {
            final Layout layout = layoutDao.getById(layoutId);
            if (layout == null) {
                throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_ID"), layoutId));
            }
            return layout;
        } catch (DataSourceException e) {
            throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_ID"), layoutId), e);
        }
    }

    /**
     * Allow to retrieve a layout according to its name.
     *
     * @param layoutName
     *         : the name of the layout as it appears in the data source.
     * @return found layout.
     * @throws LayoutNotFoundException
     *         : thrown when no eligible layout was found.
     */
    public Layout getLayout(String layoutName) throws LayoutNotFoundException {
        if (StringUtils.isNotBlank(layoutName)) {
            try {
                return layoutDao.getByName(layoutName);
            } catch (DataSourceException e) {
                throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_NAME"), layoutName), e);
            }
        }
        return null;
    }

    /**
     * Allow to retrieve a layout given a "FicheUniv" object.
     *
     * @param fiche
     *         : the "FicheUniv" object to search from.
     * @return layout attached to the "FicheUniv" object.
     * @throws LayoutNotFoundException
     *         : thrown if no attached layout was found.
     * @throws DefaultLayoutNotFoundException
     *         : thrown if no default layout was found for the fiche type.
     */
    public Layout getLayout(FicheUniv fiche) throws DefaultLayoutNotFoundException, LayoutNotFoundException {
        Metatag meta;
        Layout layout;
        try {
            meta = FicheUnivMgr.lireMeta(fiche);
        } catch (Exception e) {
            throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_FICHE"), fiche.getCode()), e);
        }
        // Tentative de récupération du layout par défaut pour la fiche.
        try {
            layout = getLayoutForMeta(meta.getIdMetatag());
            return layout;
        } catch (LayoutNotFoundException e) {
            final String classObject = ReferentielObjets.getClasseObjet(ReferentielObjets.getCodeObjet(fiche));
            final String defaultLayout = defaultLayouts.get(classObject);
            layout = getLayout(defaultLayout);
            if (layout == null) {
                throw new DefaultLayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.DEFAULT_NOT_FOUND"), classObject));
            }
            return layout;
        }
    }

    /**
     * Allow to retrieve a layout according to a Metatag id.
     *
     * @param metaId
     *         : the id of the meta to serach from.
     * @return layout attached to the meta id.
     * @throws DataSourceException
     *         : thrown if no eligible layout was found.
     */
    public Layout getLayoutForMeta(Long metaId) throws LayoutNotFoundException {
        final LayoutMetaBean layoutMeta;
        try {
            layoutMeta = layoutMetaDao.getByMetaId(metaId);
            if (layoutMeta != null) {
                return layoutDao.getById(layoutMeta.getIdLayout());
            }
        } catch (DataSourceException e) {
            throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_META_ID"), metaId), e);
        }
        throw new LayoutNotFoundException(String.format(MessageHelper.getCoreMessage("BO.LAYOUT.EXCEPTION.NOT_FOUND_META_ID"), metaId));
    }

    /**
     * Useful method to save bind a Layout and a Metatag
     *
     * @param layout
     *         : the layout to bind.
     * @param metaId
     *         : id of the Metatag to bind.
     * @throws DataSourceException
     *         : thrown if one of the following occurs : <ul> <li>The creation and persistence of the binding
     *         failed</li> <li>The existing bind could not be updated</li> </ul>
     */
    public void addLayoutForMeta(Layout layout, Long metaId) throws DataSourceException {
        LayoutMetaBean layoutMeta = layoutMetaDao.getByMetaId(metaId);
        if (layoutMeta != null) {
            layoutMeta.setIdLayout(layout.getId());
            layoutMetaDao.update(layoutMeta);
        } else {
            layoutMeta = new LayoutMetaBean();
            layoutMeta.setIdLayout(layout.getId());
            layoutMeta.setIdMeta(metaId);
            layoutMetaDao.add(layoutMeta);
        }
    }

    /**
     * Useful method to delete the reference between a layout a the given meta id.
     *
     * @param metaId
     *         : the id of the meta to delete the reference to.
     */
    public void deleteLayoutForMeta(Long metaId) throws DataSourceException {
        LayoutMetaBean layoutMeta = layoutMetaDao.getByMetaId(metaId);
        layoutMetaDao.delete(layoutMeta.getId());
    }

    /**
     * Useful method to retrieve a Collection of LayoutManager of the given or inheriting class.
     *
     * @param managerClass
     *         : the LayoutManager's class to retrieve.
     * @param <T>
     *         : should extends LayoutManager class.
     * @return a Collection of LayoutManager full-filling the class condition.
     */
    public <T extends LayoutManager> Collection<T> getLayoutManagers(Class<T> managerClass) {
        final List<T> managers = new ArrayList<>();
        for (Map.Entry<Class<?>, LayoutManager> currentEntry : this.layoutManagers.entrySet()) {
            if (ClassUtils.isAssignable(currentEntry.getKey(), managerClass)) {
                managers.add(managerClass.cast(currentEntry.getValue()));
            }
        }
        return managers;
    }
}
