/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.plansite;

import java.util.Collection;
import java.util.Iterator;

import com.univ.objetspartages.om.InfosRubriques;
import com.univ.utils.ContexteUniv;
import com.univ.utils.UnivWebFmt;

/**
 * Classe permettant de générer un plan de site.
 * 
 * @author FBI
 */
public class PlanSite {

	/** The ctx. */
	private ContexteUniv ctx = null;

	/** The rubrique mere. */
	private InfosRubriques rubriqueMere = null;

	/** The nb niveaux. */
	private int nbNiveaux = 0;

	/**
	 * Constructeur.
	 * 
	 * @param ctx
	 *            Le contexte
	 * @param rubriqueMere
	 *            La rubrique à partir de laquelle on construit le plan
	 * @param nbNiveaux
	 *            Le nombre de niveaux du plan à partir de la rubrique mère (0 = pas de limitation de niveau)
	 */
	public PlanSite(final ContexteUniv ctx, final InfosRubriques rubriqueMere, final int nbNiveaux) {
		this.ctx = ctx;
		this.rubriqueMere = rubriqueMere;
		this.nbNiveaux = nbNiveaux;
	}

	/**
	 * Renvoie le plan du site pour la rubrique et le nombre de niveaux demandés.
	 * 
	 * @return Une chaine contenant le code HTML du plan de site
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public String getPlan() throws Exception {
		return buildPlan(rubriqueMere, 1);
	}

	/**
	 * Méthode récursive construisant le plan de site noeud par noeud.
	 * 
	 * @param rubrique
	 *            La rubrique à partir de laquelle on construit le noeud
	 * @param niveau
	 *            Le niveau du noeud dans l'arbre
	 * 
	 * @return the string
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private String buildPlan(final InfosRubriques rubrique, final int niveau) throws Exception {
		final StringBuffer sb = new StringBuffer();
		final Collection<InfosRubriques> cSousRubriques = rubrique.getListeSousRubriquesFront(ctx);
		if (!cSousRubriques.isEmpty()) {
			final Iterator<InfosRubriques> it = cSousRubriques.iterator();
			InfosRubriques rubCourante = null;
			sb.append("<ul class=\"plan-site__" + niveau + "\">");
			String urlAccueil = "";
			while (it.hasNext()) {
				rubCourante = it.next();
				// ECLISSON : 0004047 : ajout du test if
				if (rubCourante != null) {
					sb.append("<li>");
					sb.append("<span class=\"plan-site__rubrique\">");
					urlAccueil = UnivWebFmt.renvoyerUrlAccueilRubrique(ctx, rubCourante.getCode());
					if (urlAccueil.length() > 0) {
						sb.append("<a href=\"" + urlAccueil + "\">");
					}
					sb.append(rubCourante.getIntitule());
					if (urlAccueil.length() > 0) {
						sb.append("</a>");
					}
					sb.append("</span>");
					if (niveau < nbNiveaux || nbNiveaux == 0) {
						sb.append(buildPlan(rubCourante, niveau + 1)); // appel récursif
					}
					sb.append("</li>");
				}
			}
			sb.append("</ul>");
		}
		return sb.toString();
	}
}
