/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.registration.dao.impl;

import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.registration.bean.Model;
import com.kosmos.registration.dao.ModelDao;
import com.kosmos.registration.utils.RegistrationJacksonMapper;
import com.kosmos.registration.wrapper.ActionConfigurationMap;
import com.univ.utils.json.Views;

import javax.sql.DataSource;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Created on 06/01/15.
 */
public class DefaultModelDao implements ModelDao {

    private DataSource dataSource;

    public void setDataSource(final DataSource dataSource) {
        this.dataSource = dataSource;
    }

    @Override
    public Model getById(Long id) {
        Model result;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `MODEL` T1 WHERE T1.ID_MODEL = ?"))) {
            stmt.setLong(1, id);
            try (ResultSet rs = stmt.executeQuery()) {
                if (!rs.first()) {
                    return null;
                }
                result = fill(rs);
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occured retrieving object with id %s from table MODEL", id.toString()), e);
        }
        return result;
    }

    @Override
    public Collection<Model> getModelByDescriptorID(String modelDescriptorID) {
        Collection<Model> result = new ArrayList<>();
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `MODEL` T1 WHERE T1.MODEL_DESCRIPTOR_ID = ?"))) {
            stmt.setString(1, modelDescriptorID);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    result.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occured retrieving object with descriptorID %s from table MODEL", modelDescriptorID), e);
        }
        return result;
    }

    @Override
    public Model add(Model model) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into MODEL (MODEL_DESCRIPTOR_ID, MODEL_DATAS, ACTIONS_CONFIGURATIONS) values (?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            stmt.setString(1, model.getModelDescriptorId());
            stmt.setString(2, RegistrationJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(model));
            stmt.setString(3, serializeActionConfigurationMap(model));
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"MODEL\" : %d row(s) affected", model.toString(), rowsAffected));
            }
            try (ResultSet rs = stmt.getGeneratedKeys()) {
                rs.next();
                model.setId(rs.getLong(1));
            }
        } catch (final Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"MODEL\"", model.toString()), e);
        }
        return model;
    }

    private String serializeActionConfigurationMap(Model model) throws IOException {
        ActionConfigurationMap actionMap = new ActionConfigurationMap();
        actionMap.putAll(model.getActionConfigurationById());
        return RegistrationJacksonMapper.getMapper().writeValueAsString(actionMap);
    }

    @Override
    public Model update(Model model) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update MODEL set MODEL_DESCRIPTOR_ID = ?, MODEL_DATAS = ?, ACTIONS_CONFIGURATIONS = ? WHERE ID_MODEL = ? ", Statement.RETURN_GENERATED_KEYS)) {
            stmt.setString(1, model.getModelDescriptorId());
            stmt.setString(2, RegistrationJacksonMapper.getMapper().writerWithView(Views.DaoView.class).writeValueAsString(model));
            stmt.setString(3, serializeActionConfigurationMap(model));
            stmt.setLong(4, model.getId());
            stmt.executeUpdate();
        } catch (final SQLException | IOException e) {
            throw new UpdateToDataSourceException(String.format("Unable to add [%s] to table \"MODEL\"", model.toString()), e);
        }
        return model;
    }

    @Override
    public void delete(Long id) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("delete from MODEL WHERE ID_MODEL = ?")) {
            stmt.setObject(1, id, Types.BIGINT);
            stmt.executeUpdate();
        } catch (final SQLException e) {
            throw new DeleteFromDataSourceException(String.format("An error occured during deletion of row with id %d from table \"MODEL\"", id), e);
        }
    }

    @Override
    public List<Model> select(String request) {
        throw new UnsupportedOperationException("This operation can't be performed in this context.");
    }

    private Model fill(final ResultSet rs) {
        Model model = new Model();
        try {
            model = RegistrationJacksonMapper.getMapper().readValue(rs.getString("MODEL_DATAS"), Model.class);
            model.setId(rs.getLong("ID_MODEL"));
            model.setModelDescriptorId(rs.getString("MODEL_DESCRIPTOR_ID"));
            model.setActionConfigurationById(RegistrationJacksonMapper.getMapper().readValue(rs.getString("ACTIONS_CONFIGURATIONS"), ActionConfigurationMap.class));
        } catch (SQLException e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"REGISTRATION\"", model.toString()), e);
        } catch (IOException e) {
            throw new UpdateToDataSourceException("Unable to map the saved json values", e);
        }
        return model;
    }
}
