/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.registration.dao.impl;

import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.registration.action.history.ActionHistory;
import com.kosmos.registration.bean.Registration;
import com.kosmos.registration.bean.RegistrationData;
import com.kosmos.registration.bean.RegistrationState;
import com.kosmos.registration.dao.RegistrationDao;
import com.kosmos.registration.utils.RegistrationJacksonMapper;
import com.kosmos.registration.wrapper.ActionHistoryByActionID;
import com.kosmos.registration.wrapper.ActionHistoryList;
import com.kosmos.registration.wrapper.RegistrationDataMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.codehaus.jackson.map.ObjectMapper;

import javax.sql.DataSource;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * DAO permettant de persister les données d'une inscription
 */
public class DefaultRegistrationDao implements RegistrationDao {

    //private static final Logger LOG = LoggerFactory.getLogger(DatabaseRegistrationDao.class);

    private DataSource dataSource;

    public void setDataSource(final DataSource dataSource) {
        this.dataSource = dataSource;
    }


    @Override
    public Registration add(Registration registration) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("INSERT INTO REGISTRATION (MODEL_ID, REGISTRATIONS_DATA_BY_ACTION, ACTION_HISTORY, STATE, CREATION_IDENTITY, CREATION_DATE, LAST_UPDATE_IDENTITY, LAST_UPDATE_DATE) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            stmt.setLong(1, registration.getModelId());
            stmt.setString(2, serializeDataByAction(registration));
            stmt.setString(3, serializeHistory(registration));
            stmt.setString(4, registration.getState().name());
            stmt.setString(5, registration.getCreationIdentity());
            stmt.setTimestamp(6, new Timestamp(registration.getCreationDate().getTime()));
            stmt.setString(7, registration.getLastUpdateIdentity());
            stmt.setTimestamp(8, new Timestamp(registration.getLastUpdateDate().getTime()));
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"REGISTRATION\" : %d row(s) affected", registration.toString(), rowsAffected));
            }
            try (ResultSet rs = stmt.getGeneratedKeys()) {
                rs.next();
                registration.setId(rs.getLong(1));
            }
        } catch (final SQLException | IOException e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"REGISTRATION\"", registration.toString()), e);
        }
        return registration;
    }

    private String serializeDataByAction(Registration registration) throws IOException {
        final RegistrationDataMap registrationDataMap = new RegistrationDataMap();
        registrationDataMap.putAll(registration.getDataByAction());
        return RegistrationJacksonMapper.getMapper().writeValueAsString(registrationDataMap);
    }

    private String serializeHistory(Registration registration) throws IOException {
        ActionHistoryList actionHistories = new ActionHistoryList();
        actionHistories.addAll(registration.getActionHistory());
        return RegistrationJacksonMapper.getMapper().writeValueAsString(actionHistories);
    }

    @Override
    public Registration update(Registration registration) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("UPDATE REGISTRATION SET MODEL_ID = ?, REGISTRATIONS_DATA_BY_ACTION = ?, ACTION_HISTORY = ?, STATE = ?, CREATION_IDENTITY = ?, CREATION_DATE= ?, LAST_UPDATE_IDENTITY = ?, LAST_UPDATE_DATE = ? WHERE ID_REGISTRATION = ?")) {
            stmt.setString(1, registration.getModelId().toString());
            stmt.setString(2, serializeDataByAction(registration));
            stmt.setString(3, serializeHistory(registration));
            stmt.setString(4, registration.getState().name());
            stmt.setString(5, registration.getCreationIdentity());
            stmt.setTimestamp(6, new Timestamp(registration.getCreationDate().getTime()));
            stmt.setString(7, registration.getLastUpdateIdentity());
            stmt.setTimestamp(8, new Timestamp(registration.getLastUpdateDate().getTime()));
            stmt.setLong(9, registration.getId());
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"REGISTRATION\"", registration.toString()), e);
        }
        return registration;
    }

    @Override
    public void delete(Long id) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("DELETE FROM REGISTRATION WHERE ID_REGISTRATION = ?")) {
            stmt.setObject(1, id, Types.BIGINT);
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected == 0) {
                throw new DeleteFromDataSourceException(String.format("Table \"REGISTRATION\" doesn't contain any row width id %d", id));
            } else if (rowsAffected > 1) {
                throw new DeleteFromDataSourceException(String.format("Table \"REGISTRATION\" contains more than one row width id %d", id));
            }
        } catch (final SQLException e) {
            throw new DeleteFromDataSourceException(String.format("An error occured during deletion of row with id %d from table \"REGISTRATION\"", id), e);
        }
    }

    @Override
    public Registration getById(Long id) {
        Registration result;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `REGISTRATION` T1 WHERE T1.ID_REGISTRATION = ?"))) {
            stmt.setLong(1, id);
            try (ResultSet rs = stmt.executeQuery()) {
                if (!rs.first()) {
                    return null;
                }
                result = fill(rs);
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occured retrieving object with id %d from table REGISTRATION", id), e);
        }
        return result;
    }

    @Override
    public Collection<Registration> getByModel(Long modelId) {
        Collection<Registration> results = new ArrayList<>();
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `REGISTRATION` T1 WHERE T1.MODEL_ID = ?"))) {
            stmt.setLong(1, modelId);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    results.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occured retrieving object with id %s from table REGISTRATION", modelId.toString()), e);
        }
        return results;
    }

    @Override
    public Collection<Registration> getByModelAndState(Long modelId, RegistrationState state) {
        Collection<Registration> results = new ArrayList<>();
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(String.format("select * from `REGISTRATION` T1 WHERE T1.MODEL_ID = ? AND STATE = ? "))) {
            stmt.setLong(1, modelId);
            stmt.setString(2, state.name());
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    results.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occured retrieving object with id %s from table REGISTRATION", modelId.toString()), e);
        }
        return results;
    }

    @Override
    public void deleteByModelId(Long modelId) {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("DELETE FROM `REGISTRATION` WHERE MODEL_ID = ?")) {
            stmt.setLong(1, modelId);
            stmt.executeUpdate();
        } catch (final SQLException e) {
            throw new DeleteFromDataSourceException(String.format("An error occured during deletion of row with MODEL_ID %d from table \"REGISTRATION\"", modelId), e);
        }
    }

    @Override
    public List<Registration> select(String request) {
        throw new UnsupportedOperationException("this operation can't be performed in this context.");
    }

    private Registration fill(final ResultSet rs) throws UpdateToDataSourceException {
        Registration registration = new Registration();
        String registrationDataStored = StringUtils.EMPTY;
        try {
            registration.setId(rs.getLong("ID_REGISTRATION"));
            registration.setModelId(rs.getLong("MODEL_ID"));
            registrationDataStored = rs.getString("REGISTRATIONS_DATA_BY_ACTION");
            ObjectMapper mapper = RegistrationJacksonMapper.getMapper();
            registration.setDataByAction(mapper.<Map<String, RegistrationData>>readValue(registrationDataStored, mapper.getTypeFactory().constructMapLikeType(Map.class, String.class, RegistrationData.class)));
            registration.setActionHistory(mapper.<List<Pair<String, ActionHistory>>>readValue(rs.getString("ACTION_HISTORY"), mapper.getTypeFactory().constructCollectionType(List.class, ActionHistoryByActionID.class)));
            registration.setState(RegistrationState.valueOf(rs.getString("STATE")));
            registration.setCreationIdentity(rs.getString("CREATION_IDENTITY"));
            registration.setCreationDate(rs.getTimestamp("CREATION_DATE"));
            registration.setCreationIdentity(rs.getString("LAST_UPDATE_IDENTITY"));
            registration.setLastUpdateDate(rs.getTimestamp("LAST_UPDATE_DATE"));
        } catch (SQLException e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"REGISTRATION\"", registration.toString()), e);
        } catch (IOException e) {
            throw new UpdateToDataSourceException(String.format("Unable to map RegistrationData from the saved value [%s]", registrationDataStored), e);
        }
        return registration;
    }

}
