/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.registration.service.impl;

import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.registration.bean.Model;
import com.kosmos.registration.bean.Registration;
import com.kosmos.registration.bean.RegistrationState;
import com.kosmos.registration.dao.RegistrationDao;

import java.util.Collection;

/**
 * Service permettant de gérer les {@link Registration}. Il sert pour les opérations CRUD
 */
public class RegistrationService {

    public static final String ID_BEAN = "registrationService";

    private RegistrationDao registrationDao;

    public void setRegistrationDao(RegistrationDao dao) {
        this.registrationDao = dao;
    }

    public Registration get(Long id) {
        return registrationDao.getById(id);
    }

    /**
     * Permet de créer une nouvelle registration
     *
     * @param registration la registration à enregistrer
     * @return la registration enregistré avec son nouvel id généré par la bdd
     * @throws AddToDataSourceException lors de la requête en bdd, (accès base, contraintes SQL non respectées...)
     */
    public Registration create(Registration registration) throws AddToDataSourceException {
        return registrationDao.add(registration);
    }

    /**
     * Met à jour la registration fourni en paramètre
     *
     * @param registration la registration à mettre à jour
     * @return La registration mise à jour
     * @throws UpdateToDataSourceException si l'update en bdd ne fonctionne plus
     */
    public Registration update(Registration registration) throws UpdateToDataSourceException {
        return registrationDao.update(registration);
    }

    /**
     * Permet de supprimer une Registration depuis son ID
     *
     * @param id l'id de la regsitration à supprimer
     * @throws DeleteFromDataSourceException Erreur de bdd
     */
    public void deleteById(Long id) throws DeleteFromDataSourceException {
        registrationDao.delete(id);
    }

    /**
     * Récupère les {@link Registration} lié au {@link Model} dont l'id est fourni en paramètre.
     *
     * @param modelId l'id du modèle dont on souhaite récupérer les Registration
     * @return L'ensemble des Registration liés au {@link Model} ou une collection vide si non trouvé
     * @throws DataSourceException lors de la requête en bdd
     */
    public Collection<Registration> getRegistrationsByModel(Long modelId) throws DataSourceException {
        return registrationDao.getByModel(modelId);
    }

    /**
     * Récupère les {@link Registration} lié au {@link Model} dont l'id est fourni en paramètre et dont le {@link RegistrationState} correspond à celui fourni.
     *
     * @param modelId l'id du modèle dont on souhaite récupérer les Registration
     * @param state   l'état dans lesquels on veut récupérer les Registrations
     * @return L'ensemble des Registration liés au {@link Model} ou une collection vide si non trouvé
     * @throws DataSourceException lors de la requête en bdd
     */
    public Collection<Registration> getRegistrationsByModelAndState(Long modelId, RegistrationState state) throws DataSourceException {
        return registrationDao.getByModel(modelId);
    }

    public void deleteByModelId(Long modelId) {
        registrationDao.deleteByModelId(modelId);
    }
}
