/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.userfront.action;

import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.mail.MessagingException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.mail.EmailException;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.email.JSBMailbox;
import com.kosmos.registration.action.DefaultActionWithoutConfiguration;
import com.kosmos.registration.action.bean.EmptyActionConfiguration;
import com.kosmos.registration.action.history.ActionHistory;
import com.kosmos.registration.bean.Registration;
import com.kosmos.registration.bean.RegistrationState;
import com.kosmos.registration.exception.RegistrationExecutionException;
import com.kosmos.registration.exception.RegistrationValidationException;
import com.kosmos.registration.service.impl.RegistrationService;
import com.kosmos.userfront.bean.UserRegistrationData;
import com.kosmos.userfront.history.EmailActionHistory;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.security.MySQLHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.ContexteDao;
import com.univ.utils.URLResolver;

/**
 * Classe permettant de transformer une Registration en Utilisateur K-Portal.
 * La validation gère l'unicité du login sur les Registration précédente ainsi que sur les utilisateurs précédents.
 *
 */
public class DefaultUserCreation extends DefaultActionWithoutConfiguration<UserRegistrationData> {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultUserCreation.class);

    @Override
    public void validatePrecondition(UserRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, final String actionId) throws RegistrationValidationException {
        super.validatePrecondition(data, configuration, registration, identity, actionId);
        final String login = data.getLogin();
        Utilisateur utilisateurByLogin = null;
        try {
            utilisateurByLogin = Utilisateur.getUtilisateur(login);
        } catch (Exception e) {
            LOG.error("erreur lors de la récupération de l'utilisateur ayant pour login " + login);
        }
        if (utilisateurByLogin != null && utilisateurByLogin.getCode().equals(login)) {
            throw new RegistrationValidationException("ERROR.USERFRONT.LOGIN_ALREADY_EXIST");
        }
        try {
            data.setPassword(MySQLHelper.encodePassword(data.getPassword()));
            data.setPasswordConfirmation(data.getPassword());
        } catch (UnsupportedEncodingException e) {
            LOG.info("l'encodage du password n'a pas fonctionné", e);
        }
        RegistrationService registrationService = ApplicationContextManager.getCoreContextBean(RegistrationService.ID_BEAN, RegistrationService.class);
        if (registration != null) {
            final Long registrationId = registration.getId();
            Collection<Registration> thisModelRegistrations = registrationService.getRegistrationsByModel(registration.getModelId());
            boolean existValues = CollectionUtils.exists(thisModelRegistrations, new Predicate() {

                @Override
                public boolean evaluate(Object object) {
                    Registration registration = (Registration) object;
                    UserRegistrationData dataValues = (UserRegistrationData) registration.getDataByAction().get(actionId);
                    List<Pair<String, ActionHistory>> actionsHistory = registration.getActionHistory();
                    boolean isValid = isWaitingForValidEmailValidation(registration, actionsHistory);
                    return isValid && !(RegistrationState.ERROR.equals(registration.getState()) || RegistrationState.FINISHED.equals(registration.getState())) && !registration.getId().equals(registrationId) && login.equals(dataValues.getLogin());
                }

                private boolean isWaitingForValidEmailValidation(final Registration registration, final List<Pair<String, ActionHistory>> actionsHistory) {
                    boolean isValid = Boolean.TRUE;
                    DateTime now = DateTime.now();
                    DateTime yesterday = now.minusDays(1);
                    if (actionsHistory.get(actionsHistory.size() - 1).getValue() instanceof EmailActionHistory && RegistrationState.WAITING.equals(registration.getState())) {
                        isValid = yesterday.isBefore(registration.getCreationDate().getTime());
                    }
                    return isValid;
                }
            });
            if (existValues) {
                throw new RegistrationValidationException("ERROR.USERFRONT.LOGIN_ALREADY_EXIST");
            }
        }
    }

    @Override
    public ActionHistory execute(UserRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, String actionId) throws RegistrationExecutionException {
        Utilisateur userToCreate = new Utilisateur();
        userToCreate.init();
        userToCreate.setCode(data.getLogin());
        userToCreate.setNom(data.getLastName());
        userToCreate.setPrenom(data.getName());
        userToCreate.setMotDePasse(data.getPassword());
        userToCreate.setAdresseMail(data.getEmail());
        setNotByDefaultValue(data, userToCreate);
        try (ContexteDao ctx = new ContexteDao()) {
            userToCreate.setCtx(ctx);
            userToCreate.add();
        } catch (Exception e) {
            throw new RegistrationExecutionException("ERROR.USERFRONT.USER.CREATION", e);
        }
        ActionHistory history = new ActionHistory();
        Locale locale = LangueUtil.getLocale(data.getIdLocaleKportal());
        String emailConfirmationTitle = MessageHelper.getCoreMessage(locale, "USERFRONT.EMAIL.ACCOUNT_CREAIED");
        InfosSite infosSite = Site.renvoyerItemSite(data.getAliasSite());
        String identificationUrl = URLResolver.getAbsoluteUrl(WebAppUtil.SG_PATH + "?PROC=IDENTIFICATION_FRONT&ACTION=CONNECTER", infosSite, URLResolver.LOGIN_FRONT);
        String emailConfirmationContent = MessageHelper.getCoreMessage(locale, "USERFRONT.EMAIL.ACCOUNT_CREATED_BODY");
        final JSBMailbox mailbox = new JSBMailbox(false);
        try {
            mailbox.sendWmasterMsg(data.getEmail(), emailConfirmationTitle, String.format(emailConfirmationContent, identificationUrl));
        } catch (MessagingException | EmailException e) {
            LOG.error("impossible d'envoyer un email à l'utilisateur ayant créé un compte", e);
        }
        history.setCreationDate(new Date());
        history.setState(RegistrationState.FINISHED);
        return history;
    }

    private void setNotByDefaultValue(UserRegistrationData data, Utilisateur userToCreate) {
        if (data.getBirthdate() != null) {
            userToCreate.setDateNaissance(new java.sql.Date(data.getBirthdate().getTime()));
        }
        if (data.getTitle() != null) {
            userToCreate.setCivilite(data.getTitle());
        }
        if (data.getInstitutionsCodes() != null) {
            userToCreate.setCodeRattachement(data.getInstitutionsCodes());
        }
        if (data.getGroups() != null) {
            userToCreate.setGroupes(data.getGroups());
        }
        if (data.getValidationRestriction() != null) {
            userToCreate.setRestrictionValidation(data.getValidationRestriction());
        }
        if (data.getEditionExtension() != null) {
            userToCreate.setExtensionModification(data.getEditionExtension());
        }
        if (data.getInterests() != null) {
            userToCreate.setCentresInteret(data.getInterests());
        }
        if (data.getProfiles() != null) {
            userToCreate.setProfilDsi(data.getProfiles());
        }
        if (data.getDsiGroups() != null) {
            userToCreate.setGroupesDsi(data.getDsiGroups());
        }
        if (data.getLdapCode() != null) {
            userToCreate.setCodeLdap(data.getLdapCode());
        }
        if (data.getDsiGroupsFromImport() != null) {
            userToCreate.setGroupesDsiImport(data.getDsiGroupsFromImport());
        }
        if (data.getRoles() != null) {
            userToCreate.setRoles(data.getRoles());
        }
        if (data.getDefaultProfile() != null) {
            userToCreate.setProfilDefaut(data.getDefaultProfile());
        }
        if (data.getMailFormat() != null) {
            userToCreate.setFormatEnvoi(data.getMailFormat());
        }
        if (data.getExpertMode() != null) {
            userToCreate.setModeSaisieExpert(data.getExpertMode());
        }
    }
}
