/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.userfront.action;

import java.util.Date;
import java.util.Locale;
import java.util.Set;
import java.util.UUID;

import javax.mail.MessagingException;
import javax.validation.ConstraintViolation;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.mail.EmailException;
import org.joda.time.DateTime;

import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.core.ProcessusHelper;
import com.jsbsoft.jtf.email.JSBMailbox;
import com.kosmos.registration.action.ActionWithStep;
import com.kosmos.registration.action.bean.EmptyActionConfiguration;
import com.kosmos.registration.action.history.ActionHistory;
import com.kosmos.registration.bean.Registration;
import com.kosmos.registration.bean.RegistrationState;
import com.kosmos.registration.exception.RegistrationExecutionException;
import com.kosmos.registration.exception.RegistrationValidationException;
import com.kosmos.registration.exception.RegistrationViolationValidationException;
import com.kosmos.registration.utils.RegistrationUtils;
import com.kosmos.userfront.bean.EmailRegistrationData;
import com.kosmos.userfront.history.EmailActionHistory;
import com.kosmos.validation.ValidationUtils;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.config.PropertyHelper;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.utils.URLResolver;

/**
 * Classe permettant d'envoyer un email de confirmation et de gérer la validation par email.
 */
public class EmailConfirmation implements ActionWithStep<EmailRegistrationData,EmptyActionConfiguration> {

	@Override
	public void validateConfiguration(EmptyActionConfiguration configuration) throws RegistrationViolationValidationException {
	}

	@Override
	public void validatePrecondition(EmailRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, String actionId) throws RegistrationValidationException {
		Set<ConstraintViolation<EmailRegistrationData>> violations = ValidationUtils.validateBean(data);
		if (CollectionUtils.isNotEmpty(violations)) {
			throw new RegistrationViolationValidationException("BEAN_VALIDATION.ERROR", violations);
		}
	}

	@Override
	public ActionHistory execute(EmailRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, String actionId) throws RegistrationExecutionException {
		EmailActionHistory history = new EmailActionHistory();
		history.setCreationDate(new Date());
		history.setGeneratedKey(UUID.randomUUID().toString());
		history.setState(RegistrationState.WAITING);
        Locale locale = LangueUtil.getLocale(data.getIdLocaleKportal());
        String confirmationEmail = MessageHelper.getCoreMessage(locale, "USERFRONT.EMAIL.CONFIRMATION_BODY");
        String titleConformation = MessageHelper.getCoreMessage(locale, "USERFRONT.EMAIL.CONFIRMATION_TITLE");
        String urlEmailConfirm = ProcessusHelper.getUrlProcessAction(null, data.getExtension(), data.getProcessus(), "CONFIRMATION", new String[][] {{"REGISTRATION_ID", registration.getId().toString()}, {"email_registration_data.generated_key", history.getGeneratedKey()}});
        InfosSite siteCourant = Site.renvoyerItemSite(data.getAliasSite());
        urlEmailConfirm = URLResolver.getAbsoluteUrl(urlEmailConfirm, siteCourant, URLResolver.LOGIN_FRONT);
        final JSBMailbox mailbox = new JSBMailbox(false);
		try {
            mailbox.sendWmasterMsg(data.getEmail(), String.format(titleConformation, siteCourant.getIntitule()), String.format(confirmationEmail, siteCourant.getIntitule(), StringUtils.replace(urlEmailConfirm,"&amp;","&")));
		} catch (MessagingException | EmailException e) {
			throw new RegistrationExecutionException("ERROR.USERFRONT.SENGING_MAIL",e);
		}
		return history;
	}


	@Override
	public void validateStepPrecondition(EmailRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, String actionID) throws RegistrationValidationException {
		if (data.getGeneratedKey() == null) {
			throw new RegistrationValidationException("ERROR.USERFRONT.EMPTY_GENERATED_KEY");
		}
		EmailActionHistory lastActionHistory = (EmailActionHistory) RegistrationUtils.getLastActionHistoryFromActionID(registration.getActionHistory(), actionID);
		DateTime date = DateTime.now();
		DateTime oneDayBefore = date.minusDays(1);
		if (oneDayBefore.isAfter(lastActionHistory.getCreationDate().getTime())) {
			throw new RegistrationValidationException("ERROR.USERFRONT.EMAIL_EXPIRED");
		}
	}

	@Override
	public ActionHistory executeStep(EmailRegistrationData data, EmptyActionConfiguration configuration, Registration registration, String identity, String actionID) throws RegistrationExecutionException {
		EmailActionHistory lastActionHistory = (EmailActionHistory) RegistrationUtils.getLastActionHistoryFromActionID(registration.getActionHistory(), actionID);
		EmailActionHistory incomingHistory = new EmailActionHistory();
		incomingHistory.setCreationDate(new Date());
		String generatedKey = lastActionHistory.getGeneratedKey();
		if (isValidKey(data,generatedKey)) {
			incomingHistory.setState(RegistrationState.FINISHED);
		} else {
			incomingHistory.setState(RegistrationState.ERROR);
		}
		return incomingHistory;
	}

	private boolean isValidKey(EmailRegistrationData userDatas, String generatedKey) {
		return StringUtils.isNotBlank(generatedKey) && generatedKey.equals(userDatas.getGeneratedKey());
	}

}
