/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.userfront.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import com.jsbsoft.jtf.datasource.dao.CommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DeleteFromDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.kosmos.userfront.bean.ModerationState;
import com.kosmos.userfront.bean.UserModerationData;

/**
 * Created by olivier.camon on 16/01/15.
 */
public class UserModerationDao implements CommonDAO<UserModerationData> {

	private DataSource dataSource;

	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}

	@Override
	public UserModerationData add(UserModerationData data) {
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement("insert into USER_MODERATION (EMAIL, LOGIN, FIRST_NAME, LAST_NAME, MODERATION_MESSAGE, MODERATED, CREATION_DATE, PARENT_ID_REGISTRATION) values (?, ?, ?, ?, ?, ?, ?, ? )", Statement.RETURN_GENERATED_KEYS)) {
			stmt.setString(1, data.getEmail());
			stmt.setString(2, data.getLogin());
			stmt.setString(3, data.getFirstName());
			stmt.setString(4, data.getLastName());
			stmt.setString(5, data.getModerationMessage());
			stmt.setString(6, data.getModerationState().name());
			stmt.setTimestamp(7, new Timestamp(data.getCreationDate().getTime()));
			stmt.setLong(8, data.getParentIdRegistration());
			final int rowsAffected = stmt.executeUpdate();
			if (rowsAffected != 1) {
				throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"USER_MODERATION\" : %d row(s) affected", data.toString(), rowsAffected));
			}
			try (ResultSet rs = stmt.getGeneratedKeys()) {
				rs.next();
				data.setId(rs.getLong(1));
			}
		} catch (final Exception e) {
			throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"USER_MODERATION\"", data.toString()), e);
		}
		return data;
	}

	@Override
	public UserModerationData update(UserModerationData data) {
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement("update USER_MODERATION set EMAIL = ? ,LOGIN = ?, FIRST_NAME = ?, LAST_NAME = ?,  MODERATION_MESSAGE = ?, MODERATED = ?, CREATION_DATE = ?, PARENT_ID_REGISTRATION = ? WHERE ID_USER_MODERATION = ? ", Statement.RETURN_GENERATED_KEYS)) {
			stmt.setString(1, data.getEmail());
			stmt.setString(2, data.getLogin());
			stmt.setString(3, data.getFirstName());
			stmt.setString(4, data.getLastName());
			stmt.setString(5, data.getModerationMessage());
			stmt.setString(6, data.getModerationState().name());
			stmt.setTimestamp(7, new Timestamp(data.getCreationDate().getTime()));
			stmt.setLong(8, data.getParentIdRegistration());
			stmt.setLong(9, data.getId());
			stmt.executeUpdate();
		} catch (final SQLException e) {
			throw new UpdateToDataSourceException(String.format("Unable to add [%s] to USER_MODERATION", data.toString()), e);
		}
		return data;
	}

	@Override
	public void delete(Long id) {
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement("delete from `USER_MODERATION` WHERE ID_USER_MODERATION = ?")) {
			stmt.setLong(1, id);
			final int rowsAffected = stmt.executeUpdate();
			if (rowsAffected == 0) {
				throw new DeleteFromDataSourceException(String.format("Table \"USER_MODERATION\" doesn't contain any row width id %d", id));
			} else if (rowsAffected > 1) {
				throw new DeleteFromDataSourceException(String.format("Table \"USER_MODERATION\" contains more than one row width id %d", id));
			}
		} catch (final SQLException e) {
			throw new DeleteFromDataSourceException(String.format("An error occured during deletion of row with id %d from table \"USER_MODERATION\"", id), e);
		}
	}

	@Override
	public UserModerationData getById(Long id) {
		UserModerationData result;
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(String.format("select * from `USER_MODERATION` T1 WHERE T1.ID_USER_MODERATION = ?"))) {
			stmt.setLong(1,id);
			try (ResultSet rs = stmt.executeQuery()) {
				if (!rs.first()) {
					return null;
				}
				result = fill(rs);
			}
		} catch (final SQLException e) {
			throw new DataSourceException(String.format("An error occured retrieving object with id %s from table USER_MODERATION", id.toString()), e);
		}
		return result;
	}

	@Override
	public List<UserModerationData> select(String request) {
		throw new UnsupportedOperationException("this method can not be called in this context");
	}

	public List<UserModerationData> selectAllOrderByCreattionDate() {
		List<UserModerationData> result = new ArrayList<>();
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement(String.format("select * from `USER_MODERATION` T1 ORDER BY CREATION_DATE DESC"))) {
			try (ResultSet rs = stmt.executeQuery()) {
				while (rs.next()) {
					result.add(fill(rs));
				}
			}
		} catch (final SQLException e) {
			throw new DataSourceException(String.format("An error occured retrieving objects from table USER_MODERATION"), e);
		}
		return result;
	}


	private UserModerationData fill(ResultSet rs) {
		UserModerationData result = new UserModerationData();
		try {
			result.setId(rs.getLong("ID_USER_MODERATION"));
			result.setEmail(rs.getString("EMAIL"));
			result.setLogin(rs.getString("LOGIN"));
			result.setFirstName(rs.getString("FIRST_NAME"));
			result.setLastName(rs.getString("LAST_NAME"));
			result.setModerationMessage(rs.getString("MODERATION_MESSAGE"));
			result.setModerationState(ModerationState.valueOf(rs.getString("MODERATED")));
			result.setCreationDate(rs.getTimestamp("CREATION_DATE"));
			result.setParentIdRegistration(rs.getLong("PARENT_ID_REGISTRATION"));
		} catch (SQLException e) {
			throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"USER_MODERATION\"", result.toString()), e);
		}
		return result;
	}

}
