/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.usinesite.service.impl;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Date;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.exception.ErreurDonneeNonTrouve;
import com.jsbsoft.jtf.exception.ErreurUniciteNonRespectee;
import com.kosmos.usinesite.exception.ErreursSaisieInfosSite;
import com.kosmos.usinesite.template.bean.TemplateSite;
import com.kosmos.usinesite.utils.InfosSiteHelper;
import com.univ.multisites.InfosSite;
import com.univ.multisites.bean.impl.InfosSiteImpl;
import com.univ.multisites.dao.InfosSiteDao;
import com.univ.multisites.service.ServiceInfosSite;
import com.univ.objetspartages.cache.CacheInfosSiteManager;
import com.univ.url.CacheUrlManager;

/**
 * Service qui permet de gérer des {@link InfosSite} depuis la source de données. Il permet de créer/modifier/supprimer et lister l'ensemble des infosSite de l'application.
 * 
 * @author pierre.cosson
 * 
 */
public class ServiceInfosSiteProcessus implements ServiceInfosSite {

	private static Logger LOG = LoggerFactory.getLogger(ServiceInfosSiteProcessus.class);

	/**
	 * DAO permettant l'accés au sauvegarde de {@link InfosSite}.
	 */
	protected InfosSiteDao infosSiteDao = null;

	public void setInfosSiteDao(final InfosSiteDao infosSiteDao) {
		this.infosSiteDao = infosSiteDao;
	}

	@Override
	public void creer(final InfosSiteImpl infosSite) throws Exception {
		creer(infosSite, null);
	}

	@Override
	public void creer(final InfosSiteImpl infosSite, final String codeUtilisateur) throws Exception {
		verfierUniciteCode(infosSite);
		checkHostUnicity(infosSite);
		peuplerDonneesSauvegardeEnCreation(infosSite, codeUtilisateur);
		infosSiteDao.creer(infosSite);
		flushAllCaches();
	}

	private void peuplerDonneesSauvegardeEnCreation(final InfosSiteImpl nouvelleSauvegarde, String codeUtilisateur) {
		if (codeUtilisateur == null) {
			codeUtilisateur = StringUtils.EMPTY;
		}
		nouvelleSauvegarde.setCodeCreateur(codeUtilisateur);
		nouvelleSauvegarde.setDateCreation(new Date());
		nouvelleSauvegarde.setCodeDernierModificateur(codeUtilisateur);
		nouvelleSauvegarde.setDateDerniereModification(new Date());
	}

	@Override
	public void modifier(final InfosSiteImpl infosSite, final TemplateSite template) throws Exception {
		modifier(infosSite, template, null);
	}

	@Override
	public void modifier(final InfosSiteImpl nouvelleSauvegarde, final TemplateSite template, final String codeUtilisateur) throws Exception {
		final InfosSite sauvegardeOriginale = infosSiteDao.getInfosSite(nouvelleSauvegarde.getAlias());
		peuplerDonneesSauvegardeEnModification(nouvelleSauvegarde, sauvegardeOriginale, codeUtilisateur);
		verfierUniciteCode(nouvelleSauvegarde);
		checkHostUnicity(nouvelleSauvegarde);
		infosSiteDao.miseAJour(nouvelleSauvegarde);
		flushAllCaches();
	}

	@Override
	public void supprimer(final String code) throws Exception {
		infosSiteDao.supprimer(code);
		flushAllCaches();
	}

	@Override
	public InfosSite getInfosSite(final String code) throws Exception {
		return infosSiteDao.getInfosSite(code);
	}

	@Override
	public Collection<InfosSite> getListeTousInfosSites() throws Exception {
		return infosSiteDao.getListeInfosSites();
	}

	@Override
	public void enregistrerFichier(final InfosSite infosSite, final File fichierTemporaireSource, final String nomPropertyTemplateSite) throws Exception {
		try {
			final String pathFichier = InfosSiteHelper.getPathAbsoluFichierPropertyTemplate(infosSite, nomPropertyTemplateSite);
			final File fichierUploade = new File(pathFichier);
			if (fichierUploade.exists()) {
				fichierUploade.delete();
			}
			FileUtils.moveFile(fichierTemporaireSource, fichierUploade);
		} catch (final ErreurDonneeNonTrouve e) {
			LOG.info("impossible de trouver le chemin du fichier", e);
		}
	}

	@Override
	public void supprimerFichier(final InfosSite infosSite, final String nomPropertyTemplateSite) throws Exception {
		try {
			final String pathFichier = InfosSiteHelper.getPathAbsoluFichierPropertyTemplate(infosSite, nomPropertyTemplateSite);
			final File fichierASupprimer = new File(pathFichier);
			if (fichierASupprimer.exists()) {
				fichierASupprimer.delete();
			}
		} catch (final ErreurDonneeNonTrouve e) {
			LOG.info("impossible de trouver le chemin du fichier", e);
		}
	}

	@Override
	public void supprimerTousFichiers(final InfosSite infosSite) throws IOException {
		final String pathDossier = InfosSiteHelper.getPathAbsoluFichiersTemplateDuSite(infosSite);
		final File dossierDepot = new File(pathDossier);
		if (dossierDepot.exists()) {
			FileUtils.deleteDirectory(dossierDepot);
		}
	}

	@Override
	public void cleanFichiers(final InfosSite infosSite) throws IOException {
		final String pathDossier = InfosSiteHelper.getPathAbsoluFichiersTemplateDuSite(infosSite);
		final File dossierDepot = new File(pathDossier);
		if (!dossierDepot.exists()) {
			return;
		}
		for (final File dossierPropertyTemplate : dossierDepot.listFiles()) {
			if (!dossierPropertyTemplate.isDirectory()) {
				// dans le dossier de dépôt, il doit seulement y avoir des
				// dossiers.
				dossierPropertyTemplate.delete();
				continue;
			}
			final String nomPropertyTemplate = dossierPropertyTemplate.getName();
			final Object valeurPropertyTemplate = infosSite.getProprieteComplementaire(nomPropertyTemplate);
			if (valeurPropertyTemplate == null || !(valeurPropertyTemplate instanceof String) || StringUtils.isEmpty((String) valeurPropertyTemplate)) {
				// le dossier ne correspond à aucuen propriété valide
				FileUtils.deleteDirectory(dossierPropertyTemplate);
				continue;
			}
			final String nomFichierAttendu = (String) valeurPropertyTemplate;
			for (final File fichierContenuDansDossier : dossierPropertyTemplate.listFiles()) {
				if (fichierContenuDansDossier.isDirectory()) {
					// pas de dossier à ce niveau, seulement des fichiers
					FileUtils.deleteDirectory(fichierContenuDansDossier);
					continue;
				}
				final String nomFichier = fichierContenuDansDossier.getName();
				if (!StringUtils.equalsIgnoreCase(nomFichierAttendu, nomFichier)) {
					fichierContenuDansDossier.delete();
				}
			}
		}
		// Suppression du dossier du site si vide
		if (dossierDepot.listFiles().length == 0) {
			FileUtils.deleteDirectory(dossierDepot);
		}
	}

	private void peuplerDonneesSauvegardeEnModification(final InfosSiteImpl nouvelleSauvegarde, final InfosSite sauvegardeOriginale, String codeUtilisateur) {
		if (codeUtilisateur == null) {
			codeUtilisateur = StringUtils.EMPTY;
		}
		nouvelleSauvegarde.setIdInfosSite(sauvegardeOriginale.getIdInfosSite());
		nouvelleSauvegarde.setCodeCreateur(sauvegardeOriginale.getCodeCreateur());
		nouvelleSauvegarde.setDateCreation(sauvegardeOriginale.getDateCreation());
		nouvelleSauvegarde.setCodeDernierModificateur(codeUtilisateur);
		nouvelleSauvegarde.setDateDerniereModification(new Date());
		nouvelleSauvegarde.setHistorique(sauvegardeOriginale.toString());
	}

	/**
	 * Vérifier que le code est déjà utilisé ou non dans la source de donnée.
	 * 
	 * @param sauvegarde
	 * @throws Exception
	 *             Erreur d'accés à la source de données.
	 */
	private void verfierUniciteCode(final InfosSite sauvegarde) throws Exception {
		final String code = sauvegarde.getAlias();
		try {
			final InfosSite sauvegardeBis = infosSiteDao.getInfosSite(code);
			if (sauvegarde.getIdInfosSite() == sauvegardeBis.getIdInfosSite()) {
				// cas d'une modification : le code est le même mais les IDs
				// aussi donc les mêmes sites
				return;
			}
		} catch (final ErreurDonneeNonTrouve e) {
			// Sauvegarde non trouvée donc code inexistant
			return;
		}
		throw new ErreurUniciteNonRespectee("Code '" + code + "' déjà utilisé pour un autre site.");
	}

	private void checkHostUnicity(final InfosSite infosSite) throws Exception {
		for(InfosSite currentInfosSite : getListeTousInfosSites()) {
			if(currentInfosSite.getAlias().equals(infosSite.getAlias())) {
				continue;
			}
			if(StringUtils.isNotBlank(infosSite.getHttpHostname())) {
				if (infosSite.getHttpHostname().equals(currentInfosSite.getHttpHostname()) || infosSite.getHttpHostname().equals(currentInfosSite.getHttpsHostname())) {
					throw new ErreursSaisieInfosSite("Le host http '" + infosSite.getHttpHostname() + "' est déjà utilisé pour le site " + currentInfosSite.getAlias(), infosSite, null);
				}
			}
			if(StringUtils.isNotBlank(infosSite.getHttpsHostname())) {
				if (infosSite.getHttpsHostname().equals(currentInfosSite.getHttpHostname()) || infosSite.getHttpsHostname().equals(currentInfosSite.getHttpsHostname())) {
					throw new ErreursSaisieInfosSite("Le host https '" + infosSite.getHttpsHostname() + "' est déjà utilisé pour le site " + currentInfosSite.getAlias(), infosSite, null);
				}
			}
		}
	}
	/**
	 * RAZ de tous les caches liés aux sites de l'application.
	 */
	private void flushAllCaches() {
		CacheInfosSiteManager.getInstance().asyncRefresh();
		CacheUrlManager.getInstance().asyncRefresh();
	}
}
