/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.usinesite.template.property.validateur.impl;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import com.kosmos.usinesite.exception.ErreursSaisieInfosSite;
import com.kosmos.usinesite.template.bean.TemplateSite;
import com.kosmos.usinesite.template.property.bean.impl.AbstractTemplateSiteProperty;
import com.kosmos.usinesite.template.property.validateur.TemplateSitePropertyValidateur;
import com.univ.multisites.InfosSite;

/**
 * Apporte des fonctions de bases pour le traitements des propriété de template utilisant des chaines de caractéres.
 *
 * @author pierre.cosson
 *
 */
public abstract class AbstractTemplateSitePropertyStringValidateur {

	protected static final String MSG_ERREUR_PROPRIETE_NON_AUTORISEE = "La propriété '%s' du template contient une valeur non autorisée : '%s'";

	protected static final String MSG_ERREUR_PROPRIETE_INVALIDE = "La propriété '%s' du template contient une valeur invalide : '%s'";

	protected static final String MSG_ERREUR_PROPRIETE_TAILLE = "La propriété '%s' du template dépasse les %d caractéres.";

	protected void testChampObligatoire(final String valeur, final AbstractTemplateSiteProperty property, final ArrayList<String> fluxErreurs) {
		if (property.isObligatoire() && StringUtils.isEmpty(valeur)) {
			final String messageErreur = String.format(TemplateSitePropertyValidateur.MSG_ERREUR_PROPRIETE_OBLIGATOIRE, property.getLibelle());
			fluxErreurs.add(messageErreur);
		}
	}

	protected void testTailleMaximum(final String valeur, final int tailleMaximum, final String libelle, final ArrayList<String> fluxErreurs) {
		if (StringUtils.isEmpty(valeur) || tailleMaximum == 0) {
			return;
		}
		if (StringUtils.length(valeur) > tailleMaximum) {
			final String messageErreur = String.format(MSG_ERREUR_PROPRIETE_TAILLE, libelle, tailleMaximum);
			fluxErreurs.add(messageErreur);
		}
	}

	protected void testRegExp(final String valeur, final String regExpValidation, final String libelle, final ArrayList<String> fluxErreurs) {
		if (StringUtils.isEmpty(valeur)) {
			return;
		}
		if (StringUtils.isEmpty(regExpValidation)) {
			return;
		}
		if (!valeur.matches(regExpValidation)) {
			final String messageErreur = String.format(MSG_ERREUR_PROPRIETE_INVALIDE, libelle, valeur);
			fluxErreurs.add(messageErreur);
		}
	}

	protected void testListeValeurs(final String valeur, final Collection<String> listeValeursAutorisees, final String libelle, final ArrayList<String> fluxErreurs) {
		if (StringUtils.isEmpty(valeur)) {
			return;
		}
		if (CollectionUtils.isEmpty(listeValeursAutorisees)) {
			return;
		}
		if (!listeValeursAutorisees.contains(valeur)) {
			final String messageErreur = String.format(MSG_ERREUR_PROPRIETE_NON_AUTORISEE, libelle, valeur);
			fluxErreurs.add(messageErreur);
		}
	}

	protected void traiterFluxErreurs(final ArrayList<String> fluxErreurs, final InfosSite infosSite, final TemplateSite template) throws ErreursSaisieInfosSite {
		if (!fluxErreurs.isEmpty()) {
			throw new ErreursSaisieInfosSite(fluxErreurs, infosSite, template);
		}
	}
}
