/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.usinesite.utils;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.kosmos.usinesite.reference.ServiceBeanReference;
import com.kportal.cms.objetspartages.annotation.GetterAnnotationHelper;
import com.kportal.extension.module.IModule;
import com.kportal.extension.module.bean.FicheBeanExport;
import com.kportal.extension.module.bean.PluginFicheBeanExport;
import com.kportal.extension.module.service.ServiceBeanDeletion;
import com.kportal.extension.module.service.ServiceBeanExport;
import com.kportal.extension.module.service.ServiceBeanImport;
import com.kportal.extension.module.service.ServiceContenuModule;
import com.kportal.extension.module.utils.ServiceContenuModuleHelper;
import com.univ.objetspartages.bean.MediaBean;
import com.univ.objetspartages.bean.RessourceBean;
import com.univ.objetspartages.om.AbstractFiche;
import com.univ.objetspartages.om.AbstractOm;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.Ressource;

public class UASServicesHelper {

	/** Logger available to subclasses. */
	private static final Logger LOGGER = LoggerFactory.getLogger(UASServicesHelper.class);

	/**
	 * export des ressources d'une fiche ou d'un modele de mail (fichiers joints au modèle)
	 *
	 * @param vRessources
	 * @return
	 */
	public static List<RessourceBean> getRessources(final Vector<Ressource> vRessources) {
		final List<RessourceBean> ressources = new ArrayList<>();
		for (final Ressource ressource : vRessources) {
			final RessourceBean ressourceBean = new RessourceBean();
			try {
				UASOmHelper.copyProperties(ressourceBean, ressource);
				ressources.add(ressourceBean);
			} catch (IllegalAccessException | InvocationTargetException e) {
				LOGGER.error("impossible de copier le bean ressource", e);
			}
		}
		return ressources;
	}

	/**
	 * export des ressources d'une fiche ou d'un modele de mail (fichiers joints au modèle)
	 *
	 * @param vRessources
	 * @param pathExport
	 * @return
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IOException
	 */
	public static Map<String, MediaBean> getMediasByRessources(final Collection<Ressource> vRessources, final String pathExport) throws IllegalAccessException,
		InvocationTargetException, IOException {
		final Map<String, MediaBean> medias = new HashMap<>();
		boolean save = false;
		final Iterator<Ressource> it = vRessources.iterator();
		while (it.hasNext()) {
			final Ressource ressource = it.next();
			save = false;
			final Media media = ressource.getMedia();
			save = processMedia(media, pathExport);
			if (save) {
				final MediaBean mediaBean = new MediaBean();
				UASOmHelper.copyProperties(mediaBean, media);
				medias.put(media.getIdMedia().toString(), mediaBean);
			} else {
				// on supprime la ressource si le média n'est pas exporté
				it.remove();
			}
		}
		return medias;
	}

	public static Map<String, MediaBean> getMedias(final OMContext ctx, final Object object, final String pathExport) {
		return getMedias(ctx, object, pathExport, Boolean.TRUE);
	}

	/**
	 * recupere les medias insérés dans une toolbox (pour les toolbox pas associées à une fiche : newsletter, rubrique, encadrés) et les medias uniques associés à un champ
	 *
	 * @return
	 */
	public static Map<String, MediaBean> getMedias(final OMContext ctx, final Object object, final String pathExport, final boolean inToolbox) {
		final HashMap<String, MediaBean> listeMedias = new HashMap<>();
		final ArrayList<Long> listeIdMedias = new ArrayList<>();
		if (inToolbox) {
			final List<Method> methodsToolBox = GetterAnnotationHelper.getMethodToolbox(object);
			if (CollectionUtils.isNotEmpty(methodsToolBox)) {
				for (final Method method : methodsToolBox) {
					try {
						final Object contenuToolbox = method.invoke(object);
						if (contenuToolbox instanceof String && !contenuToolbox.equals("")) {
							final String[] listeIdFichiers = StringUtils.substringsBetween((String) contenuToolbox, "[id-fichier]", "[/id-fichier]");
							final String[] listeIdImages = StringUtils.substringsBetween((String) contenuToolbox, "[id-image]", "[/id-image]");
							if (listeIdFichiers != null) {
								for (final String idMedia : listeIdFichiers) {
									listeIdMedias.add(new Long(idMedia));
								}
							}
							if (listeIdImages != null) {
								for (final String idMedia : listeIdImages) {
									listeIdMedias.add(new Long(idMedia));
								}
							}
						}
					} catch (final Exception e) {
						// nothing
					}
				}
			}
		}
		final List<Method> methodIdMedia = GetterAnnotationHelper.getMethodIdMedia(object);
		if (CollectionUtils.isNotEmpty(methodIdMedia)) {
			for (final Method method : methodIdMedia) {
				try {
					final Object idMedia = method.invoke(object);
					final Media media = new Media();
					media.setCtx(ctx);
					media.setIdMedia(new Long(idMedia.toString()));
					media.retrieve();
					listeIdMedias.add(media.getIdMedia());
				} catch (final Exception e) {
					// nothing
				}
			}
		}
		Media media;
		boolean save = false;
		for (final Long idMedia : listeIdMedias) {
			try {
				save = false;
				media = new Media();
				media.setCtx(ctx);
				media.setIdMedia(idMedia);
				media.retrieve();
				save = processMedia(media, pathExport);
				if (save) {
					final MediaBean mediaBean = new MediaBean();
					UASOmHelper.copyProperties(mediaBean, media);
					listeMedias.put(media.getIdMedia().toString(), mediaBean);
				}
			} catch (final Exception e) {
				LOGGER.error("Erreur lors de la recuperation d'un média de contenu de toolbox", e);
			}
		}
		return listeMedias;
	}

	/**
	 *
	 * @param media
	 * @return
	 * @throws IOException
	 */
	private static boolean processMedia(final Media media, final String pathExport) throws IOException {
		boolean save = false;
		if (media.isLocal()) {
			final File source = new File(media.getPathAbsolu());
			if (source.exists()) {
				final File destination = new File(pathExport, media.getUrl());
				FileUtils.copyFile(source, destination);
				save = true;
			}
		}
		// garde fou si on a un fichier obsolète qui possède encore une url absolue /home/... ou c:\
		else if (!media.getUrl().startsWith("/") && !media.getUrl().contains("\\")) {
			save = true;
		}
		if (media.getUrlVignette().length() > 0) {
			final File source = new File(media.getPathVignetteAbsolu());
			if (source.exists()) {
				final File destination = new File(pathExport, media.getUrlVignette());
				FileUtils.copyFile(source, destination);
			} else {
				media.setUrlVignette("");
			}
		}
		return save;
	}

	public static <T extends Serializable> T getBean(final Object object, final Class<T> clazz) throws Exception {
		if (ClassUtils.isAssignable(object.getClass(), AbstractFiche.class)) {
			final AbstractFiche<?> fiche = (AbstractFiche<?>) object;
			return clazz.cast(fiche.getPersistenceBean());
		} else if (ClassUtils.isAssignable(object.getClass(), AbstractOm.class)) {
			final AbstractOm<?, ?> om = (AbstractOm<?, ?>) object;
			return clazz.cast(om.getPersistenceBean());
		} else {
			final T clone = clazz.newInstance();
			UASOmHelper.copyProperties(clone, object);
			return clone;
		}
	}

	@SuppressWarnings("unchecked")
	public static <T extends Serializable> Class<T> getBeanClass(final Object object) {
		if (ClassUtils.isAssignable(object.getClass(), AbstractFiche.class)) {
			final AbstractFiche<?> fiche = (AbstractFiche<?>) object;
			return (Class<T>) fiche.getPersistenceBean().getClass();
		} else if (ClassUtils.isAssignable(object.getClass(), AbstractOm.class)) {
			final AbstractOm<?, ?> om = (AbstractOm<?, ?>) object;
			return (Class<T>) om.getPersistenceBean().getClass();
		} else {
			return getOldParentBeanClass(object.getClass());
		}
	}

	@SuppressWarnings("unchecked")
	public static <T extends Serializable> Class<T> getOldParentBeanClass(final Class<?> clazz) {
		final Iterator<Class<?>> hierarchy = ClassUtils.hierarchy(clazz, ClassUtils.Interfaces.EXCLUDE).iterator();
		Class<?> beanType = null;
		int counter = 0;
		while (hierarchy.hasNext() && counter < 3) {
			beanType = hierarchy.next();
			counter++;
		}
		return (Class<T>) beanType;
	}

	public static <T extends Serializable> FicheBeanExport<T> getFicheBeanExport(final FicheUniv ficheUniv, final Class<T> clazz) throws Exception {
		final FicheBeanExport<T> beanExport = new FicheBeanExport<>();
		beanExport.setBean(UASServicesHelper.getBean(ficheUniv, clazz));
		return beanExport;
	}

	public static <T extends Serializable> PluginFicheBeanExport<T> getPluginFicheBeanExport(final Object objet, final Class<T> clazz) throws Exception {
		final PluginFicheBeanExport<T> beanExport = new PluginFicheBeanExport<>();
		beanExport.setBean(UASServicesHelper.getBean(objet, clazz));
		return beanExport;
	}

	public static ServiceBeanExport<?> getServiceBeanExport(final IModule module) {
		ServiceBeanExport<?> serviceARetourner = null;
		final Collection<ServiceContenuModule> servicesModulesDisponibles = ServiceContenuModuleHelper.getServiceContenuModules();
		for (final ServiceContenuModule serviceModuleDispo : servicesModulesDisponibles) {
			if (serviceModuleDispo.getModules().contains(module)) {
				serviceARetourner = serviceModuleDispo.getServiceBeanExport();
			}
		}
		return serviceARetourner;
	}

	public static ServiceBeanImport<?> getServiceBeanImport(final IModule module) {
		ServiceBeanImport<?> serviceARetourner = null;
		final Collection<ServiceContenuModule> servicesModulesDisponibles = ServiceContenuModuleHelper.getServiceContenuModules();
		for (final ServiceContenuModule serviceModuleDispo : servicesModulesDisponibles) {
			if (serviceModuleDispo.getModules().contains(module)) {
				serviceARetourner = serviceModuleDispo.getServiceBeanImport();
			}
		}
		return serviceARetourner;
	}

	public static ServiceBeanDeletion getServiceBeanSuppression(final IModule module) {
		ServiceBeanDeletion serviceARetourner = null;
		final Collection<ServiceContenuModule> servicesModulesDisponibles = ServiceContenuModuleHelper.getServiceContenuModules();
		for (final ServiceContenuModule serviceModuleDispo : servicesModulesDisponibles) {
			if (serviceModuleDispo.getModules().contains(module)) {
				serviceARetourner = serviceModuleDispo.getServiceBeanDeletion();
			}
		}
		return serviceARetourner;
	}

	public static ServiceBeanReference getServiceBeanReference(final IModule module) {
		ServiceBeanReference serviceARetourner = null;
		final Collection<ServiceContenuModule> servicesModulesDisponibles = ServiceContenuModuleHelper.getServiceContenuModules();
		for (final ServiceContenuModule serviceModuleDispo : servicesModulesDisponibles) {
			if (serviceModuleDispo.getModules().contains(module)) {
				serviceARetourner = serviceModuleDispo.getServiceBeanReference();
			}
		}
		return serviceARetourner;
	}

	public static boolean hasAttachedService(final IModule module) {
		return getServiceBeanExport(module) != null;
	}
}
