/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kosmos.validation;

import com.kosmos.validation.message.LocaleMessageInterpolator;
import org.hibernate.validator.resourceloading.AggregateResourceBundleLocator;
import org.hibernate.validator.spi.resourceloading.ResourceBundleLocator;

import javax.validation.Configuration;
import javax.validation.Constraint;
import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import java.util.Arrays;
import java.util.Locale;
import java.util.Set;

/**
 * Classe utilitaire permettant de valider un bean via la validation api. Si le bean fourni en paramètre contient des
 * {@link Constraint} sur ses champs, ce service s'assurera que celles-ci sont bien respectées.
 */
public class ValidationUtils {

    /**
     * Permet de valider le bean fourni en paramètre si celui ci contient des {@link Constraint} sur ses champs,
     *
     * @param objectToValidate
     *         le bean contenant des contraintes à valider
     * @param currentLocale
     *         la locale courante permettant d'avoir des messages d'erreurs dans la bonne locale
     * @param idExtension
     *         l'id de l'extension pour laquelle il faut rajouter un bundle de messages
     * @return l'ensemble des violations.
     */
    public static <T> Set<ConstraintViolation<T>> validateBean(T objectToValidate, Locale currentLocale, String idExtension) {
        final ResourceBundleLocator bundleLocator = new AggregateResourceBundleLocator(Arrays.asList(idExtension, "ValidationMessages"));
        Configuration<?> config = Validation.byDefaultProvider().configure();
        Validator validator = config.messageInterpolator(new LocaleMessageInterpolator(currentLocale, bundleLocator)).buildValidatorFactory().getValidator();
        return validator.validate(objectToValidate);
    }

    /**
     * Permet de valider le bean fourni en paramètre si celui ci contient des {@link Constraint} sur ses champs,
     *
     * @param objectToValidate
     *         le bean contenant des contraintes à valider
     * @param currentLocale
     *         la locale courante permettant d'avoir des messages d'erreurs dans la bonne locale
     * @return l'ensemble des violations.
     */
    public static <T> Set<ConstraintViolation<T>> validateBean(T objectToValidate, Locale currentLocale) {
        Configuration<?> config = Validation.byDefaultProvider().configure();
        config.messageInterpolator(new LocaleMessageInterpolator(currentLocale));
        ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
        Validator validator = factory.getValidator();
        return validator.validate(objectToValidate);
    }

    /**
     * Permet de valider le bean fourni en paramètre si celui ci contient des {@link Constraint} sur ses champs,
     *
     * @param objectToValidate
     *         le bean contenant des contraintes à valider
     * @return l'ensemble des violations.
     */
    public static <T> Set<ConstraintViolation<T>> validateBean(T objectToValidate) {
        ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
        Validator validator = factory.getValidator();
        return validator.validate(objectToValidate);
    }
}
