/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.captcha;

import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.captcha.image.Gimpy;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.octo.captcha.service.CaptchaServiceException;
import com.octo.captcha.service.captchastore.FastHashMapCaptchaStore;
import com.octo.captcha.service.multitype.MultiTypeCaptchaService;

public class CaptchaServlet extends HttpServlet {

	private static final long serialVersionUID = -3833735545073866482L;

	@Override
	protected void doGet(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws ServletException, IOException {
		// get the session id that will identify the generated captcha.
		//the same id must be used to validate the response, the session id is a good candidate!
		String captchaId = httpServletRequest.getSession(Boolean.FALSE).getId();
		if (httpServletRequest.getParameter("image") != null) {
			byte[] captchaChallengeAsJpeg = null;
			// the output stream to render the captcha image as jpeg into
			ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
			try {
				// call the ImageCaptchaService getChallenge method
				MultiTypeCaptchaService captchaService = (MultiTypeCaptchaService) ApplicationContextManager.getCoreContextBean("captchaService");
				BufferedImage image = captchaService.getImageChallengeForID(captchaId, httpServletRequest.getLocale());
				ImageIO.write(image, "jpeg", jpegOutputStream);
			} catch (IllegalArgumentException e) {
				httpServletResponse.sendError(HttpServletResponse.SC_NOT_FOUND);
				return;
			} catch (CaptchaServiceException e) {
				httpServletResponse.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
				return;
			}
			captchaChallengeAsJpeg = jpegOutputStream.toByteArray();
			// flush it in the response
			httpServletResponse.setHeader("Cache-Control", "no-store");
			httpServletResponse.setHeader("Pragma", "no-cache");
			httpServletResponse.setDateHeader("Expires", 0);
			httpServletResponse.setContentType("image/jpeg");
			ServletOutputStream responseOutputStream = httpServletResponse.getOutputStream();
			responseOutputStream.write(captchaChallengeAsJpeg);
			responseOutputStream.flush();
			responseOutputStream.close();
		} else if (httpServletRequest.getParameter("audio") != null) {
			FastHashMapCaptchaStore captchaStore = (FastHashMapCaptchaStore) ApplicationContextManager.getCoreContextBean("captchaStore");
			Gimpy captcha = (Gimpy) captchaStore.getCaptcha(captchaId);
			String response = captcha.getResponse();
			File f = new File(WebAppUtil.getAbsolutePath() + PropertyHelper.getCoreProperty("captcha.audio_path") + response + ".mp3");
			if (f.exists()) {
				ServletOutputStream responseOutputStream = httpServletResponse.getOutputStream();
				FileInputStream fis = new FileInputStream(f);
				httpServletResponse.setHeader("Cache-Control", "no-store");
				httpServletResponse.setHeader("Pragma", "no-cache");
				httpServletResponse.setHeader("Content-Disposition", "inline;filename=\"mot.mp3\"");
				httpServletResponse.setDateHeader("Expires", 0);
				httpServletResponse.setContentType("audio/mpeg3");
				BufferedInputStream bis = new BufferedInputStream(fis);
				try {
					byte[] buf = new byte[1024 * 4];
					int nbBytes;
					while ((nbBytes = bis.read(buf)) > 0) {
						responseOutputStream.write(buf, 0, nbBytes);
					}
				} finally {
					responseOutputStream.flush();
					responseOutputStream.close();
					if (bis != null) {
						bis.close();
					}
					if (fis != null) {
						fis.close();
					}
				}
			}
		}
	}
}
