/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.cms.objetspartages;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ClassBeanManager;
import com.kportal.extension.module.AbstractBeanManager;
import com.kportal.extension.module.IModule;
import com.univ.objetspartages.cache.CacheStructureManager;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.IEtat;
import com.univ.objetspartages.om.StructureModele;

/**
 * The Class ReferentielManager.
 */
public class ReferentielManager extends AbstractBeanManager {

	/** Logger available to subclasses. */
	private static final Logger LOG = LoggerFactory.getLogger(ReferentielManager.class);

	/** The Constant ID_BEAN. */
	public static final String ID_BEAN = "referentielManager";

	/** The objets by code. */
	private static Map<String, Objetpartage> objetsByCode = null;

	/** The codes objets by nom. */
	private static Map<String, String> codesObjetsByNom = null;

	/** The codes objets by classe. */
	private static Map<String, String> codesObjetsByClasse = null;

	/** The codes objets structure. */
	private static Collection<String> codesObjetsStructure = null;

	/** The codes objets tries. */
	private static List<String> codesObjetsTries = null;

	public CacheStructureManager cacheStructureManager;

	public List<String> etatsObjet = null;

	public List<String> etatsObjetFront = null;

	/**
	 * Inits the.
	 */
	@Override
	public void refresh() {
		final Collection<Objetpartage> objetsPartages = moduleManager.getModules(Objetpartage.class);
		objetsByCode = new HashMap<>();
		codesObjetsByNom = new HashMap<>();
		codesObjetsByClasse = new HashMap<>();
		codesObjetsStructure = new ArrayList<>();
		for (final IModule module : objetsPartages) {
			final Objetpartage objet = (Objetpartage) module;
			String nomObjet = objet.getNomObjet().toLowerCase();
			String codeObjet = objet.getCodeObjet();
			if (objetsByCode.get(codeObjet) == null && codesObjetsByNom.get(nomObjet) == null) {
				try {
					final Class<?> classeObjet = Class.forName(objet.getNomClasse());
					if (classeObjet.newInstance() instanceof StructureModele) {
						codesObjetsStructure.add(codeObjet);
					}
				} catch (final Exception e) {
					LOG.error("La classe " + objet.getNomClasse() + " n'a pas pu être instanciée, l'objet " + nomObjet + " est ignoré");
					continue;
				}
				objet.setNomObjet(nomObjet);
				objetsByCode.put(codeObjet, objet);
				codesObjetsByNom.put(nomObjet, codeObjet);
				codesObjetsByClasse.put(objet.getNomClasse(), codeObjet);
			} else {
				LOG.warn("L'objet " + objet.getNomObjet() + " n'a pas été chargé car son code ou nom existe déjà!");
			}
		}
		codesObjetsTries = new ArrayList<>(objetsByCode.keySet());
		Collections.sort(codesObjetsTries);
		// chargement des états
		etatsObjet = new ArrayList<>();
		etatsObjetFront = new ArrayList<>();
		// Enum par défaut
		for (final EtatFiche etat : EtatFiche.values()) {
			etatsObjet.add(etat.getEtat());
			if (etat.isFront()) {
				etatsObjetFront.add(etat.getEtat());
			}
		}
		// Chargement dynamique d'états supplémentaires
		for (final IEtat etat : ClassBeanManager.getInstance().getBeanOfType(IEtat.class)) {
			etatsObjet.add(etat.getEtat());
			if (etat.isFront()) {
				etatsObjetFront.add(etat.getEtat());
			}
		}
		// chargement des structures
		try {
			cacheStructureManager.flush();
			cacheStructureManager.getListeStructures();
		} catch (final Exception e) {
			LOG.error(e.getMessage(), e);
		}
	}

	/**
	 * Gets the objet by nom.
	 *
	 * @param nom
	 *            the nom
	 * @return the objet by nom
	 */
	public Objetpartage getObjetByNom(final String nom) {
		return objetsByCode.get(codesObjetsByNom.get(nom.toLowerCase()));
	}

	/**
	 * Gets the objets by code.
	 *
	 * @return the objets by code
	 */
	public Map<String, Objetpartage> getObjetsByCode() {
		return objetsByCode;
	}

	/**
	 * Gets the codes objets by nom.
	 *
	 * @return the codes objets by nom
	 */
	public Map<String, String> getCodesObjetsByNom() {
		return codesObjetsByNom;
	}

	/**
	 * Gets the codes objets by classe.
	 *
	 * @return the codes objets by classe
	 */
	public Map<String, String> getCodesObjetsByClasse() {
		return codesObjetsByClasse;
	}

	/**
	 * Gets the codes objets structure.
	 *
	 * @return the codes objets structure
	 */
	public Collection<String> getCodesObjetsStructure() {
		return codesObjetsStructure;
	}

	/**
	 * Gets the codes objets tries.
	 *
	 * @return the codes objets tries
	 */
	public List<String> getCodesObjetsTries() {
		return codesObjetsTries;
	}

	public CacheStructureManager getCacheStructureManager() {
		return cacheStructureManager;
	}

	public void setCacheStructureManager(final CacheStructureManager cacheStructureManager) {
		this.cacheStructureManager = cacheStructureManager;
	}

	public List<String> getEtatsObjet() {
		return etatsObjet;
	}

	public List<String> getEtatsObjetFront() {
		return etatsObjetFront;
	}
}
