/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.core.config;

import java.io.IOException;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.Resource;
import org.springframework.util.PropertyPlaceholderHelper;

/**
 * The Class PropertyConfigurer. !!! CETTE CLASSE NE DOIT PAS INSTANCIER DE LOGGER CAR ELLE CHARGE LES PROPRIETES UTILES AU LOGGER !!!
 */
public class PropertyConfigurer extends PropertyPlaceholderConfigurer implements ApplicationContextAware {

	protected static final String ID_BEAN = "propertyConfigurer";

	private static ApplicationContext applicationContext;

	/** The properties. */
	private Properties properties;

	private PropertyPlaceholderHelper placeHolderHelper;

	public Properties getProperties() {
		return properties;
	}

	/** The locations. */
	private Resource[] locations;

	public Resource[] getLocations() {
		return locations;
	}

	/*
	 * Surcharge de la méthode
	 * @see org.springframework.core.io.support.PropertiesLoaderSupport#setLocations(org.springframework.core.io.Resource[])
	 */
	@Override
	public void setLocations(Resource... locs) {
		super.setLocations(locs);
		locations = locs;
	}

	/*
	 * Surcharge de la méthode pour stocker la liste finale des properties chargées dans l'ordre des fichiers #locations
	 * @see org.springframework.beans.factory.config.PropertyPlaceholderConfigurer#processProperties(org.springframework.beans.factory.config.ConfigurableListableBeanFactory, java.util.Properties)
	 */
	@Override
	protected void processProperties(ConfigurableListableBeanFactory beanFactory, Properties props) throws BeansException {
		placeHolderHelper = new PropertyPlaceholderHelper(placeholderPrefix, placeholderSuffix, valueSeparator, ignoreUnresolvablePlaceholders);
		super.processProperties(beanFactory, props);
		properties = props;
	}

	/*
	 * Cette méthode permet de recharger les properties à partir de la liste des fichiers #locations,
	 * liste qui pourrait être mise à jour avec un module de surcharge par ex
	 */
	public synchronized void reload() throws IOException {
		properties = mergeProperties();
	}

	public static PropertyConfigurer getInstance() {
		return (PropertyConfigurer) applicationContext.getBean(ID_BEAN);
	}

	public PropertyPlaceholderHelper getPlaceHolderHelper() {
		return placeHolderHelper;
	}

	public static String getProperty(String key) {
		String message = getInstance().getProperties().getProperty(key);
		if (StringUtils.isNotEmpty(message)) {
			// le helper permet de substituer la valeur d'une propriété ${key} par la valeur de 'key'
			return getInstance().getPlaceHolderHelper().replacePlaceholders(message, getInstance().getProperties());
		}
		return null;
	}

	@Override
	public void setApplicationContext(ApplicationContext appCtx) throws BeansException {
		applicationContext = appCtx;
	}
}
