/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.core.resources.optimizer.model;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ro.isdc.wro.config.Context;
import ro.isdc.wro.model.group.DefaultGroupExtractor;
import ro.isdc.wro.model.group.GroupExtractor;
import ro.isdc.wro.model.resource.ResourceType;

public class CustomGroupExtractor implements GroupExtractor {

	private static final Logger LOG = LoggerFactory.getLogger(CustomGroupExtractor.class);

	/**
	 * The name of the attribute where the servlet path is stored when requestDispatcher.include is called.
	 */
	public static final String ATTR_INCLUDE_PATH = "javax.servlet.include.servlet_path";

	/**
	 * The name of the parameter used to decide if the group must be minimized.
	 */
	public static final String PARAM_MINIMIZE = "minimize";

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getGroupName(final HttpServletRequest request) {
		if (request == null) {
			throw new IllegalArgumentException("Request cannot be NULL!");
		}
		final String includeUriPath = (String) request.getAttribute(ATTR_INCLUDE_PATH);
		String uri = request.getRequestURI();
		// check if include or uri path are present and use one of these as request uri.
		uri = includeUriPath != null ? includeUriPath : uri;
		// ex : /wro/groupe/hash.txt
		String[] groupes = uri.split("/");
		String groupe = "";
		if (groupes.length == 4) {
			groupe = groupes[2];
		}
		final String groupName = groupe;
		return StringUtils.isEmpty(groupName) ? null : groupName;
	}

	/**
	 * Extracts the resource type, by parsing the uri & finds the extension. If extension is valid ('css' or 'js'), returns corresponding ResourceType, otherwise throws exception.
	 * <p>
	 * Valid examples of uri are: <code>/context/somePath/test.js</code> or <code>/context/somePath/test.css</code> {@inheritDoc}
	 */
	@Override
	public ResourceType getResourceType(final HttpServletRequest request) {
		Validate.notNull(request);
		final String uri = request.getRequestURI();
		ResourceType type = null;
		try {
			type = ResourceType.get(getExtensionFromUri(uri));
		} catch (final IllegalArgumentException e) {
			LOG.debug("[FAIL] Cannot identify resourceType for uri: " + uri);
		}
		return type;
	}

	/**
	 * The uri is cleaned up (the ;jsessionID is removed).
	 * 
	 * @return the extension of the resource.
	 */
	private String getExtensionFromUri(final String uri) {
		return FilenameUtils.getExtension(uri.replaceFirst("(.*)(;.*)", "$1"));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String encodeGroupUrl(final String groupName, final ResourceType resourceType, final boolean minimize) {
		return String.format("%s.%s?" + PARAM_MINIMIZE + "=%s", groupName, resourceType.name().toLowerCase(), minimize);
	}

	/**
	 * The minimization is can be switched off only in debug mode.
	 * 
	 * @return false if the request contains parameter {@link DefaultGroupExtractor#PARAM_MINIMIZE} with value false, otherwise returns true.
	 */
	@Override
	public boolean isMinimized(final HttpServletRequest request) {
		if (request == null) {
			throw new IllegalArgumentException("Request cannot be NULL!");
		}
		final String minimizeAsString = request.getParameter(PARAM_MINIMIZE);
		return !(Context.get().getConfig().isDebug() && "false".equalsIgnoreCase(minimizeAsString));
	}
}
